package ai.causalfoundry.android.sdk.loyalty.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.LevelObject
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.utils.LoyaltyConstants

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

class CfLogLevelEvent {

    /**
     * CfLogLevelEvent is to log the update of user level. It required the user level before update and
     * user level after the update. You need to pass module Id as well if the level update event is
     * triggered because of the e-learning content block, any achievement in the e-learning platform.
     */
    data class Builder(
        var prev_level: Int? = null,
        var new_level: Int? = null,
        var module_id: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setPreviousLevel is required to set the previous score/level number for the user.
         */
        fun setPreviousLevel(prev_level: Int?) = apply { this.prev_level = prev_level }

        /**
         * setNewLevel is required to set the new score/level number for the user. This can be
         * from any source, e-learning or e-commerce or even social.
         */
        fun setNewLevel(new_level: Int?) = apply { this.new_level = new_level }

        /**
         * setModuleId is for specific use-case when update of level is from e-learning. In
         * such case module id is required, otherwise you can pass null for this as well.
         */
        fun setModuleId(module_id: String?) = apply { this.module_id = module_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the prev_level provided is null or no value is
                 * provided at all.
                 */
                prev_level == null -> {
                    ExceptionManager.throwIsRequiredException("previous_level")
                }

                /**
                 * Will throw and exception if the new_level provided is null or no value is
                 * provided at all.
                 */
                new_level == null -> {
                    ExceptionManager.throwIsRequiredException("new_level")
                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val levelObject = LevelObject(
                        prev_level!!, new_level!!, module_id,
                        meta
                    )
                    CFSetup().track(
                        LoyaltyConstants.contentBlockName, LoyaltyEventType.level.name,
                        levelObject, update_immediately
                    )
                }
            }
        }
    }
}
