package ai.causalfoundry.android.sdk.loyalty.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.PromoItemObject
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.PromoObject
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoAction
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoType
import ai.causalfoundry.android.sdk.loyalty.utils.LoyaltyConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

class CfLogPromoEvent {

    /**
     * CfLogPromoEvent is to log the events associated of the promo lists and promo items and when they are clicked on.
     */
    data class Builder(
        var promo_id: String? = null,
        var promo_action: String? = null,
        var promo_title: String? = null,
        var promo_type: String? = null,
        var promo_items_list: ArrayList<PromoItemObject> = arrayListOf(),
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setPromoId is required to set the Id of the promo
         */
        fun setPromoId(promo_id: String?) = apply { this.promo_id = promo_id }

        /**
         * setPromoAction is required to set the action for the promo
         */
        fun setPromoAction(promo_action: PromoAction) =
            apply { this.promo_action = promo_action.name }

        fun setPromoAction(promo_action: String) = apply {
            if (CoreConstants.enumContains<PromoAction>(promo_action)) {
                this.promo_action = promo_action
            } else {
                ExceptionManager.throwEnumException(PromoAction::class.java.simpleName)
            }
        }

        /**
         * setPromoTitle is required to set the title of the promo (if any)
         */
        fun setPromoTitle(promo_title: String?) = apply { this.promo_title = promo_title }

        /**
         * setPromoType is required to set the type of the promo
         */
        fun setPromoType(promo_type: PromoType) = apply { this.promo_type = promo_type.name }
        fun setPromoType(promo_type: String) = apply {
            if (CoreConstants.enumContains<PromoType>(promo_type)) {
                this.promo_type = promo_type
            } else {
                ExceptionManager.throwEnumException(PromoType::class.java.simpleName)
            }
        }

        /**
         * addItem can be used to add the item promo is being applied to. This log can
         * be used to add one item to the log at a time. Promo item should be in a valid format.
         * With elements of the orderObject as:
         * PromoItemObject(itemID, type)
         */
        fun addItem(itemModel: PromoItemObject) = apply {
            LoyaltyConstants.isItemTypeObjectValid(itemModel)
            this.promo_items_list.add(itemModel)
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun addItem(itemJsonString: String) = apply {
            val item: PromoItemObject = Gson().fromJson(itemJsonString, PromoItemObject::class.java)
            LoyaltyConstants.isItemTypeObjectValid(item)
            this.promo_items_list.add(item)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<PromoItemObject> to log the event successfully. Order item should be in a
         * valid format. With elements as:
         * PromoItemObject(itemID, type)
         */
        fun addItemList(itemList: ArrayList<PromoItemObject>) = apply {
            for (promoItem in itemList) {
                LoyaltyConstants.isItemTypeObjectValid(promoItem)
            }
            this.promo_items_list.addAll(itemList)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<PromoItemObject> to log the event successfully. But the input param is of type
         * string , this is special use case for react native logs where list can be passed as
         * Json string and can be used to log the event. Order item should be in a valid format.
         * With elements of the orderObject as:
         * PromoItemObject(itemID, type)
         */
        fun addItemList(itemListString: String) = apply {
            val itemModels: Array<PromoItemObject> = Gson().fromJson(
                itemListString,
                Array<PromoItemObject>::class.java
            )
            val itemList = java.util.ArrayList(itemModels.toMutableList())
            for (promoItem in itemList) {
                LoyaltyConstants.isItemTypeObjectValid(promoItem)
            }
            this.promo_items_list.addAll(itemList)
        }


        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the promo_id provided is null or no value is
                 * provided at all.
                 */
                promo_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("promo_id")
                }

                /**
                 * Will throw and exception if the promo_action provided is null or no value is
                 * provided at all.
                 */
                promo_action.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("promo_action")
                }

                /**
                 * Will throw and exception if the promo_type provided is null or no value is
                 * provided at all.
                 */
                promo_type.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("promo_type")
                }

//                /**
//                 * Will throw and exception if the subject_ids provided is null or no value is
//                 * provided at all.
//                 */
//                promo_items_list.isEmpty() -> {
//                    ExceptionManager.throwIsRequiredException("promo_items_list")
//                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val promoObject = PromoObject(
                        promo_id = promo_id!!,
                        promo_action = promo_action!!,
                        promo_title = CoreConstants.checkIfNull(promo_title),
                        promo_type = promo_type!!,
                        promo_items_list = promo_items_list.distinctBy { it.item_id },
                        meta
                    )
                    CFSetup().track(
                        LoyaltyConstants.contentBlockName, LoyaltyEventType.promo.name,
                        promoObject, update_immediately
                    )
                }
            }
        }
    }
}
