package ai.causalfoundry.android.sdk.loyalty.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.event_types.CurrencyCode
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.RewardEventObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.RedeemObject
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.RedeemType
import ai.causalfoundry.android.sdk.loyalty.event_types.RewardAction
import ai.causalfoundry.android.sdk.loyalty.event_types.internal_types.InternalCurrencyCode
import ai.causalfoundry.android.sdk.loyalty.utils.LoyaltyConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 05 April, 2023
 */

class CfLogRewardEvent {

    /**
     * CfLogRewardEvent is to log the reward related events for user, which includes viewing,
     * redeeming, adding.
     */
    data class Builder(
        var reward_id: String? = null,
        var action_value: String? = null,
        var acc_points: Float? = 0f,
        var total_points: Float? = null,
        var redeem_object: RedeemObject? = null,
        var usd_rate: Float? = null,
        private var isCashEvent: Boolean = false,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setRewardId is for the providing Id for the reward event. Can be userId if the
         * reward is not redeemed individually.
         */
        fun setRewardId(reward_id: String) = apply {
            this.reward_id = reward_id
        }

        /**
         * setAction is required to set the Action type for the Reward Action. SDK provides
         * enum classes to support available log types. 1 main is achieved.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type.
         */
        fun setAction(action: RewardAction) = apply { this.action_value = action.name }
        fun setAction(action: String) = apply {
            if (CoreConstants.enumContains<RewardAction>(action)) {
                this.action_value = action
            } else {
                ExceptionManager.throwEnumException(RewardAction::class.java.simpleName)
            }
        }

        /**
         * setAccumulatedPoints is for the providing achieved points in case of add reward event.
         */
        fun setAccumulatedPoints(acc_points: Float) = apply {
            this.acc_points = acc_points
        }

        /**
         * setTotalPoints logs the total points achieved so far by the user.
         */
        fun setTotalPoints(total_points: Float) = apply {
            this.total_points = total_points
        }


        /**
         * setRedeemObject is for the providing details about reward redeeming.
         * The object should be based on the RedeemObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setRedeemObject(redeem_object: RedeemObject) = apply {
            this.redeem_object = redeem_object
        }

        fun setRedeemObject(redeem_object: String) = apply {
            this.redeem_object = Gson().fromJson(redeem_object, RedeemObject::class.java)
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the reward_id provided is null or no value is
                 * provided at all.
                 */
                reward_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("reward_id")
                }

                /**
                 * Will throw and exception if the action_value provided is null or no value is
                 * provided at all.
                 */
                action_value.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("action_value")
                }

                /**
                 * Will throw and exception if the acc_points provided is null or no value is
                 * provided at all in-case of add action.
                 */
                action_value == RewardAction.add.name && acc_points == 0f -> {
                    ExceptionManager.throwIsRequiredException("acc_points")
                }

                /**
                 * Will throw and exception if the total_points provided is null or no value is
                 * provided at all.
                 */
                total_points == null -> {
                    ExceptionManager.throwIsRequiredException("total_points")
                }

                else -> {

                    if (action_value == RewardAction.redeem.name) {
                        acc_points = null

                        if (redeem_object!!.points_withdrawn < 0) {
                            ExceptionManager.throwIsRequiredException("points_withdrawn")
                        } else if (!CoreConstants.enumContains<RedeemType>(redeem_object!!.type)) {
                            ExceptionManager.throwEnumException(RedeemType::class.java.simpleName)
                        } else if (redeem_object!!.converted_value < 0) {
                            ExceptionManager.throwIsRequiredException("converted_value")
                        } else if (redeem_object!!.is_successful == null) {
                            ExceptionManager.throwIsRequiredException("redeem is_successful")
                        } else if (redeem_object!!.type == RedeemType.cash.name) {
                            if (redeem_object!!.currency.isNullOrEmpty()) {
                                ExceptionManager.throwIsRequiredException("redeem currency")
                            } else if (!CoreConstants.enumContains<CurrencyCode>(redeem_object!!.currency!!)) {
                                ExceptionManager.throwEnumException(CurrencyCode::class.java.simpleName)
                            } else {
                                isCashEvent = true
                            }
                        } else {
                            redeem_object!!.currency = null
                        }
                    } else {
                        redeem_object = null
                        isCashEvent = false
                    }

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val rewardEventObject = RewardEventObject(
                        reward_id!!, action_value!!, acc_points, total_points!!,
                        redeem_object, usd_rate, meta
                    )

                    if (isCashEvent) {
                        if (rewardEventObject.redeem!!.currency == InternalCurrencyCode.USD.name) {
                            rewardEventObject.usd_rate = 1f
                            callEvenTrack(rewardEventObject)
                        } else {
                            CFSetup().getUSDRate(
                                rewardEventObject.redeem!!.currency!!,
                                ::getUSDRateAndLogEvent
                            )
                        }
                    } else {
                        callEvenTrack(rewardEventObject)
                    }


                }
            }
        }

        private fun getUSDRateAndLogEvent(usdRate: Float) {

            callEvenTrack(
                RewardEventObject(
                    reward_id!!, action_value!!, acc_points, total_points!!,
                    redeem_object, usdRate, meta
                )
            )
        }

        private fun callEvenTrack(rewardEventObject: RewardEventObject) {
            CFSetup().track(
                LoyaltyConstants.contentBlockName, LoyaltyEventType.reward.name,
                rewardEventObject, update_immediately
            )
        }

    }
}
