package ai.causalfoundry.android.sdk.loyalty.utils

import ai.causalfoundry.android.sdk.core.catalog.catalog_models.CatalogSubject
import ai.causalfoundry.android.sdk.core.utils.Converters
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.RewardCatalogModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.SurveyCatalogModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.internal_catalog_models.InternalRewardModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.internal_catalog_models.InternalSurveyModel
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.PromoItemObject
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoItemType

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

object LoyaltyConstants {

    // SDK API endpoints
    const val contentBlockName: String = "loyalty"

    fun isItemTypeObjectValid(itemValue: PromoItemObject, eventType: LoyaltyEventType) {

        val eventName = eventType.name

        if (itemValue.item_id.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_id")
        } else if (itemValue.item_type.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_type")
        } else if (!CoreConstants.enumContains<PromoItemType>(itemValue.item_type)) {
            ExceptionManager.throwEnumException(eventName, "ItemType")
        }
    }


    fun verifyCatalogForSurvey(
        surveyId: String,
        surveyCatalogModel: SurveyCatalogModel
    ): InternalSurveyModel {

        val catalogName = CatalogSubject.survey.name + " catalog"

        if (surveyId.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey Id")
        }

        if (surveyCatalogModel.name.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey name")
        } else if (surveyCatalogModel.type.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey type")
        } else if (surveyCatalogModel.questions_list.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey questions")
        }

        return InternalSurveyModel(
            id = surveyId,
            name = CoreConstants.checkIfNull(surveyCatalogModel.name),
            duration = surveyCatalogModel.duration,
            type = CoreConstants.checkIfNull(surveyCatalogModel.type),
            reward_id = CoreConstants.checkIfNull(surveyCatalogModel.reward_id),
            questions_list = surveyCatalogModel.questions_list,
            description = CoreConstants.checkIfNull(surveyCatalogModel.description),
            organization_id = CoreConstants.checkIfNull(surveyCatalogModel.organization_id),
            organization_name = CoreConstants.checkIfNull(surveyCatalogModel.organization_name),
            creation_date = getTimeConvertedToString(surveyCatalogModel.creation_date),
            expiry_date = getTimeConvertedToString(surveyCatalogModel.expiry_date),
        )

    }

    fun verifyCatalogForReward(
        rewardId: String,
        rewardCatalogModel: RewardCatalogModel
    ): InternalRewardModel {

        val catalogName = CatalogSubject.reward.name + " catalog"

        if (rewardId.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward Id")
        }

        if (rewardCatalogModel.name.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward name")
        } else if (rewardCatalogModel.type.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward type")
        } else if (rewardCatalogModel.required_points < 1) {
            ExceptionManager.throwIsRequiredException(catalogName, "required_points")
        }

        return InternalRewardModel(
            id = rewardId,
            name = CoreConstants.checkIfNull(rewardCatalogModel.name),
            description = CoreConstants.checkIfNull(rewardCatalogModel.description),
            type = CoreConstants.checkIfNull(rewardCatalogModel.type),
            required_points = rewardCatalogModel.required_points,
            creation_date = getTimeConvertedToString(rewardCatalogModel.creation_date),
            expiry_date = getTimeConvertedToString(rewardCatalogModel.expiry_date),
            organization_id = CoreConstants.checkIfNull(rewardCatalogModel.organization_id),
            organization_name = CoreConstants.checkIfNull(rewardCatalogModel.organization_name)

        )

    }


    private fun getTimeConvertedToString(eventTime: Long): String {
        return if (eventTime != 0L) {
            Converters.convertMillisToTimeString(eventTime)
        } else {
            ""
        }

    }

}
