package ai.causalfoundry.android.sdk.loyalty.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.SurveyEventObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.SurveyObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.SurveyResponseItem
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.SurveyAction
import ai.causalfoundry.android.sdk.loyalty.event_types.SurveyType
import ai.causalfoundry.android.sdk.loyalty.utils.LoyaltyConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 04 April, 2023
 */

class CfLogSurveyEvent {

    /**
     * CfLogSurveyEvent is to log the user viewing, attempting the survey.
     */
    data class Builder(
        internal var action_value: String? = null,
        internal var survey_object: SurveyObject? = null,
        internal var response_list: ArrayList<SurveyResponseItem> = arrayListOf(),
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setAction is required to set the Action type for the Survey Action. SDK provides
         * enum classes to support available log types. 1 main is achieved.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type.
         */
        fun setAction(action: SurveyAction) = apply { this.action_value = action.name }
        fun setAction(action: String) = apply {
            if (CoreConstants.enumContains<SurveyAction>(action)) {
                this.action_value = action
            } else {
                ExceptionManager.throwEnumException(
                    LoyaltyEventType.survey.name,
                    SurveyAction::class.java.simpleName
                )
            }
        }


        /**
         * setSurveyObject is for the providing item info details about survey.
         * The object should be based on the SurveyObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setSurveyObject(survey_object: SurveyObject) = apply {
            this.survey_object = survey_object
        }

        fun setSurveyObject(survey_object: String) = apply {
            this.survey_object = Gson().fromJson(survey_object, SurveyObject::class.java)
        }

        /**
         * setResponseList is for the providing responses item details about survey.
         * The object should be based on the SurveyResponseItem List or a string list that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setResponseList(response_list: ArrayList<SurveyResponseItem>) = apply {
            this.response_list.clear()
            this.response_list.addAll(response_list)
        }

        fun setResponseList(response_list: String) = apply {
            this.response_list.clear()
            if (response_list.isNotEmpty()) {
                val itemsList = java.util.ArrayList(
                    Gson().fromJson(
                        response_list,
                        Array<SurveyResponseItem>::class.java
                    ).toMutableList()
                )
                this.response_list.addAll(itemsList)
            }
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the action_value provided is null or no value is
                 * provided at all.
                 */
                action_value.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.survey.name,
                        "action_value"
                    )
                }

                /**
                 * Will throw and exception if the survey_object provided is null or no value is
                 * provided at all.
                 */
                survey_object == null -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.survey.name,
                        "survey_object"
                    )
                }

                /**
                 * Will throw and exception if the response_list provided is null or no value is
                 * provided at all.
                 */
                action_value == "submit" && response_list.isEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.survey.name,
                        "response_list"
                    )
                }

                survey_object!!.id.isEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.survey.name,
                        "survey_id"
                    )
                }

                !CoreConstants.enumContains<SurveyType>(survey_object!!.type) -> {
                    ExceptionManager.throwEnumException(
                        LoyaltyEventType.survey.name,
                        SurveyType::class.java.simpleName
                    )
                }

                survey_object!!.is_completed == null -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.survey.name,
                        "survey is_completed"
                    )
                }

                else -> {
                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    if(LoyaltyConstants.verifySurveyResponseList(response_list)){
                        val surveyEventObject = SurveyEventObject(
                            action_value!!, survey_object!!, response_list, meta
                        )
                        CFSetup().track(
                            LoyaltyConstants.contentBlockName, LoyaltyEventType.survey.name,
                            surveyEventObject, update_immediately
                        )
                    }
                }
            }
        }
    }
}
