package ai.causalfoundry.android.sdk.loyalty.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.MilestoneObject
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.MilestoneAction
import ai.causalfoundry.android.sdk.loyalty.utils.LoyaltyConstants

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

class CfLogMilestoneEvent {

    /**
     * CfLogMilestoneEvent is to log actions regarding milestones which can be when the user
     * achieved a milestone.
     */
    data class Builder(
        var milestoneIdValue: String? = null,
        var actionValue: String? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setMilestoneId is required for logging the milestone user achieved. The is should be in
         * a string format and must in accordance to the catalog provided.
         */
        fun setMilestoneId(milestoneId: String?) = apply { this.milestoneIdValue = milestoneId }

        /**
         * setAction is required to set the Action type for the Milestone event. SDK provides
         * enum classes to support available log types. 1 main is achieved.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using enum.
         */
        fun setAction(action: MilestoneAction) = apply { this.actionValue = action.name }

        /**
         * setAction is required to set the Action type for the Milestone event. SDK provides
         * enum classes to support available log types. 1 main is achieved.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using string. Remember to note that
         * values provided using string should be the same as provided in enum or else the
         * events will be discarded.
         */
        fun setAction(action: String?) = apply {
            if (action != null) {
                if (CoreConstants.enumContains<MilestoneAction>(action)) {
                    this.actionValue = action
                } else {
                    ExceptionManager.throwEnumException(
                        LoyaltyEventType.milestone.name,
                        MilestoneAction::class.java.simpleName
                    )
                }
            } else {
                this.actionValue = action
            }
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the milestone_id provided is null or no value is
                 * provided at all.
                 */
                milestoneIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.milestone.name,
                        "milestone_id"
                    )
                }

                /**
                 * Will throw and exception if the action provided is null or no value is
                 * provided at all.
                 */
                actionValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.milestone.name,
                        MilestoneAction::class.java.simpleName
                    )
                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val milestoneObject = MilestoneObject(milestoneIdValue!!, actionValue!!, meta)
                    CFSetup().track(
                        LoyaltyConstants.contentBlockName,
                        LoyaltyEventType.milestone.name,
                        milestoneObject,
                        isUpdateImmediately
                    )
                }
            }
        }
    }
}
