package ai.causalfoundry.android.sdk.loyalty.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.PromoObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.PromoItemObject
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoAction
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoType
import ai.causalfoundry.android.sdk.loyalty.utils.LoyaltyConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

class CfLogPromoEvent {

    /**
     * CfLogPromoEvent is to log the events associated of the promo lists and promo items and when they are clicked on.
     */
    data class Builder(
        var promoIdValue: String? = null,
        var promoActionValue: String? = null,
        var promoTitleValue: String? = null,
        var promoTypeValue: String? = null,
        var promoItemsListValue: ArrayList<PromoItemObject> = arrayListOf(),
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setPromoId is required to set the Id of the promo
         */
        fun setPromoId(promoId: String?) = apply { this.promoIdValue = promoId }

        /**
         * setPromoAction is required to set the action for the promo
         */
        fun setPromoAction(promoAction: PromoAction) =
            apply { this.promoActionValue = promoAction.name }

        fun setPromoAction(promoAction: String) = apply {
            if (CoreConstants.enumContains<PromoAction>(promoAction)) {
                this.promoActionValue = promoAction
            } else {
                ExceptionManager.throwEnumException(
                    LoyaltyEventType.promo.name,
                    PromoAction::class.java.simpleName
                )
            }
        }

        /**
         * setPromoTitle is required to set the title of the promo (if any)
         */
        fun setPromoTitle(promoTitle: String?) = apply { this.promoTitleValue = promoTitle }

        /**
         * setPromoType is required to set the type of the promo
         */
        fun setPromoType(promoType: PromoType) = apply { this.promoTypeValue = promoType.name }
        fun setPromoType(promoType: String) = apply {
            if (CoreConstants.enumContains<PromoType>(promoType)) {
                this.promoTypeValue = promoType
            } else {
                ExceptionManager.throwEnumException(
                    LoyaltyEventType.promo.name,
                    PromoType::class.java.simpleName
                )
            }
        }

        /**
         * addItem can be used to add the item promo is being applied to. This log can
         * be used to add one item to the log at a time. Promo item should be in a valid format.
         * With elements of the orderObject as:
         * PromoItemObject(itemID, type)
         */
        fun addItem(itemModel: PromoItemObject) = apply {
            LoyaltyConstants.isItemTypeObjectValid(itemModel, LoyaltyEventType.promo)
            this.promoItemsListValue.add(itemModel)
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun addItem(itemModel: String) = apply {
            val item: PromoItemObject = Gson().fromJson(itemModel, PromoItemObject::class.java)
            LoyaltyConstants.isItemTypeObjectValid(item, LoyaltyEventType.promo)
            this.promoItemsListValue.add(item)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<PromoItemObject> to log the event successfully. Order item should be in a
         * valid format. With elements as:
         * PromoItemObject(itemID, type)
         */
        fun addItemList(itemList: ArrayList<PromoItemObject>) = apply {
            for (promoItem in itemList) {
                LoyaltyConstants.isItemTypeObjectValid(promoItem, LoyaltyEventType.promo)
            }
            this.promoItemsListValue.addAll(itemList)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<PromoItemObject> to log the event successfully. But the input param is of type
         * string , this is special use case for react native logs where list can be passed as
         * Json string and can be used to log the event. Order item should be in a valid format.
         * With elements of the orderObject as:
         * PromoItemObject(itemID, type)
         */
        fun addItemList(itemList: String) = apply {
            val itemModels: Array<PromoItemObject> = Gson().fromJson(
                itemList,
                Array<PromoItemObject>::class.java
            )
            val itemModelList = java.util.ArrayList(itemModels.toMutableList())
            for (promoItem in itemModelList) {
                LoyaltyConstants.isItemTypeObjectValid(promoItem, LoyaltyEventType.promo)
            }
            this.promoItemsListValue.addAll(itemModelList)
        }


        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the promo_id provided is null or no value is
                 * provided at all.
                 */
                promoIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.promo.name,
                        "promo_id"
                    )
                }

                /**
                 * Will throw and exception if the promo_action provided is null or no value is
                 * provided at all.
                 */
                promoActionValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.promo.name,
                        "promo_action"
                    )
                }

                /**
                 * Will throw and exception if the promo_type provided is null or no value is
                 * provided at all.
                 */
                promoTypeValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.promo.name,
                        "promo_type"
                    )
                }

//                /**
//                 * Will throw and exception if the subject_ids provided is null or no value is
//                 * provided at all.
//                 */
//                promo_items_list.isEmpty() -> {
//                    ExceptionManager.throwIsRequiredException("promo_items_list")
//                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val promoObject = PromoObject(
                        promoId = promoIdValue!!,
                        promoAction = promoActionValue!!,
                        promoTitle = CoreConstants.checkIfNull(promoTitleValue),
                        promoType = promoTypeValue!!,
                        promoItemsList = promoItemsListValue.distinctBy { it.itemId },
                        meta
                    )
                    CFSetup().track(
                        LoyaltyConstants.contentBlockName, LoyaltyEventType.promo.name,
                        promoObject, isUpdateImmediately
                    )
                }
            }
        }
    }
}
