package ai.causalfoundry.android.sdk.loyalty.utils

import ai.causalfoundry.android.sdk.core.catalog.catalog_models.CatalogSubject
import ai.causalfoundry.android.sdk.core.utils.Converters
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.RewardCatalogModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.SurveyCatalogModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.internal_catalog_models.InternalRewardModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.internal_catalog_models.InternalSurveyModel
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.PromoItemObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.SurveyResponseItem
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoItemType
import ai.causalfoundry.android.sdk.loyalty.event_types.SurveyType

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

object LoyaltyConstants {

    // SDK API endpoints
    const val contentBlockName: String = "loyalty"

    fun isItemTypeObjectValid(itemValue: PromoItemObject, eventType: LoyaltyEventType) {

        val eventName = eventType.name

        if (itemValue.itemId.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_id")
        } else if (itemValue.itemType.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_type")
        } else if (!CoreConstants.enumContains<PromoItemType>(itemValue.itemType)) {
            ExceptionManager.throwEnumException(eventName, "ItemType")
        }
    }


    fun verifyCatalogForSurvey(
        surveyId: String,
        surveyCatalogModel: SurveyCatalogModel
    ): InternalSurveyModel {

        val catalogName = CatalogSubject.survey.name + " catalog"

        if (surveyId.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey Id")
        }

        if (surveyCatalogModel.name.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey name")
        } else if (surveyCatalogModel.type.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey type")
        } else if (surveyCatalogModel.questionsList.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey questions")
        }

        return InternalSurveyModel(
            id = surveyId,
            name = CoreConstants.checkIfNull(surveyCatalogModel.name),
            duration = surveyCatalogModel.duration,
            type = CoreConstants.checkIfNull(surveyCatalogModel.type),
            rewardId = CoreConstants.checkIfNull(surveyCatalogModel.rewardId),
            questionsList = surveyCatalogModel.questionsList,
            description = CoreConstants.checkIfNull(surveyCatalogModel.description),
            organizationId = CoreConstants.checkIfNull(surveyCatalogModel.organizationId),
            organizationName = CoreConstants.checkIfNull(surveyCatalogModel.organizationName),
            creationDate = Converters.convertMillisToTimeString(surveyCatalogModel.creationDate),
            expiryDate = Converters.convertMillisToTimeString(surveyCatalogModel.expiryDate),
        )

    }

    fun verifyCatalogForReward(
        rewardId: String,
        rewardCatalogModel: RewardCatalogModel
    ): InternalRewardModel {

        val catalogName = CatalogSubject.reward.name + " catalog"

        if (rewardId.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward Id")
        }

        if (rewardCatalogModel.name.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward name")
        } else if (rewardCatalogModel.type.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward type")
        } else if (rewardCatalogModel.requiredPoints < 1) {
            ExceptionManager.throwIsRequiredException(catalogName, "required_points")
        }

        return InternalRewardModel(
            id = rewardId,
            name = CoreConstants.checkIfNull(rewardCatalogModel.name),
            description = CoreConstants.checkIfNull(rewardCatalogModel.description),
            type = CoreConstants.checkIfNull(rewardCatalogModel.type),
            requiredPoints = rewardCatalogModel.requiredPoints,
            creationDate = Converters.convertMillisToTimeString(rewardCatalogModel.creationDate),
            expiryDate = Converters.convertMillisToTimeString(rewardCatalogModel.expiryDate),
            organizationId = CoreConstants.checkIfNull(rewardCatalogModel.organizationId),
            organizationName = CoreConstants.checkIfNull(rewardCatalogModel.organizationName)

        )

    }

    fun verifySurveyResponseList(responseList : ArrayList<SurveyResponseItem>) : Boolean{
        for (item in responseList) {
            if (!CoreConstants.enumContains<SurveyType>(item.type)) {
                return ExceptionManager.throwEnumException(
                    LoyaltyEventType.survey.name,
                    SurveyType::class.java.simpleName
                )
            } else if (item.id.isEmpty()) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.survey.name,
                    "response_question_id"
                )
            } else if (item.question.isEmpty()) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.survey.name,
                    "response_question"
                )
            }
        }
        return true
    }

}
