package ai.causalfoundry.android.sdk.loyalty.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.event_types.CurrencyCode
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.RewardEventObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.RedeemObject
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.RedeemType
import ai.causalfoundry.android.sdk.loyalty.event_types.RewardAction
import ai.causalfoundry.android.sdk.loyalty.utils.LoyaltyConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 05 April, 2023
 */

class CfLogRewardEvent {

    /**
     * CfLogRewardEvent is to log the reward related events for user, which includes viewing,
     * redeeming, adding.
     */
    data class Builder(
        internal var rewardIdValue: String? = null,
        internal var actionValue: String? = null,
        internal var accPointsValue: Float? = 0f,
        internal var totalPointsValue: Float? = null,
        internal var redeemObjectValue: RedeemObject? = null,
        private var isCashEventValue: Boolean = false,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setRewardId is for the providing Id for the reward event. Can be userId if the
         * reward is not redeemed individually.
         */
        fun setRewardId(rewardId: String) = apply {
            this.rewardIdValue = rewardId
        }

        /**
         * setAction is required to set the Action type for the Reward Action. SDK provides
         * enum classes to support available log types. 1 main is achieved.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type.
         */
        fun setAction(action: RewardAction) = apply { this.actionValue = action.name }
        fun setAction(action: String) = apply {
            if (CoreConstants.enumContains<RewardAction>(action)) {
                this.actionValue = action
            } else {
                ExceptionManager.throwEnumException(
                    LoyaltyEventType.reward.name,
                    RewardAction::class.java.simpleName
                )
            }
        }

        /**
         * setAccumulatedPoints is for the providing achieved points in case of add reward event.
         */
        fun setAccumulatedPoints(accPoints: Float) = apply {
            this.accPointsValue = accPoints
        }

        /**
         * setTotalPoints logs the total points achieved so far by the user.
         */
        fun setTotalPoints(totalPoints: Float) = apply {
            this.totalPointsValue = totalPoints
        }


        /**
         * setRedeemObject is for the providing details about reward redeeming.
         * The object should be based on the RedeemObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setRedeemObject(redeemObject: RedeemObject) = apply {
            this.redeemObjectValue = redeemObject
        }

        fun setRedeemObject(redeemObject: String) = apply {
            this.redeemObjectValue = Gson().fromJson(redeemObject, RedeemObject::class.java)
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the reward_id provided is null or no value is
                 * provided at all.
                 */
                rewardIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.reward.name,
                        "reward_id"
                    )
                }

                /**
                 * Will throw and exception if the action_value provided is null or no value is
                 * provided at all.
                 */
                actionValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.reward.name,
                        "action_value"
                    )
                }

                /**
                 * Will throw and exception if the acc_points provided is null or no value is
                 * provided at all in-case of add action.
                 */
                actionValue == RewardAction.add.name && accPointsValue == 0f -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.reward.name,
                        "acc_points"
                    )
                }

                /**
                 * Will throw and exception if the total_points provided is null or no value is
                 * provided at all.
                 */
                totalPointsValue == null -> {
                    ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.reward.name,
                        "total_points"
                    )
                }

                else -> {

                    if (actionValue == RewardAction.redeem.name && redeemObjectValue == null) {
                        ExceptionManager.throwIsRequiredException(
                            LoyaltyEventType.reward.name,
                            "redeem_object"
                        )
                    } else if (actionValue == RewardAction.redeem.name && redeemObjectValue != null) {
                        accPointsValue = null

                        if (redeemObjectValue!!.pointsWithdrawn < 0) {
                            ExceptionManager.throwIsRequiredException(
                                LoyaltyEventType.reward.name,
                                "points_withdrawn"
                            )
                        } else if (!CoreConstants.enumContains<RedeemType>(redeemObjectValue!!.type)) {
                            ExceptionManager.throwEnumException(
                                LoyaltyEventType.reward.name,
                                RedeemType::class.java.simpleName
                            )
                        } else if (redeemObjectValue!!.convertedValue < 0) {
                            ExceptionManager.throwIsRequiredException(
                                LoyaltyEventType.reward.name,
                                "converted_value"
                            )
                        } else if (redeemObjectValue!!.isSuccessful == null) {
                            ExceptionManager.throwIsRequiredException(
                                LoyaltyEventType.reward.name,
                                "redeem is_successful"
                            )
                        } else if (redeemObjectValue!!.type == RedeemType.cash.name) {
                            if (redeemObjectValue!!.currency.isNullOrEmpty()) {
                                ExceptionManager.throwIsRequiredException(
                                    LoyaltyEventType.reward.name,
                                    "redeem currency"
                                )
                            } else if (!CoreConstants.enumContains<CurrencyCode>(redeemObjectValue!!.currency!!)) {
                                ExceptionManager.throwEnumException(
                                    LoyaltyEventType.reward.name,
                                    CurrencyCode::class.java.simpleName
                                )
                            } else {
                                isCashEventValue = true
                            }
                        } else {
                            redeemObjectValue!!.currency = null
                        }
                    } else {
                        redeemObjectValue = null
                        isCashEventValue = false
                    }

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val rewardEventObject = RewardEventObject(
                        rewardIdValue!!, actionValue!!, accPointsValue, totalPointsValue!!,
                        redeemObjectValue, meta
                    )

                    callEvenTrack(rewardEventObject)
                }
            }
        }

        private fun callEvenTrack(rewardEventObject: RewardEventObject) {
            CFSetup().track(
                LoyaltyConstants.contentBlockName, LoyaltyEventType.reward.name,
                rewardEventObject, isUpdateImmediately
            )
        }

    }
}
