package ai.causalfoundry.android.sdk.loyalty.utils

import ai.causalfoundry.android.sdk.core.catalog.catalog_models.CatalogSubject
import ai.causalfoundry.android.sdk.core.event_types.CurrencyCode
import ai.causalfoundry.android.sdk.core.utils.Converters
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.RewardCatalogModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.SurveyCatalogModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.internal_catalog_models.InternalRewardModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.internal_catalog_models.InternalSurveyModel
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.PromoItemObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.RedeemObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.SurveyResponseItem
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoItemType
import ai.causalfoundry.android.sdk.loyalty.event_types.RedeemType
import ai.causalfoundry.android.sdk.loyalty.event_types.RewardAction
import ai.causalfoundry.android.sdk.loyalty.event_types.SurveyType

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

object LoyaltyConstants {

    // SDK API endpoints
    const val contentBlockName: String = "loyalty"

    fun isItemTypeObjectValid(itemValue: PromoItemObject, eventType: LoyaltyEventType) : Boolean {

        val eventName = eventType.toString()

        if (itemValue.itemId.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(eventName, "item_id")
        } else if (itemValue.itemType.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(eventName, "item_type")
        } else if (!CoreConstants.enumContainsString<PromoItemType>(itemValue.itemType)) {
            return ExceptionManager.throwEnumException(eventName, "ItemType")
        }
        return true
    }


    fun verifyCatalogForSurvey(
        surveyId: String,
        surveyCatalogModel: SurveyCatalogModel
    ): InternalSurveyModel {

        val catalogName = CatalogSubject.Survey.toString() + " catalog"

        if (surveyId.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey Id")
        }

        if (surveyCatalogModel.name.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey name")
        } else if (surveyCatalogModel.type.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey type")
        } else if (surveyCatalogModel.questionsList.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey questions")
        }

        return InternalSurveyModel(
            id = surveyId,
            name = CoreConstants.checkIfNull(surveyCatalogModel.name),
            duration = surveyCatalogModel.duration,
            type = CoreConstants.checkIfNull(surveyCatalogModel.type),
            rewardId = CoreConstants.checkIfNull(surveyCatalogModel.rewardId),
            questionsList = surveyCatalogModel.questionsList,
            description = CoreConstants.checkIfNull(surveyCatalogModel.description),
            organizationId = CoreConstants.checkIfNull(surveyCatalogModel.organizationId),
            organizationName = CoreConstants.checkIfNull(surveyCatalogModel.organizationName),
            creationDate = Converters.convertMillisToTimeString(surveyCatalogModel.creationDate),
            expiryDate = Converters.convertMillisToTimeString(surveyCatalogModel.expiryDate),
        )

    }

    fun verifyCatalogForReward(
        rewardId: String,
        rewardCatalogModel: RewardCatalogModel
    ): InternalRewardModel {

        val catalogName = CatalogSubject.Reward.toString() + " catalog"

        if (rewardId.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward Id")
        }

        if (rewardCatalogModel.name.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward name")
        } else if (rewardCatalogModel.type.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward type")
        } else if (rewardCatalogModel.requiredPoints < 1) {
            ExceptionManager.throwIsRequiredException(catalogName, "required_points")
        }

        return InternalRewardModel(
            id = rewardId,
            name = CoreConstants.checkIfNull(rewardCatalogModel.name),
            description = CoreConstants.checkIfNull(rewardCatalogModel.description),
            type = CoreConstants.checkIfNull(rewardCatalogModel.type),
            requiredPoints = rewardCatalogModel.requiredPoints,
            creationDate = Converters.convertMillisToTimeString(rewardCatalogModel.creationDate),
            expiryDate = Converters.convertMillisToTimeString(rewardCatalogModel.expiryDate),
            organizationId = CoreConstants.checkIfNull(rewardCatalogModel.organizationId),
            organizationName = CoreConstants.checkIfNull(rewardCatalogModel.organizationName)

        )

    }

    fun verifySurveyResponseList(responseList : ArrayList<SurveyResponseItem>) : Boolean{
        for (item in responseList) {
            if (!CoreConstants.enumContainsString<SurveyType>(item.type)) {
                return ExceptionManager.throwEnumException(
                    LoyaltyEventType.Survey.toString(),
                    SurveyType::class.java.simpleName
                )
            } else if (item.id.isEmpty()) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Survey.toString(),
                    "response_question_id"
                )
            } else if (item.question.isEmpty()) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Survey.toString(),
                    "response_question"
                )
            }
        }
        return true
    }


    fun verifyRewardEventForRedeem(actionValue : String, redeemObjectValue : RedeemObject?) : Boolean {
        if (actionValue == RewardAction.Redeem.toString() && redeemObjectValue != null) {

            if (redeemObjectValue.pointsWithdrawn < 0) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Reward.toString(),
                    "points_withdrawn"
                )
            } else if (!CoreConstants.enumContainsString<RedeemType>(redeemObjectValue.type)) {
                return ExceptionManager.throwEnumException(
                    LoyaltyEventType.Reward.toString(),
                    RedeemType::class.java.simpleName
                )
            } else if (redeemObjectValue.convertedValue < 0) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Reward.toString(),
                    "converted_value"
                )
            } else if (redeemObjectValue.isSuccessful == null) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Reward.toString(),
                    "redeem is_successful"
                )
            } else if (redeemObjectValue.type == RedeemType.Cash.toString()) {
                if (redeemObjectValue.currency.isNullOrEmpty()) {
                    return ExceptionManager.throwIsRequiredException(
                        LoyaltyEventType.Reward.toString(),
                        "redeem currency"
                    )
                } else if (!CoreConstants.enumContains<CurrencyCode>(redeemObjectValue.currency!!)) {
                    return ExceptionManager.throwEnumException(
                        LoyaltyEventType.Reward.toString(),
                        CurrencyCode::class.java.simpleName
                    )
                }
            }
        }
        return true
    }


}
