package ai.causalfoundry.android.sdk.loyalty.utils

import ai.causalfoundry.android.sdk.core.catalog.catalog_models.CatalogSubject
import ai.causalfoundry.android.sdk.core.event_types.CurrencyCode
import ai.causalfoundry.android.sdk.core.utils.Converters
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.RewardCatalogModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.SurveyCatalogModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.internal_catalog_models.InternalRewardModel
import ai.causalfoundry.android.sdk.loyalty.catalog.catalog_models.internal_catalog_models.InternalSurveyModel
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.LevelObject
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.MilestoneObject
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.PromoObject
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.RewardEventObject
import ai.causalfoundry.android.sdk.loyalty.event_models.event_objects.SurveyEventObject
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.PromoItem
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.RedeemItem
import ai.causalfoundry.android.sdk.loyalty.event_models.item_objects.SurveyResponseItem
import ai.causalfoundry.android.sdk.loyalty.event_types.LoyaltyEventType
import ai.causalfoundry.android.sdk.loyalty.event_types.MilestoneAction
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoAction
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoItemType
import ai.causalfoundry.android.sdk.loyalty.event_types.PromoType
import ai.causalfoundry.android.sdk.loyalty.event_types.RedeemType
import ai.causalfoundry.android.sdk.loyalty.event_types.RewardAction
import ai.causalfoundry.android.sdk.loyalty.event_types.SurveyAction
import ai.causalfoundry.android.sdk.loyalty.event_types.SurveyType

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

internal object LoyaltyConstants {

    fun isValidLevelEvent(logObject: LevelObject): Boolean {
        if (logObject.prevLevel < 0) {
            return ExceptionManager.throwInvalidException(
                LoyaltyEventType.Level.toString(),
                "previous_level"
            )
        } else if (logObject.newLevel < 0) {
            return ExceptionManager.throwInvalidException(
                LoyaltyEventType.Level.toString(),
                "new_level"
            )
        }
        return true
    }

    fun isValidMilestoneEvent(logObject: MilestoneObject): Boolean {
        if (logObject.id.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Milestone.toString(),
                "milestone_id"
            )
        } else if (!CoreConstants.enumContainsString<MilestoneAction>(logObject.action.toString())) {
            return ExceptionManager.throwEnumException(
                LoyaltyEventType.Milestone.toString(),
                MilestoneAction::class.java.simpleName
            )
        }
        return true
    }

    fun isValidPromoObject(logObject: PromoObject): Boolean {
        if (logObject.promoId.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Promo.toString(),
                "promo_id"
            )
        } else if (!CoreConstants.enumContainsString<PromoAction>(logObject.promoAction.toString())) {
            return ExceptionManager.throwEnumException(
                LoyaltyEventType.Promo.toString(),
                PromoAction::class.java.simpleName
            )
        } else if (!CoreConstants.enumContainsString<PromoType>(logObject.promoType.toString())) {
            return ExceptionManager.throwEnumException(
                LoyaltyEventType.Promo.toString(),
                PromoType::class.java.simpleName
            )
        } else if (logObject.promoItemsList.isNotEmpty()) {
            logObject.promoItemsList.forEach {
                if (!isPromoItemObjectValid(it)) {
                    return false
                }
            }
        }
        return true
    }

    private fun isPromoItemObjectValid(logObject: PromoItem): Boolean {
        if (logObject.itemId.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Promo.toString(),
                "item_id"
            )
        } else if (!CoreConstants.enumContainsString<PromoItemType>(logObject.itemType.toString())) {
            return ExceptionManager.throwEnumException(
                LoyaltyEventType.Promo.toString(),
                "ItemType"
            )
        }
        return true
    }

    fun isValidRewardEvent(logObject: RewardEventObject): Boolean {
        if (logObject.rewardId.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Reward.toString(),
                "reward_id"
            )
        } else if (!CoreConstants.enumContainsString<RewardAction>(logObject.action.toString())) {
            return ExceptionManager.throwEnumException(
                LoyaltyEventType.Reward.toString(),
                RewardAction::class.java.simpleName
            )
        } else if (logObject.action == RewardAction.Add && logObject.accPoints == 0f) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Reward.toString(),
                "acc_points"
            )
        } else if (logObject.totalPoints == null) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Reward.toString(),
                "total_points"
            )
        } else if (logObject.action == RewardAction.Redeem && logObject.redeem == null) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Reward.toString(),
                "redeem_object"
            )
        }
        return verifyRewardEventForRedeem(logObject.action, logObject.redeem)
    }


    private fun verifyRewardEventForRedeem(actionValue: RewardAction, redeemItemValue: RedeemItem?): Boolean {
        if (actionValue == RewardAction.Redeem && redeemItemValue != null) {

            if ((redeemItemValue.pointsWithdrawn ?: 0f) < 0) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Reward.toString(),
                    "points_withdrawn"
                )
            } else if (!CoreConstants.enumContainsString<RedeemType>(redeemItemValue.type.toString())) {
                return ExceptionManager.throwEnumException(
                    LoyaltyEventType.Reward.toString(),
                    RedeemType::class.java.simpleName
                )
            } else if ((redeemItemValue.convertedValue ?: 0f) < 0) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Reward.toString(),
                    "converted_value"
                )
            } else if (redeemItemValue.isSuccessful == null) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Reward.toString(),
                    "redeem is_successful"
                )
            } else if (redeemItemValue.type == RedeemType.Cash && !CoreConstants.enumContains<CurrencyCode>(
                    redeemItemValue.currency!!.name
                )
            ) {
                return ExceptionManager.throwEnumException(
                    LoyaltyEventType.Reward.toString(),
                    CurrencyCode::class.java.simpleName
                )
            }
        }
        return true
    }

    fun isValidSurveyEvent(logObject: SurveyEventObject): Boolean {
        if (!CoreConstants.enumContainsString<SurveyAction>(logObject.action.toString())) {
            return ExceptionManager.throwEnumException(
                LoyaltyEventType.Survey.toString(),
                SurveyAction::class.java.simpleName
            )
        } else if (logObject.action == SurveyAction.Submit && logObject.responseList.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Survey.toString(),
                "response_list"
            )
        } else if (logObject.survey.id.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(
                LoyaltyEventType.Survey.toString(),
                "survey_id"
            )
        } else if (!CoreConstants.enumContainsString<SurveyType>(logObject.survey.type.toString())) {
            return ExceptionManager.throwEnumException(
                LoyaltyEventType.Survey.toString(),
                SurveyType::class.java.simpleName
            )
        }
        return isValidSurveyResponseList(logObject.responseList)
    }

    private fun isValidSurveyResponseList(responseList: List<SurveyResponseItem>): Boolean {
        for (item in responseList) {
            if (!CoreConstants.enumContainsString<SurveyType>(item.type.toString())) {
                return ExceptionManager.throwEnumException(
                    LoyaltyEventType.Survey.toString(),
                    SurveyType::class.java.simpleName
                )
            } else if (item.id.isEmpty()) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Survey.toString(),
                    "response_question_id"
                )
            } else if (item.question.isEmpty()) {
                return ExceptionManager.throwIsRequiredException(
                    LoyaltyEventType.Survey.toString(),
                    "response_question"
                )
            }
        }
        return true
    }

    fun verifySurveyCatalog(
        surveyCatalogModel: SurveyCatalogModel
    ): InternalSurveyModel {

        val catalogName = CatalogSubject.Survey.toString() + " catalog"

        if (surveyCatalogModel.surveyId.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey Id")
        }

        if (surveyCatalogModel.name.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey name")
        } else if (surveyCatalogModel.type.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey type")
        } else if (surveyCatalogModel.questionsList.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "survey questions")
        }

        return InternalSurveyModel(
            id = surveyCatalogModel.surveyId,
            name = CoreConstants.checkIfNull(surveyCatalogModel.name),
            duration = surveyCatalogModel.duration ?: 0,
            type = CoreConstants.checkIfNull(surveyCatalogModel.type),
            rewardId = CoreConstants.checkIfNull(surveyCatalogModel.rewardId),
            questionsList = surveyCatalogModel.questionsList ?: arrayListOf(),
            description = CoreConstants.checkIfNull(surveyCatalogModel.description),
            organizationId = CoreConstants.checkIfNull(surveyCatalogModel.organizationId),
            organizationName = CoreConstants.checkIfNull(surveyCatalogModel.organizationName),
            creationDate = Converters.convertMillisToTimeString(
                surveyCatalogModel.creationDate ?: 0
            ),
            expiryDate = Converters.convertMillisToTimeString(surveyCatalogModel.expiryDate ?: 0),
        )

    }

    fun verifyRewardCatalog(
        rewardCatalogModel: RewardCatalogModel
    ): InternalRewardModel {

        val catalogName = CatalogSubject.Reward.toString() + " catalog"

        if (rewardCatalogModel.rewardId.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward Id")
        }

        if (rewardCatalogModel.name.isEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward name")
        } else if (rewardCatalogModel.type.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(catalogName, "reward type")
        } else if ((rewardCatalogModel.requiredPoints ?: 1f) < 1) {
            ExceptionManager.throwIsRequiredException(catalogName, "required_points")
        }

        return InternalRewardModel(
            id = rewardCatalogModel.rewardId,
            name = CoreConstants.checkIfNull(rewardCatalogModel.name),
            description = CoreConstants.checkIfNull(rewardCatalogModel.description),
            type = CoreConstants.checkIfNull(rewardCatalogModel.type),
            requiredPoints = rewardCatalogModel.requiredPoints ?: 1f,
            creationDate = Converters.convertMillisToTimeString(
                rewardCatalogModel.creationDate ?: 0
            ),
            expiryDate = Converters.convertMillisToTimeString(rewardCatalogModel.expiryDate ?: 0),
            organizationId = CoreConstants.checkIfNull(rewardCatalogModel.organizationId),
            organizationName = CoreConstants.checkIfNull(rewardCatalogModel.organizationName)

        )

    }


}
