package ai.chronon.online

import ai.chronon.api.Constants
import ai.chronon.online.Fetcher.{Request, Response}

import scala.collection.{Seq, mutable}
import scala.concurrent.{ExecutionContext, Future}
import scala.util.ScalaJavaConversions.IterableOps
import scala.util.{Failure, Success}

// users can simply register external endpoints with a lambda that can return the future of a response given keys
// keys and values need to match schema in ExternalSource - chronon will validate automatically
class ExternalSourceRegistry extends Serializable {
  class ContextualHandler extends ExternalSourceHandler {
    override def fetch(requests: Seq[Request]): Future[Seq[Response]] = {
      Future(requests.map { request =>
        Response(request = request, values = Success(request.keys))
      })
    }
  }

  val handlerMap: mutable.Map[String, ExternalSourceHandler] = {
    val result = new mutable.HashMap[String, ExternalSourceHandler]()
    result.put(Constants.ContextualSourceName, new ContextualHandler())
    result
  }

  def add(name: String, handler: ExternalSourceHandler): Unit = {
    assert(!handlerMap.contains(name),
           s"A handler by the name $name already exists. Existing: ${handlerMap.keys.mkString("[", ", ", "]")}")
    handlerMap.put(name, handler)
  }

  // TODO: validation of externally fetched data
  // 1. keys match
  // 2. report missing & extra values
  // 3. schema integrity of returned values
  def fetchRequests(requests: Seq[Request], context: Metrics.Context)(implicit
      ec: ExecutionContext): Future[Seq[Response]] = {
    val startTime = System.currentTimeMillis()
    // we make issue one batch request per external source and flatten out it later
    val responsesByNameF: List[Future[Seq[Response]]] = requests
      .groupBy(_.name)
      .map {
        case (name, requests) =>
          if (handlerMap.contains(name)) {
            val ctx = context.copy(groupBy = s"${Constants.ExternalPrefix}_$name")
            val responses = handlerMap(name).fetch(requests)
            responses.map { responses =>
              val failures = responses.count(_.values.isFailure)
              ctx.histogram("response.latency", System.currentTimeMillis() - startTime)
              ctx.histogram("response.failures", failures)
              ctx.histogram("response.successes", responses.size - failures)
              responses
            }
          } else {
            val failure = Failure(
              new IllegalArgumentException(
                s"$name is not registered among handlers: [${handlerMap.keys.mkString(", ")}]"))
            Future(requests.map(request => Response(request, failure)))
          }
      }
      .toList

    Future.sequence(responsesByNameF).map { responsesByName =>
      val allResponses = responsesByName.flatten
      requests.map(req =>
        allResponses
          .find(_.request == req)
          .getOrElse(Response(
            req,
            Failure( // logic error - some handler missed returning a response
              new IllegalStateException(s"Missing response for request $req among \n $allResponses"))
          )))
    }
  }
}
