package ai.connectif.sdk.data.source

import ai.connectif.sdk.Utils
import ai.connectif.sdk.data.error.ConnectifApiException
import ai.connectif.sdk.data.error.UnauthorizedApiException
import ai.connectif.sdk.data.error.ValidationException
import ai.connectif.sdk.data.model.push.AddPushTokenInternal
import ai.connectif.sdk.data.model.push.TrackPushClickData
import ai.connectif.sdk.data.model.push.TrackPushOpenData
import ai.connectif.sdk.data.repository.SettingRepository
import ai.connectif.sdk.data.source.model.response.SendEventResponse
import retrofit2.HttpException

internal class PushDataSourceImpl(
    private val settingRepository: SettingRepository,
    private val localPushTokenDataSource: LocalPushTokenDataSource
) : PushDataSource {

    private suspend fun <T> safeApiCall(apiCall: suspend () -> T): T? = try {
        apiCall()
    } catch (e: HttpException) {
        when (e.code()) {
            422 -> throw ValidationException.fromHttpException(e)
            401 -> throw UnauthorizedApiException("Invalid api key, please check it.", e)
            else -> throw ConnectifApiException(e.message(), e)
        }
    }

    override suspend fun addPushToken(
        addPushTokenInternal: AddPushTokenInternal
    ): SendEventResponse? = safeApiCall {
        localPushTokenDataSource.pushToken = addPushTokenInternal.token
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postAddPushToken(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                addPushTokenInternal.mapToApiModel()
            )
    }

    override suspend fun trackPushClick(trackPushClickData: TrackPushClickData) = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postPushClick(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                trackPushClickData.mapToApiModel()
            )
    }

    override suspend fun trackPushOpen(trackPushOpenData: TrackPushOpenData) = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postPushOpen(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                trackPushOpenData.mapToApiModel()
            )
    }
}
