package ai.connectif.sdk.manager.push

import ai.connectif.sdk.Constants
import ai.connectif.sdk.data.ObjectId
import ai.connectif.sdk.data.model.push.Action
import ai.connectif.sdk.data.model.push.AddPushTokenInternal
import ai.connectif.sdk.data.model.push.ExtraNotificationData
import ai.connectif.sdk.data.model.push.NotificationData
import ai.connectif.sdk.data.model.push.TrackPushClickData
import ai.connectif.sdk.data.model.push.TrackPushOpenData
import ai.connectif.sdk.data.repository.SettingRepository
import ai.connectif.sdk.data.source.LocalPushTokenDataSource
import ai.connectif.sdk.data.source.PushDataSource
import ai.connectif.sdk.data.source.model.response.SendEventResponse
import ai.connectif.sdk.data.source.typeadapter.ObjectIdTypeAdapter
import ai.connectif.sdk.manager.DeviceInfoManager
import ai.connectif.sdk.manager.Logger
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.content.res.Resources
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.os.Build
import androidx.core.app.NotificationCompat
import com.google.gson.Gson
import com.google.gson.GsonBuilder
import com.google.gson.JsonSyntaxException
import com.google.gson.reflect.TypeToken
import java.io.IOException
import java.net.URL
import java.util.Date
import java.util.Random
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

internal class PushManagerImpl(
    private val pushDataSource: PushDataSource,
    private val settingRepository: SettingRepository,
    private val deviceInfoManager: DeviceInfoManager,
    private val localPushTokenDataSource: LocalPushTokenDataSource
) : PushManager {

    private fun createChannel(notificationManager: NotificationManager) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val importance = NotificationManager.IMPORTANCE_DEFAULT
            val channel = NotificationChannel(
                settingRepository.pushChannelId,
                settingRepository.pushChannelName,
                importance
            )
            notificationManager.createNotificationChannel(channel)
        }
    }

    override fun handleConnectifPushNotification(
        notificationData: NotificationData,
        context: Context
    ) {
        CoroutineScope(Dispatchers.IO).launch {
            Logger.i(notificationData.toString())
            val notificationManager =
                context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

            checkNotificationChannel(notificationManager)
            showNotification(notificationData, notificationManager, context)
            trackPushOpen(notificationData)
        }
    }

    override fun parseToNotificationData(data: String): NotificationData {
        val gson: Gson = GsonBuilder()
            .registerTypeAdapter(ObjectId::class.java, ObjectIdTypeAdapter())
            .create()

        return try {
            gson.fromJson(
                data,
                object : TypeToken<NotificationData>() {}.type
            )
        } catch (e: Exception) {
            throw IllegalArgumentException(
                "Malformed Json on notification data: " + e.message
            )
        }
    }

    internal fun getActionData(actions: String): List<Action>? {
        val gson = Gson()
        return try {
            gson.fromJson(
                actions,
                object : TypeToken<List<Action>>() {}.type
            )
        } catch (e: JsonSyntaxException) {
            throw IllegalArgumentException(
                "Malformed Json on push actions: " + e.message
            )
        }
    }

    override suspend fun trackPushClick(trackPushClickData: TrackPushClickData) {
        try {
            trackPushClickData.trackerId = settingRepository.trackerId
            trackPushClickData.deviceInfo = deviceInfoManager.getDeviceInfo()
            pushDataSource.trackPushClick(trackPushClickData)
        } catch (exception: Exception) {
            Logger.e("An error occurred while trying to add track push click")
        }
    }

    override suspend fun trackPushOpen(notificationData: NotificationData) {
        try {
            val trackPushOpenData = TrackPushOpenData(notificationData)
            trackPushOpenData.trackerId = settingRepository.trackerId
            trackPushOpenData.deviceInfo = deviceInfoManager.getDeviceInfo()
            pushDataSource.trackPushOpen(trackPushOpenData)
        } catch (exception: Exception) {
            Logger.e("An error occurred while trying to add track push open")
        }
    }

    private fun handleResult(result: SendEventResponse?) {
        if (result?.trackerId != null) {
            settingRepository.trackerId = result.trackerId
        }
    }

    private suspend fun showNotification(
        notificationData: NotificationData,
        notificationManager: NotificationManager,
        context: Context
    ) {
        val notificationRequestCode = Random().nextInt()

        val notificationBuilder =
            NotificationCompat.Builder(context, settingRepository.pushChannelId)
                .setContentTitle(notificationData.title)
                .setContentText(notificationData.body)
                .setSmallIcon(getSmallIcon(context))
                .setGroup(notificationData.sendUuid)
                .setAutoCancel(true)

        notificationData.imageUrl?.let { imageUrl ->
            getBitmapFromUrl(imageUrl)?.also { bitmap ->
                notificationBuilder.setStyle(
                    NotificationCompat.BigPictureStyle().bigPicture(bitmap)
                )
            }
        }

        notificationData.actions?.forEach {
            notificationBuilder.addAction(
                0,
                it.title,
                generateActionPendingIntent(
                    extraNotificationData = ExtraNotificationData(
                        notificationRequestCode,
                        notificationData.sendUuid
                    ),
                    trackPushClickData = TrackPushClickData(
                        notificationData.workflowDefinitionId,
                        notificationData.contentId,
                        notificationData.sendUuid,
                        linkUrl = it.redirectUrl,
                        originalLinkUrl = it.originalLinkUrl
                    ),
                    context = context
                )
            )
        }

        notificationBuilder.setContentIntent(
            generatePendingIntent(
                extraNotificationData = ExtraNotificationData(
                    notificationRequestCode,
                    notificationData.sendUuid
                ),
                trackPushClickData = TrackPushClickData(
                    notificationData.workflowDefinitionId,
                    notificationData.contentId,
                    notificationData.sendUuid,
                    linkUrl = notificationData.linkUrl,
                    originalLinkUrl = notificationData.originalLinkUrl
                ),
                context = context
            )
        )
        notificationManager.notify(
            notificationData.sendUuid,
            notificationRequestCode,
            notificationBuilder.build()
        )
    }

    internal fun generatePendingIntent(
        extraNotificationData: ExtraNotificationData,
        trackPushClickData: TrackPushClickData,
        context: Context
    ): PendingIntent = PendingIntent.getActivity(
        context,
        extraNotificationData.requestCode,
        getTrackPushClickActivityIntent(extraNotificationData, trackPushClickData, context),
        PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
    )

    internal fun generateActionPendingIntent(
        extraNotificationData: ExtraNotificationData,
        trackPushClickData: TrackPushClickData,
        context: Context
    ): PendingIntent = PendingIntent.getActivity(
        context,
        Random().nextInt(),
        getTrackPushClickActivityIntent(extraNotificationData, trackPushClickData, context),
        PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
    )

    private fun getTrackPushClickActivityIntent(
        extraNotificationData: ExtraNotificationData,
        trackPushClickData: TrackPushClickData,
        context: Context
    ): Intent = Intent(context, TrackPushClickActivity::class.java).apply {
        putExtra(Constants.PushNotification.EXTRA_TRACK_DATA, trackPushClickData)
        putExtra(
            Constants.PushNotification.EXTRA_NOTIFICATION_DATA,
            extraNotificationData
        )
        addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TOP)
    }

    suspend fun getBitmapFromUrl(url: String): Bitmap? = withContext(Dispatchers.IO) {
        try {
            val input = URL(url).openStream()
            BitmapFactory.decodeStream(input)
        } catch (e: IOException) {
            Logger.e("Error downloading the image for push")
            null
        }
    }

    private fun checkNotificationChannel(notificationManager: NotificationManager) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channelList = notificationManager.notificationChannels
            for (channel in channelList) {
                if (channel.id == settingRepository.pushChannelId) {
                    return
                }
            }
            createChannel(notificationManager)
        }
        return
    }

    private fun getSmallIcon(context: Context): Int {
        try {
            context.resources.getDrawable(settingRepository.pushSmallIcon, null)
        } catch (e: Resources.NotFoundException) {
            Logger.e(
                "Error when trying to find the small icon for resource " +
                    "${settingRepository.pushSmallIcon}, check the settings"
            )
        }

        return settingRepository.pushSmallIcon
    }

    override suspend fun addPushToken(token: String) {
        if (localPushTokenDataSource.needSyncToken(token)) {
            val pushData = AddPushTokenInternal(token)
            pushData.trackerId = settingRepository.trackerId
            pushData.deviceInfo = deviceInfoManager.getDeviceInfo()
            handleResult(pushDataSource.addPushToken(pushData))
            localPushTokenDataSource.pushToken = token
            localPushTokenDataSource.lastUpdate = Date()
        }
    }

    override suspend fun pushStoredPushToken() {
        localPushTokenDataSource.pushToken?.let { token ->
            addPushToken(token)
        }
    }
}
