package ai.connectif.sdk.model

import ai.connectif.sdk.data.source.model.request.ProductRequest
import ai.connectif.sdk.model.Product.Availability
import java.math.BigDecimal
import java.util.Date

/**
 * Represents a product in the Connectif
 *
 * @property productDetailUrl Product page URL (URL must be unencoded).
 * @property productId Unique product identifier (in the eCommerce system).
 * @property name Product name.
 * @property unitPrice Product unit price.
 * @property description (Optional) Product description. It can contain HTML code.
 * @property imageUrl (Optional) Product image URL.
 * @property availability (Optional) Product availability status.
 *  Can be either [Availability.IN_STOCK] or [Availability.OUT_OF_STOCK].
 * @property categories (Optional) Categories the product belongs to.
 *  If the eCommerce allows subcategories, you can indicate them with the full category path,
 *  separating each level with the '/' character.
 *  For example: "/Computers/Keyboards/Logitech".
 *  the full path (e.g. "/Computers/Keyboards/Logitech").
 * @property tags (Optional) Tags associated with the product.
 * @property brand (Optional) Product brand.
 * @property reviewCount (Optional) Number of product reviews.
 * @property rating (Optional) Product rating on a scale of 0 to 5.
 * @property ratingCount (Optional) Number of product ratings.
 * @property thumbnailUrl (Optional) Product thumbnail image URL.
 * @property relatedExternalProductIds (Optional) IDs of related products.
 * @property priority (Optional) Product priority.
 * @property unitPriceOriginal (Optional) Original product unit price before any discount.
 * @property unitPriceWithoutVAT (Optional) Product unit price without VAT.
 * @property discountedAmount (Optional) Discounted amount for the product.
 * @property discountedPercentage (Optional) Discount percentage (0–100, no '%' symbol).
 * @property publishedAt (Optional) Date the product was published in the store catalog.
 * @property customField1 (Optional) Custom product field.
 * @property customField2 (Optional) Custom product field.
 * @property customField3 (Optional) Custom product field.
 */
data class Product(
    val productDetailUrl: String,
    val productId: String,
    val name: String,
    val unitPrice: BigDecimal,
    val description: String? = null,
    val imageUrl: String? = null,
    val availability: Availability? = null,
    val categories: List<String>? = null,
    val tags: List<String>? = null,
    val brand: String? = null,
    val reviewCount: Int? = null,
    val rating: Double? = null,
    val ratingCount: Int? = null,
    val thumbnailUrl: String? = null,
    val relatedExternalProductIds: List<String>? = null,
    val priority: Int? = null,
    val unitPriceOriginal: BigDecimal? = null,
    val unitPriceWithoutVAT: BigDecimal? = null,
    val discountedAmount: BigDecimal? = null,
    val discountedPercentage: Double? = null,
    val publishedAt: Date? = null,
    val customField1: String? = null,
    val customField2: String? = null,
    val customField3: String? = null
) {
    internal fun mapTopProductVisit(): ProductRequest = ProductRequest(
        productDetailUrl = this.productDetailUrl,
        productId = this.productId,
        name = this.name,
        unitPrice = this.unitPrice,
        description = this.description,
        imageUrl = this.imageUrl,
        availability = this.availability?.value,
        categories = this.categories,
        tags = this.tags,
        brand = this.brand,
        reviewCount = this.reviewCount,
        rating = this.rating,
        ratingCount = this.ratingCount,
        thumbnailUrl = this.thumbnailUrl,
        relatedExternalProductIds = this.relatedExternalProductIds,
        priority = this.priority,
        unitPriceOriginal = this.unitPriceOriginal,
        unitPriceWithoutVAT = this.unitPriceWithoutVAT,
        discountedAmount = this.discountedAmount,
        discountedPercentage = this.discountedPercentage,
        publishedAt = this.publishedAt,
        customField1 = this.customField1,
        customField2 = this.customField2,
        customField3 = this.customField3
    )

    enum class Availability(val value: String) {
        IN_STOCK("instock"),
        OUT_OF_STOCK("outofstock")
    }
}
