package ai.engagely.openbot.model.repositories.impl

import ai.engagely.openbot.model.network.ApiClient
import ai.engagely.openbot.model.network.ApiConstants
import ai.engagely.openbot.model.network.interfaces.LiveChatApi
import ai.engagely.openbot.model.pojos.external.apirequests.livechat.*
import ai.engagely.openbot.model.pojos.internal.livechat.ILiveChatForm
import ai.engagely.openbot.model.repositories.LiveChatRepository
import ai.engagely.openbot.model.utils.general.LogUtils
import com.google.gson.Gson
import io.socket.client.Ack
import io.socket.client.Socket
import kotlinx.coroutines.*
import java.net.HttpURLConnection
import kotlin.coroutines.resume

class AppLiveChatRepository(
    private val dispatcher: CoroutineDispatcher,
    private val socket: Socket,
) : LiveChatRepository {

    override suspend fun getLiveAgentConnectionDetails(
        intId: String?,
        sessionId: String?
    ): ILiveChatForm? {
        return withContext(dispatcher) {
            try {
                val agentDetailsApi =
                    ApiClient.getInstance().getClient().create(LiveChatApi::class.java)
                val liveAgentConnectionDetailsRequest =
                    LiveAgentConnectionDetailsRequest(intId = intId, sessionId = sessionId)
                val response =
                    agentDetailsApi.getLiveAgentConnectionDetails(liveAgentConnectionDetailsRequest)
                if (response.isSuccessful && response.code() == HttpURLConnection.HTTP_OK
                    && response.body()?.success == true
                ) {
                    response.body()?.response?.let {
                        return@withContext ILiveChatForm(
                            displayField = it.displayField,
                            displayOutput = it.displayOutput,
                            field = it.field,
                            formData = it.formData,
                            inputMethod = it.inputMethod,
                            inputType = it.inputType,
                            settingId = it.settingId,
                            value = it.value
                        )
                    }

                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            return@withContext null
        }
    }

    override suspend fun sendAgentConnectRequest(
        botId: String?,
        channel: String,
        displayOutput: String?,
        fetchApi: Boolean,
        formData: Any?,
        prevLangCode: String?,
        sessionId: String?,
        settingId: String?,
        userId: String?,
        userIntId: String?,
        userIntName: String?,
        osName: String?,
        osVersion: String?,
        screenWidth: Int?,
        screenHeight: Int?,
        screenOrientation: String?,
        userName: String?
    ): Boolean {

        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val agentConnectRequest = Gson().toJson(
                        AgentConnectRequest(
                            botId = botId,
                            channel = channel,
                            displayOutput = displayOutput,
                            fetchApi = fetchApi,
                            formData = formData,
                            prevLangCode = prevLangCode,
                            sessionId = sessionId,
                            settingId = settingId,
                            userId = userId,
                            userIntId = userIntId,
                            userIntName = userIntName,
                            userProperty = UserProperty(
                                deviceInfo = DeviceInfo(
                                    browser = Browser(
                                        name = "Chrome",
                                        version = "100.0.4896.127",
                                        major = "100",
                                    ),
                                    os = Os(
                                        name = osName,
                                        version = osVersion,
                                    ),
                                    screen = Screen(
                                        height = screenHeight,
                                        width = screenWidth,
                                        orientation = "landscape-primary",
                                        pixelDepth = 30,
                                    )
                                ),
                                engagementInfo = ArrayList<EngagementInfo>()
                                    .apply {
                                        add(
                                            EngagementInfo(
                                                objUpdatedAt = 1650523185301,
                                                timeOnPage = 394745000,
                                                pageViewsCount = 353,
                                                url = "https://engagelybots.ai/chatbot/"
                                            )
                                        )
                                    },
                                entity = ArrayList(),
                                intent = ArrayList<String>().apply { add("receive") },
                                locationInfo = LocationInfo(
                                    latitude = 20,
                                    longitude = 77
                                )
                            ),
                            userName = userName
                        )
                    )
                    emitMessage(
                        ApiConstants.CHANNEL_AGENT_CONNECT_REQUEST,
                        agentConnectRequest,
                        cnt
                    )
                }
            } ?: false
        }
    }

    private fun emitMessage(
        channelToSendTo: String,
        flowRequest: String?,
        cnt: CancellableContinuation<Boolean?>
    ) {
        if (socket.connected()) {
            LogUtils.log("Emitting on channel = ${channelToSendTo}, message = $flowRequest")
            socket.emit(channelToSendTo, flowRequest, object : Ack {
                override fun call(vararg args: Any?) {
                    LogUtils.log("Callback from emitting $channelToSendTo $args")
                    cnt.resume(true)
                }
            })
        } else {
            cnt.resume(false)
        }
    }
}