package ai.engagely.openbot.model.utils.general

import android.os.Build
import android.text.format.DateUtils
import androidx.annotation.RequiresApi
import java.text.SimpleDateFormat
import java.time.ZonedDateTime
import java.time.format.DateTimeFormatter
import java.util.*

object DateUtils {

    private const val ISO8601_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"
    private const val SIMPLE_TIME = "h:mm a"
    private const val SIMPLE_DATE = "dd/MM/yyyy"
    private const val SERVER_TIME_FORMAT = "EEE, dd MMM yyyy HH:mm:ss z"
    private const val SERVER_DATE_FORMAT = "yyyy-MM-dd"
    const val INVALID_TIME = "--:--"

    fun getCurrentTimeInISO8601(): String {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            getZonedDateTimeInIso8601()
        } else {
            getSimpleDateTimeInIso8601()
        }
    }

    fun getSimpleDateTimeInIso8601(): String {
        val dateFormat = SimpleDateFormat(ISO8601_FORMAT, Locale.getDefault())
        dateFormat.timeZone = TimeZone.getTimeZone("UTC")
        return dateFormat.format(Date())
    }


    @RequiresApi(Build.VERSION_CODES.O)
    fun getZonedDateTimeInIso8601(): String =
        ZonedDateTime.now().format(DateTimeFormatter.ISO_INSTANT)

    fun getCurrentTime(): String {
        return SimpleDateFormat(SIMPLE_TIME, Locale.getDefault()).format(Date())
    }

    fun convertFromGmtToLocalTime(gmtTime: String?): String {
        if (gmtTime.isNullOrBlank()) {
            return INVALID_TIME
        }
        return try {
            convertFromUtcToLocalDate(gmtTime)?.let {
                val localTimeFormat = SimpleDateFormat(SIMPLE_TIME, Locale.getDefault())
                localTimeFormat.format(it)
            } ?: INVALID_TIME
        } catch (e: Exception) {
            INVALID_TIME
        }
    }

    fun convertFromUtcToLocalDate(gmtTime: String?): Date? {
        if (gmtTime.isNullOrBlank()) {
            return null
        }
        return try {
            val serverDateFormat = SimpleDateFormat(SERVER_TIME_FORMAT, Locale.getDefault())
            serverDateFormat.timeZone = TimeZone.getTimeZone("UTC")
            serverDateFormat.parse(gmtTime)
        } catch (e: Exception) {
            null
        }
    }

    fun convertFromLocalDateToTime(date: Date?): String {
        if (date == null) {
            return INVALID_TIME
        }
        val dateFormat = SimpleDateFormat(SIMPLE_TIME, Locale.getDefault())
        return dateFormat.format(date)
    }

    fun convertFromLocalDateToDateString(date: Date?): String {
        if (date == null) {
            return INVALID_TIME
        }
        val dateFormat = SimpleDateFormat(SIMPLE_DATE, Locale.getDefault())
        return dateFormat.format(date)
    }

    fun isTodayDate(date: Date?): Boolean {
        date?.let {
            return DateUtils.isToday(it.time)
        } ?: run { return false }
    }

    fun isFutureOrToday(date: Date?): Boolean {
        date?.let {
            return it.time >= Date().time || isTodayDate(it)
        } ?: run { return false }
    }

    fun isPastOrToday(date: Date?): Boolean {
        date?.let {
            return it.time <= Date().time || isTodayDate(it)
        } ?: run { return false }
    }

    fun convertFromServerDate(dateString: String?): Date? {
        try {
            if (!dateString.isNullOrBlank()) {
                val dateFormat = SimpleDateFormat(SERVER_DATE_FORMAT, Locale.getDefault())
                return dateFormat.parse(dateString)
            }
        } catch (ignore: Exception) {
        }
        return null
    }

    fun convertToServerDateFormat(date: Date): String {
        return SimpleDateFormat(SERVER_DATE_FORMAT, Locale.getDefault()).format(date)
    }
}