package ai.engagely.openbot.model.utils.general

import android.content.ContentResolver
import android.content.Context
import android.net.Uri
import android.provider.OpenableColumns
import android.text.format.Formatter
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.io.File
import java.io.FileOutputStream
import java.util.*

object FileUtils {

    suspend fun fetchFileMetadataFromUri(
        fileUri: Uri,
        contentResolver: ContentResolver
    ): FileMetadata? {
        return withContext(Dispatchers.Default) {
            contentResolver.getType(fileUri)?.let { mimeType ->
                contentResolver.query(fileUri, null, null, null, null)?.use { cursor ->
                    val nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME)
                    val sizeIndex = cursor.getColumnIndex(OpenableColumns.SIZE)
                    cursor.moveToFirst()

                    val name = cursor.getString(nameIndex)
                    val size = cursor.getLong(sizeIndex)

                    return@withContext FileMetadata(
                        fileName = name,
                        fileSize = size,
                        mimeType = mimeType
                    )
                }
            }
            return@withContext null
        }

    }

    suspend fun createTempFileForUri(
        context: Context,
        contentResolver: ContentResolver,
        fileUri: Uri,
        fileMetadata: FileMetadata
    ): File? {
        return withContext(Dispatchers.IO) {
            try {
                contentResolver.openInputStream(fileUri)?.use { inputStream ->

                    val fileParent = File(context.cacheDir, Date().time.toString())
                    fileParent.mkdir()
                    val tempFile = File(fileParent, fileMetadata.fileName)
                    tempFile.createNewFile()
                    tempFile.deleteOnExit()
                    fileParent.deleteOnExit()
                    FileOutputStream(tempFile).use { outputStream ->
                        val buffer = ByteArray(1024)
                        while (true) {
                            val bytesRead = inputStream.read(buffer)
                            if (bytesRead == -1) {
                                break
                            }
                            outputStream.write(buffer, 0, bytesRead);
                        }
                    }
                    return@withContext tempFile
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            return@withContext null
        }
    }

    fun deleteParentAndSelfIfExists(file: File?) {
        try {
            if (file?.parentFile?.exists() == true) {
                file.parentFile?.deleteRecursively()
            }
        } catch (ignore: Exception) {
        }
    }

    fun formatToFileSize(context: Context, size: Long): String {
        return try {
            Formatter.formatFileSize(context, size)
        } catch (e: Exception) {
            ""
        }
    }

    data class FileMetadata(
        val fileName: String,
        val fileSize: Long,
        val mimeType: String
    )
}