package ai.engagely.openbot.model.utils.helpers

import ai.engagely.openbot.model.constants.ViewConstants
import ai.engagely.openbot.model.pojos.internal.botsettings.IBotChatWindowConfig
import ai.engagely.openbot.model.utils.general.ViewUtils
import ai.engagely.openbot.view.activities.BotChatActivity
import ai.engagely.openbot.view.activities.ExoPlayerActivity
import ai.engagely.openbot.view.customviews.BotContainerView
import ai.engagely.openbot.view.customviews.WelcomeVideoView
import ai.engagely.openbot.viewmodel.OpenBotViewModel
import ai.engagely.openbot.viewmodel.VideoPlayerViewModel
import android.app.Activity
import android.content.Intent
import android.content.res.Configuration
import android.view.View
import androidx.activity.result.contract.ActivityResultContracts
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.ViewModelProvider
import com.google.android.exoplayer2.ui.PlayerView

class BotViewHelper(
    activity: FragmentActivity,
    private val botViewContainerView: BotContainerView?
) : DefaultLifecycleObserver {

    private val openBotViewModel by lazy {
        ViewModelProvider(activity)[OpenBotViewModel::class.java]
    }

    private val playerViewModel by lazy {
        ViewModelProvider(activity)[VideoPlayerViewModel::class.java]
    }

    private var playerView: PlayerView? = null

    private val chatResult =
        activity.registerForActivityResult(ActivityResultContracts.StartActivityForResult()) {
            it?.let { activityResult ->
                openBotViewModel.resetWelcomeData()
                activityResult.data?.getParcelableExtra<IBotChatWindowConfig>(IBotChatWindowConfig.UNIQUE_NAME)
                    ?.let { botConfigData ->
                        openBotViewModel.updateBotConfigData(botConfigData)
                    }
                when (activityResult.resultCode) {
                    RESULT_SESSION_UPDATE_REQUESTED -> {
                        openBotViewModel.refreshSession()
                    }
                }
            }
        }

    private val exoPlayerResult =
        activity.registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { activityResult ->
            if (Activity.RESULT_OK == activityResult.resultCode) {
                activityResult.data?.let { resultData ->
                    resultData.getStringExtra(ExoPlayerActivity.VIDEO_URL_TAG)?.let { url ->
                        playerViewModel.updateInitialData(
                            resultData.getLongExtra(
                                ExoPlayerActivity.VIDEO_CURRENT_POSITION_TAG,
                                0
                            ),
                            resultData.getBooleanExtra(
                                ExoPlayerActivity.VIDEO_PLAY_WHEN_READY_TAG,
                                false
                            ),
                            url
                        )
                    }
                }
            }
        }

    init {
        activity.lifecycle.addObserver(this)
        setObservers(activity)
        setClickListeners(activity)
    }

    fun init(botId: String, extras: Map<String, String>? = null) {
        openBotViewModel.init(botId, extras)
        openBotViewModel.fetchData()
    }

    private fun setClickListeners(activity: FragmentActivity) {
        botViewContainerView?.getBotView()?.setImageClickListener {
            val botChatConfig = openBotViewModel.botChatWindowConfig
            botChatConfig?.let {
                chatResult.launch(Intent(activity, BotChatActivity::class.java).apply {
                    putExtra(IBotChatWindowConfig.UNIQUE_NAME, botChatConfig)
                    putExtras(openBotViewModel.getExtras())
                })
            }
        }
        botViewContainerView?.getBotView()?.setImageLongClickListener {
            if (openBotViewModel.welcomeDataLiveData.value?.isNotEmpty() == true) {
                botViewContainerView.getWelcomeData()?.visibility = View.VISIBLE
                scrollWelcomeDataToEnd()
                true
            } else {
                false
            }
        }
        botViewContainerView?.getBotView()?.setErrorRetryClickListener {
            openBotViewModel.fetchData()
        }
    }

    private fun setObservers(activity: FragmentActivity) {
        openBotViewModel.isLoadingInitialApi.observe(activity) { value ->
            botViewContainerView?.getBotView()?.setLoading(value)
        }
        openBotViewModel.imageBitmapLiveData.observe(activity) { value ->
            value?.let { bitmap ->
                botViewContainerView?.getBotView()?.setImage(bitmap)
            }
        }
        openBotViewModel.errorStateLiveData.observe(activity) {
            botViewContainerView?.getBotView()?.setErrorView(it != null)
        }
        openBotViewModel.welcomeDataLiveData.observe(activity) {
            if (it?.isNotEmpty() == true) {
                val themeColor = ViewUtils.parseColorSafely(
                    openBotViewModel.botChatWindowConfig?.botThemColor,
                    ViewConstants.VIEW_DEFAULT_COLOR_STRING_DARK
                )
                botViewContainerView?.getWelcomeData()
                    ?.setData(it, themeColor, object : WelcomeVideoView.OnPlayClickListener {
                        override fun onPlayClicked(playerView: PlayerView?, url: String?) {
                            playerViewModel.startPlaying(playerView, url)
                        }

                        override fun onFullScreenClicked(playerView: PlayerView?) {
                            playerViewModel.getUrl()?.let { url ->
                                val currentPosition: Long = playerViewModel.getCurrentPosition()
                                val playWhenReady: Boolean = playerViewModel.getPlayWhenReady()
                                ExoPlayerActivity.getRequestIntent(
                                    activity,
                                    currentPosition,
                                    playWhenReady,
                                    url
                                ).also { intent ->
                                    playerViewModel.stopPlayingAndSaveState()
                                    exoPlayerResult.launch(intent)
                                }
                            }
                            this@BotViewHelper.playerView = playerView
                        }
                    })
                botViewContainerView?.getWelcomeData()?.visibility = View.VISIBLE
                scrollWelcomeDataToEnd()
            } else {
                botViewContainerView?.getWelcomeData()?.visibility = View.GONE
            }
        }
        openBotViewModel.botImageDimensionsLiveData.observe(activity) { botDimension ->
            botViewContainerView?.getBotView()?.setBotDimensions(botDimension)
        }
    }

    private fun scrollWelcomeDataToEnd() {
        botViewContainerView?.getWelcomeDataContainer()?.post {
            botViewContainerView.getWelcomeDataContainer()?.fullScroll(View.FOCUS_DOWN)
        }
    }

    override fun onResume(owner: LifecycleOwner) {
        super.onResume(owner)
        resumePlayingIfReturnedFromVideoPlayer()
    }

    fun onConfigurationChanged(newConfig: Configuration) {
        scrollWelcomeDataToEnd()
    }

    private fun resumePlayingIfReturnedFromVideoPlayer() {
        if (playerView != null) {
            playerViewModel.resumePlaying(playerView)
            playerView = null
        }
    }

    companion object {
        const val RESULT_SESSION_UPDATE_REQUESTED = 999
        const val RESULT_BOT_CONFIG_UPDATED = 1000
    }
}