package ai.engagely.openbot.view.activities

import ai.engagely.openbot.R
import ai.engagely.openbot.databinding.ActivityExoPlayerBinding
import ai.engagely.openbot.viewmodel.VideoPlayerViewModel
import android.app.Activity
import android.content.Intent
import android.content.res.Configuration
import android.os.Bundle
import android.view.View
import android.widget.ImageButton
import androidx.lifecycle.ViewModelProvider

class ExoPlayerActivity : BaseActivity() {

    private lateinit var binding: ActivityExoPlayerBinding
    private var isFullScreen = false
    private lateinit var fullScreenButton: ImageButton

    private val playerViewModel by lazy {
        ViewModelProvider(this)[VideoPlayerViewModel::class.java]
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        binding = ActivityExoPlayerBinding.inflate(layoutInflater)
        setContentView(binding.root)

        setupActionBar()
        setupVideo()

        fullScreenButton = findViewById(R.id.ib_full_screen)
        fullScreenButton.setOnClickListener {
            toggleFullScreen()
        }
    }

    private fun setupVideo() {
        intent.getStringExtra(VIDEO_URL_TAG)?.let { url ->
            playerViewModel.updateInitialData(
                intent.getLongExtra(VIDEO_CURRENT_POSITION_TAG, 0),
                intent.getBooleanExtra(VIDEO_PLAY_WHEN_READY_TAG, true), url
            )
            playerViewModel.startPlaying(binding.playerView, url)

        }
    }

    private fun setupActionBar() {
        setSupportActionBar(binding.toolbar)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.setHomeAsUpIndicator(R.drawable.ic_arrow_left_white)
        supportActionBar?.title = intent.getStringExtra(VIDEO_TITLE_TAG) ?: ""
        supportActionBar?.subtitle =
            intent.getStringExtra(VIDEO_SUBTITLE_TAG) ?: ""
    }

    private fun toggleFullScreen() {
        if (!isFullScreen) {
            hideControls()
        } else {
            showControls()
        }
    }

    private fun showControls() {
        unhideSystemBars()
        hideToolbar(false)
        fullScreenButton.setImageResource(R.drawable.ic_fullscreen)
        isFullScreen = false
    }

    private fun hideControls() {
        hideSystemBars()
        hideToolbar(true)
        fullScreenButton.setImageResource(R.drawable.ic_fullscreen_exit)
        isFullScreen = true
    }

    private fun hideToolbar(shouldHide: Boolean) {
        binding.toolbar.visibility = if (shouldHide) View.GONE else View.VISIBLE
    }

    override fun onSupportNavigateUp(): Boolean {
        onBackPressed()
        return super.onSupportNavigateUp()
    }

    override fun onBackPressed() {
        if (isLandscapeOrientation()) {
            setPortraitOrientation()
        } else {
            setResult(
                Activity.RESULT_OK,
                getResponseIntent(
                    playerViewModel.getCurrentPosition(),
                    playerViewModel.getPlayWhenReady(),
                    playerViewModel.getUrl()
                )
            )
            playerViewModel.stopPlayingAndSaveState()
            super.onBackPressed()
        }
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            hideControls()
        } else if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
            showControls()
        }
    }

    companion object {
        const val VIDEO_URL_TAG = "ExoPlayer_Video_Url"
        const val VIDEO_TITLE_TAG = "ExoPlayer_Video_Title"
        const val VIDEO_SUBTITLE_TAG = "ExoPlayer_Video_Subtitle"
        const val VIDEO_CURRENT_POSITION_TAG = "ExoPlayer_Current_Position_Subtitle"
        const val VIDEO_PLAY_WHEN_READY_TAG = "ExoPlayer_Play_When_Ready_Subtitle"

        fun getRequestIntent(
            activity: Activity,
            currentPosition: Long,
            playWhenReady: Boolean,
            url: String
        ): Intent {
            return Intent(activity, ExoPlayerActivity::class.java).apply {
                putExtra(
                    VIDEO_CURRENT_POSITION_TAG,
                    currentPosition
                )
                putExtra(
                    VIDEO_PLAY_WHEN_READY_TAG,
                    playWhenReady
                )
                putExtra(VIDEO_URL_TAG, url)
            }
        }

        fun getResponseIntent(
            currentPosition: Long,
            playWhenReady: Boolean,
            url: String?
        ): Intent {
            return Intent().apply {
                putExtra(
                    VIDEO_CURRENT_POSITION_TAG,
                    currentPosition
                )
                putExtra(
                    VIDEO_PLAY_WHEN_READY_TAG,
                    playWhenReady
                )
                putExtra(VIDEO_URL_TAG, url)
            }
        }
    }
}