package ai.engagely.openbot.view.adapters

import ai.engagely.openbot.model.constants.ViewConstants
import ai.engagely.openbot.model.pojos.internal.botsettings.IOptionsStyle
import ai.engagely.openbot.model.pojos.internal.chat.*
import ai.engagely.openbot.model.utils.helpers.ChatDiffUtilsHelper
import ai.engagely.openbot.view.adapters.viewholders.*
import ai.engagely.openbot.view.customviews.*
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.RecyclerView
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

class ChatItemsRecyclerViewAdapter(
    private val viewClickListener: View.OnClickListener,
    private val screenWidth: Int
) :
    RecyclerView.Adapter<RecyclerView.ViewHolder>() {

    private val chatItems: ArrayList<IChatItem> = ArrayList()
    private var botButtonColor: Int? = null
    private var botChatBubbleBg: Int? = null
    private var botChatTextColor: Int? = null
    private var userChatBubbleColor: Int? = null
    private var userChatTextColor: Int? = null
    private var themeColor: Int? = null
    private var linkColor: Int? = null
    private var speechListener: SpeechListener? = null

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
        return when (viewType) {
            VIEW_TYPE_MULTIPLE_OPTIONS -> {
                MultipleOptionsItemHolder(ItemMultipleOptionsView(parent.context))
            }
            VIEW_TYPE_BOT_MESSAGE -> {
                BotMessageItemHolder(ItemBotMessageView(parent.context))
            }
            VIEW_TYPE_USER_MESSAGE -> {
                UserMessageItemHolder(ItemUserMessageView(parent.context))
            }
            VIEW_GREETING_CAROUSAL_IMAGE -> {
                GreetingImageHolder(GreetingImageCarousalView(parent.context))
            }
            VIEW_TYPE_MULTIPLE_OPTIONS_WITH_HEADER -> {
                MultipleOptionsWithHeaderItemHolder(MultipleOptionsWithHeaderView(parent.context))
            }
            VIEW_JOURNEY_IMAGE -> {
                val itemImageView = ItemImageView(parent.context)
                itemImageView.calculateAndUpdateImageHeight(screenWidth)
                ImageItemHolder(itemImageView)
            }
            VIEW_USER_MESSAGES -> {
                MessagesItemHolder(ItemMessagesView(parent.context))
            }
            VIEW_DIALOGUE_CAROUSAL -> {
                val dialogueCarousalView = DialogueCarousalView(parent.context)
                dialogueCarousalView.calculateDynamicSizes(screenWidth)
                DialogueCarousalImagesHolder(dialogueCarousalView)
            }
            VIEW_FORM -> {
                val itemFormView = ItemFormView(parent.context)
                itemFormView.calculateDynamicSizes(screenWidth)
                ChatFormItemHolder(itemFormView)
            }
            VIEW_FILE -> {
                ChatFileItemHolder(ItemFileView(parent.context))
            }
            VIEW_URLS -> {
                ChatUrlsItemHolder(ItemUrlsView(parent.context))
            }
            VIEW_VIDEO -> {
                ChatVideoItemHolder(ItemVideoView(parent.context))
            }
            VIEW_YOUTUBE_VIDEO -> {
                ChatYouTubeVideoItemHolder(ItemYouTubeVideoView(parent.context))
            }
            VIEW_CUSTOM_FORM -> {
                CustomFormHolder(ItemCustomFormView(parent.context))
            }
            VIEW_CUSTOM_FORM_NO_CARD -> {
                CustomFormNoBgHolder(ItemCustomFormNoCardView(parent.context))
            }
            VIEW_FAQ_CONFLICT -> {
                ChatFaqConflictHolder(ItemFaqConflictView(parent.context))
            }
            VIEW_TABULAR_INFO -> {
                TabularInfoHolder(TabularInfoView(parent.context))
            }
            VIEW_TABULAR_INFO_WITH_MAP -> {
                TabularInfoWithMapHolder(TabularInfoWithMapView(parent.context))
            }
            VIEW_ADVANCED_URL -> {
                AdvancedUrlItemHolder(ItemAdvancedUrlView(parent.context))
            }
            VIEW_BOT_LOADING -> {
                ChatLoadMoreHolder(ItemLoadMoreView(parent.context))
            }
            else -> ChatLoadMoreHolder(ItemLoadMoreView(parent.context))
        }
    }

    override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
        if (holder is ChatLoadMoreHolder) {
            holder.setData(botChatBubbleBg ?: ViewConstants.VIEW_DEFAULT_COLOR)
            return
        }

        val item = chatItems[position]
        when (holder) {
            is MultipleOptionsItemHolder -> {
                holder.setData(
                    item as? IMultipleOptionsItem,
                    botButtonColor ?: ViewConstants.VIEW_DEFAULT_COLOR,
                    viewClickListener
                )
            }
            is BotMessageItemHolder -> {
                holder.setData(
                    item as? IMessageItem,
                    botChatBubbleBg ?: ViewConstants.VIEW_DEFAULT_COLOR,
                    botChatTextColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    speechListener
                )
            }
            is UserMessageItemHolder -> {
                holder.setData(
                    item as? IMessageItem,
                    userChatBubbleColor ?: ViewConstants.VIEW_DEFAULT_COLOR,
                    userChatTextColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    viewClickListener
                )
            }
            is GreetingImageHolder -> {
                holder.setData(
                    item as? ICarousalImageItem,
                    themeColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    viewClickListener
                )
            }
            is MultipleOptionsWithHeaderItemHolder -> {
                holder.setData(item as? IMultipleOptionsItem, viewClickListener)
            }
            is ImageItemHolder -> {
                holder.setData(item as? IImageItem)
            }
            is MessagesItemHolder -> {
                holder.setData(
                    item as? IMessagesItem,
                    botChatBubbleBg ?: ViewConstants.VIEW_DEFAULT_COLOR,
                    botChatTextColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    speechListener,
                    viewClickListener
                )
            }
            is DialogueCarousalImagesHolder -> {
                holder.setData(item as? IDialogueCarousalItem, viewClickListener)
            }
            is ChatFormItemHolder -> {
                holder.setData(
                    item as? IChatFormItem,
                    themeColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    viewClickListener
                )
            }
            is ChatFileItemHolder -> {
                holder.setData(
                    item as? IFileItem,
                    themeColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    viewClickListener
                )
            }
            is ChatUrlsItemHolder -> {
                holder.setData(
                    item as? IUrlsItem,
                    botChatBubbleBg ?: ViewConstants.VIEW_DEFAULT_COLOR,
                    linkColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    viewClickListener
                )
            }
            is ChatVideoItemHolder -> {
                holder.setData(item as? IVideoItem, viewClickListener)
            }
            is ChatYouTubeVideoItemHolder -> {
                holder.setData(item as? IYouTubeVideoItem, viewClickListener)
            }
            is CustomFormHolder -> {
                holder.setData(
                    item as? IChatCustomFormItem,
                    themeColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK
                )
            }
            is CustomFormNoBgHolder -> {
                holder.setData(item as? IChatCustomFormItem)
            }
            is ChatFaqConflictHolder -> {
                holder.setData(
                    item as? IChatConflictItem,
                    themeColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    viewClickListener
                )
            }
            is TabularInfoHolder -> {
                holder.setData(
                    item as? ITabularInfoItem,
                    botChatBubbleBg ?: ViewConstants.VIEW_DEFAULT_COLOR,
                    viewClickListener
                )
            }
            is TabularInfoWithMapHolder -> {
                holder.setData(
                    item as? ITabularInfoItem,
                    botChatBubbleBg ?: ViewConstants.VIEW_DEFAULT_COLOR,
                    viewClickListener
                )
            }
            is AdvancedUrlItemHolder -> {
                holder.setData(
                    item as? IAdvancedUrlItem,
                    linkColor ?: ViewConstants.VIEW_DEFAULT_COLOR_DARK,
                    viewClickListener
                )
            }
        }
    }

    override fun getItemViewType(position: Int): Int {
        when (val itemToBeCreated = chatItems[position]) {
            is IBotLoadingItem -> return VIEW_BOT_LOADING
            is IMultipleOptionsItem -> {
                return if (itemToBeCreated.optionsHeader?.isNotBlank() == true
                    && itemToBeCreated.optionsStyle == IOptionsStyle.LIST
                ) {
                    VIEW_TYPE_MULTIPLE_OPTIONS_WITH_HEADER
                } else {
                    VIEW_TYPE_MULTIPLE_OPTIONS
                }
            }
            is IMessageItem -> {
                return if (itemToBeCreated.isBotItem) {
                    VIEW_TYPE_BOT_MESSAGE
                } else {
                    VIEW_TYPE_USER_MESSAGE
                }
            }
            is ICarousalImageItem -> {
                return VIEW_GREETING_CAROUSAL_IMAGE
            }
            is IImageItem -> {
                return VIEW_JOURNEY_IMAGE
            }
            is IMessagesItem -> {
                return VIEW_USER_MESSAGES
            }
            is IDialogueCarousalItem -> {
                return VIEW_DIALOGUE_CAROUSAL
            }
            is IChatFormItem -> {
                return VIEW_FORM
            }
            is IFileItem -> {
                return VIEW_FILE
            }
            is IUrlsItem -> {
                return VIEW_URLS
            }
            is IVideoItem -> {
                return VIEW_VIDEO
            }
            is IYouTubeVideoItem -> {
                return VIEW_YOUTUBE_VIDEO
            }
            is IChatCustomFormItem -> {
                return if (itemToBeCreated.showBubble) {
                    VIEW_CUSTOM_FORM
                } else {
                    VIEW_CUSTOM_FORM_NO_CARD
                }
            }
            is IChatConflictItem -> {
                return VIEW_FAQ_CONFLICT
            }
            is ITabularInfoItem -> {
                return if (itemToBeCreated.isMapDataAvailable()) {
                    VIEW_TABULAR_INFO_WITH_MAP
                } else {
                    VIEW_TABULAR_INFO
                }
            }
            is IAdvancedUrlItem -> {
                return VIEW_ADVANCED_URL
            }
            else -> return super.getItemViewType(position)
        }
    }

    override fun getItemCount(): Int {
        return chatItems.size
    }

    suspend fun updateChatItems(newChatItems: List<IChatItem>) {
        val adapter = this
        withContext(Dispatchers.Default) {
            val result: DiffUtil.DiffResult =
                DiffUtil.calculateDiff(ChatDiffUtilsHelper(chatItems, newChatItems))
            chatItems.clear()
            chatItems.addAll(newChatItems)
            withContext(Dispatchers.Main) {
                result.dispatchUpdatesTo(adapter)
            }
        }
    }

    fun updateBothChatConfig(
        botChatButtonColor: Int?,
        botChatBubbleColor: Int?,
        botChatTextColor: Int?,
        userChatBubbleColor: Int?,
        userChatTextColor: Int?,
        themeColor: Int?,
        linkColor: Int?,
        speechListener: SpeechListener?
    ) {
        this.botButtonColor = botChatButtonColor
        this.botChatBubbleBg = botChatBubbleColor
        this.botChatTextColor = botChatTextColor
        this.userChatBubbleColor = userChatBubbleColor
        this.userChatTextColor = userChatTextColor
        this.themeColor = themeColor
        this.linkColor = linkColor
        this.speechListener = speechListener
        //These colors may be used by one or mor items displayed currently.
        // So we have to update all the visible items.
        notifyDataSetChanged()
    }

    interface SpeechListener {
        fun onSpeechRequested(chatItem: IChatItem)
    }

    companion object {
        const val VIEW_BOT_LOADING = 1
        const val VIEW_TYPE_MULTIPLE_OPTIONS = 2
        const val VIEW_TYPE_BOT_MESSAGE = 3
        const val VIEW_TYPE_USER_MESSAGE = 4
        const val VIEW_GREETING_CAROUSAL_IMAGE = 5
        const val VIEW_TYPE_MULTIPLE_OPTIONS_WITH_HEADER = 6
        const val VIEW_JOURNEY_IMAGE = 8
        const val VIEW_USER_MESSAGES = 9
        const val VIEW_DIALOGUE_CAROUSAL = 10
        const val VIEW_FORM = 11
        const val VIEW_FILE = 12
        const val VIEW_URLS = 13
        const val VIEW_VIDEO = 14
        const val VIEW_YOUTUBE_VIDEO = 15
        const val VIEW_CUSTOM_FORM = 16
        const val VIEW_CUSTOM_FORM_NO_CARD = 17
        const val VIEW_FAQ_CONFLICT = 18
        const val VIEW_TABULAR_INFO = 19
        const val VIEW_TABULAR_INFO_WITH_MAP = 20
        const val VIEW_ADVANCED_URL = 21
    }
}