package ai.engagely.openbot.model.repositories.impl

import ai.engagely.openbot.model.network.ApiConstants
import ai.engagely.openbot.model.pojos.external.apirequests.chat.faq.FaqRequest
import ai.engagely.openbot.model.pojos.external.apirequests.chat.flow.FlowRequest
import ai.engagely.openbot.model.pojos.external.apirequests.chat.message.ChatRequest
import ai.engagely.openbot.model.pojos.external.apirequests.chat.tabularinfo.ResponseParams
import ai.engagely.openbot.model.pojos.external.apirequests.chat.tabularinfo.SelectedDimensions
import ai.engagely.openbot.model.pojos.external.apirequests.chat.tabularinfo.TabularInfoChildDropDownRequest
import ai.engagely.openbot.model.pojos.internal.chat.ITabularInfoResponseParams
import ai.engagely.openbot.model.repositories.ChatRepository
import ai.engagely.openbot.model.utils.general.DateUtils
import ai.engagely.openbot.model.utils.general.LogUtils
import com.google.gson.Gson
import io.socket.client.Ack
import io.socket.client.Socket
import kotlinx.coroutines.*
import kotlin.coroutines.resume

class AppChatRepository(
    private val dispatcher: CoroutineDispatcher,
    private val socket: Socket,
) : ChatRepository {

    override suspend fun sendMessage(
        channelName: String,
        userInput: String,
        fieldText: String?,
        isFromProcessTree: Boolean?,
        queryType: String?,
        botId: String?,
        intId: String?,
        intName: String?,
        sessionId: String?,
        sessionObj: Any?,
        userId: String?,
        prevLangCode: String?,
        userLang: String?
    ): Boolean {
        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val chatRequest = Gson().toJson(
                        ChatRequest(
                            botId = botId,
                            channel = ApiConstants.REQUEST_CHANNEL,
                            inputDateNtime = DateUtils.getCurrentTimeInISO8601(),
                            intId = intId,
                            intName = intName,
                            pageUrl = "",
                            queryType = queryType,
                            resBy = "user",
                            resTime = DateUtils.getCurrentTime(),
                            responseType = "text",
                            sessionId = sessionId,
                            sessionObj = sessionObj,
                            userId = userId,
                            userInput = userInput,
                            fieldText = fieldText,
                            fromProcesstree = isFromProcessTree,
                            prevLangCode = prevLangCode,
                            userLang = userLang
                        )
                    )

                    emitMessage(channelName, chatRequest, cnt)
                }
            } ?: false
        }
    }

    override suspend fun sendFlow(
        dynamicSelectedField: String?,
        inputId: String?,
        intId: String?,
        journeyId: String?,
        languageCode: String?,
        messageType: String?,
        msgId: String?,
        nodeId: Int?,
        queryType: String?,
        selectedField: String?,
        sessionId: String?,
        sessionObj: Any?,
        userInput: String?
    ): Boolean {
        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val flowRequest = Gson().toJson(
                        FlowRequest(
                            dynamicSelectedField = dynamicSelectedField,
                            inputId = inputId,
                            intId = intId,
                            journeyId = journeyId,
                            languageCode = languageCode,
                            messageType = messageType,
                            msgId = msgId,
                            nodeId = nodeId,
                            queryType = queryType,
                            selectedField = selectedField,
                            sessionId = sessionId,
                            sessionObj = sessionObj,
                            userInput = userInput
                        )
                    )
                    emitMessage(ApiConstants.CHANNEL_FLOW, flowRequest, cnt)
                }
            } ?: false
        }
    }

    override suspend fun sendFaq(
        botId: String?,
        channel: String?,
        faqId: String?,
        inputDateNTime: String?,
        intId: String?,
        intName: String?,
        journeyID: String?,
        pageUrl: String?,
        prevLangCode: String?,
        question: String?,
        resBy: String?,
        resTime: String?,
        sessionId: String?,
        sessionObj: Any?,
        userId: String?,
        userLang: String?
    ): Boolean {
        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val faqRequest = Gson().toJson(
                        FaqRequest(
                            botId = botId,
                            channel = channel,
                            fAQID = faqId,
                            inputDateNtime = inputDateNTime,
                            intId = intId,
                            intName = intName,
                            journeyID = journeyID,
                            pageUrl = pageUrl,
                            prevLangCode = prevLangCode,
                            question = question,
                            resBy = resBy,
                            resTime = resTime,
                            sessionId = sessionId,
                            sessionObj = sessionObj,
                            userId = userId,
                            userLang = userLang
                        )
                    )
                    emitMessage(ApiConstants.CHANNEL_FAQ_CHAT, faqRequest, cnt)
                }
            } ?: false
        }
    }

    override suspend fun sendTabularChildDropDown(
        headings: List<String>?,
        inputId: String?,
        intId: String?,
        isMapEnabled: Boolean?,
        journeyId: String?,
        languageCode: String?,
        messageType: String?,
        msgId: String?,
        nextDimension: String?,
        responseId: String?,
        responseParams: ITabularInfoResponseParams?,
        selectedEnVal: Map<String, String>?,
        selectedTransVal: Map<String, String>?,
        sessionId: String?,
        sessionObj: Any?,
        userInput: String?,
    ): Boolean {
        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val responseParamsToBeSent = ResponseParams(
                        search = responseParams?.search,
                        searchLocation = responseParams?.searchLocation,
                        select = responseParams?.select,
                        submit = responseParams?.submit
                    )
                    val selectedDimensions = SelectedDimensions(
                        enVal = selectedEnVal,
                        transVal = selectedTransVal
                    )
                    val tabularInfoChildDropDownRequest = Gson().toJson(
                        TabularInfoChildDropDownRequest(
                            headings = headings,
                            inputId = inputId,
                            intId = intId,
                            isMapEnabled = isMapEnabled,
                            journeyId = journeyId,
                            languageCode = languageCode,
                            messageType = messageType,
                            msgId = msgId,
                            nextDimension = nextDimension,
                            responseId = responseId,
                            responseParams = responseParamsToBeSent,
                            selectedDimensions = selectedDimensions,
                            sessionId = sessionId,
                            sessionObj = sessionObj,
                            userInput = userInput,
                        )
                    )
                    emitMessage(ApiConstants.CHANNEL_TAB_INFO, tabularInfoChildDropDownRequest, cnt)
                }
            } ?: false
        }
    }

    private fun emitMessage(
        channelToSendTo: String,
        flowRequest: String?,
        cnt: CancellableContinuation<Boolean?>
    ) {
        if (socket.connected()) {
            LogUtils.log("Emitting on channel = ${channelToSendTo}, message = $flowRequest")
            socket.emit(channelToSendTo, flowRequest, object : Ack {
                override fun call(vararg args: Any?) {
                    LogUtils.log("Callback from emitting $channelToSendTo $args")
                    cnt.resume(true)
                }
            })
        } else {
            cnt.resume(false)
        }
    }
}