package ai.engagely.openbot.model.repositories.impl

import ai.engagely.openbot.model.localstorage.SimpleKeyValueStorage
import ai.engagely.openbot.model.localstorage.SimpleStorageConstants
import ai.engagely.openbot.model.network.ApiClient
import ai.engagely.openbot.model.network.ApiConstants
import ai.engagely.openbot.model.network.interfaces.SessionApi
import ai.engagely.openbot.model.pojos.external.apirequests.session.GetSessionRequest
import ai.engagely.openbot.model.pojos.external.apirequests.session.Location
import ai.engagely.openbot.model.pojos.external.apirequests.session.QueryParams
import ai.engagely.openbot.model.pojos.external.apiresponses.session.GetSessionResponse
import ai.engagely.openbot.model.repositories.SessionRepository
import ai.engagely.openbot.model.utils.general.IdUtils
import ai.engagely.openbot.model.utils.general.LogUtils
import com.google.gson.Gson
import com.google.gson.internal.LinkedTreeMap
import com.google.gson.reflect.TypeToken
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import java.lang.reflect.Type
import java.net.HttpURLConnection

class AppSessionRepository(
    private val dispatcherForDiskIo: CoroutineDispatcher,
    private val dispatcherForNetworkIo: CoroutineDispatcher,
    private val simpleKeyValueStorage: SimpleKeyValueStorage
) : SessionRepository {

    override suspend fun ensureSession(botId: String) {
        withContext(dispatcherForDiskIo) {
            try {
                val prevSessionId =
                    simpleKeyValueStorage.readStringDataForKey(SimpleStorageConstants.KEY_SESSION_ID)
                if (prevSessionId.isNullOrBlank()) {
                    val sessionData = getSession(botId)
                    simpleKeyValueStorage.storeStringDataForKey(
                        SimpleStorageConstants.KEY_SESSION_ID,
                        sessionData?.sessionId
                    )
                    sessionData?.sessionObj?.let {
                        updateSessionObj(it)
                    }

                    if (sessionData?.sessionObj is LinkedTreeMap<*, *>) {
                        if (sessionData.sessionObj.containsKey(ApiConstants.SESSION_OBJ_USER_ID)
                            && sessionData.sessionObj[ApiConstants.SESSION_OBJ_USER_ID] is String
                        ) {
                            simpleKeyValueStorage.storeStringDataForKey(
                                SimpleStorageConstants.KEY_USER_ID,
                                sessionData.sessionObj[ApiConstants.SESSION_OBJ_USER_ID] as String
                            )
                        }
                    }
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
        }
    }

    override suspend fun updateSessionObj(sessionObj: Any) {
        simpleKeyValueStorage.storeStringDataForKey(
            SimpleStorageConstants.KEY_SESSION_OBJ,
            Gson().toJson(sessionObj)
        )
    }

    override suspend fun storeSocketId(id: String?) {
        simpleKeyValueStorage.storeStringDataForKey(SimpleStorageConstants.KEY_SOCKET_ID, id)
    }

    override suspend fun storeLiveChatSocketId(id: String?) {
        simpleKeyValueStorage.storeStringDataForKey(
            SimpleStorageConstants.KEY_LIVE_CHAT_SOCKET_ID,
            id
        )
    }

    override suspend fun getSocketId(): String? {
        return simpleKeyValueStorage.readStringDataForKey(SimpleStorageConstants.KEY_SOCKET_ID)
    }

    override suspend fun getLiveChatSocketId(): String? {
        return simpleKeyValueStorage.readStringDataForKey(SimpleStorageConstants.KEY_LIVE_CHAT_SOCKET_ID)
    }

    override suspend fun clearSession() {
        withContext(dispatcherForDiskIo) {
            simpleKeyValueStorage.storeStringDataForKey(
                SimpleStorageConstants.KEY_SESSION_ID,
                null
            )
        }
    }

    override suspend fun getUserId(): String? {
        return simpleKeyValueStorage.readStringDataForKey(SimpleStorageConstants.KEY_USER_ID)
    }

    override suspend fun getSessionId(): String? {
        return simpleKeyValueStorage.readStringDataForKey(SimpleStorageConstants.KEY_SESSION_ID)
    }

    override suspend fun getSessionObj(): Any? {
        simpleKeyValueStorage.readStringDataForKey(SimpleStorageConstants.KEY_SESSION_OBJ)?.let {
            val mapType: Type = object : TypeToken<Map<String?, Any?>?>() {}.type
            return Gson().fromJson<LinkedTreeMap<*, *>?>(it, mapType)
        } ?: run {
            return null
        }
    }

    private suspend fun getSession(botId: String): GetSessionResponse? {
        return withContext(dispatcherForNetworkIo) {
            val getSessionApi = ApiClient.getInstance().getClient().create(SessionApi::class.java)
            val getSessionRequest = GetSessionRequest(
                intId = botId,
                userId = getUserId() ?: generateUserId(),
                location = Location(latitude = 0.0, longitude = 0.0),
                queryParams = QueryParams(botId = botId)
            )
            val response = getSessionApi.getSession(getSessionRequest)
            if (response.isSuccessful && response.code() == HttpURLConnection.HTTP_OK
                && response.body()?.success == true
            ) {
                return@withContext response.body()
            }
            return@withContext null
        }
    }

    private fun generateUserId() = IdUtils.generateUserId()
}