package ai.engagely.openbot.model.utils.validators

import ai.engagely.openbot.R
import ai.engagely.openbot.model.constants.ServerConstants
import ai.engagely.openbot.model.pojos.internal.chat.IChatFormItem
import ai.engagely.openbot.model.pojos.internal.history.*
import ai.engagely.openbot.model.utils.exts.isValidEmail
import ai.engagely.openbot.model.utils.general.DateUtils
import android.content.Context

object FormItemValidator {

    fun validate(context: Context, iChatFormItem: IChatFormItem): Boolean {
        var isValid = true
        iChatFormItem.formItemContainers.forEach { formItemsContainer ->
            formItemsContainer.formsItems.forEach { formItem ->
                if (!validateFormItem(formItem, context)) {
                    isValid = false
                }
            }
        }
        return isValid
    }

    fun validateFormItem(iFormItem: IFormItem, context: Context): Boolean {
        var errorMessage: String? = null
        if (iFormItem is IFormTextItem) {
            if (iFormItem.textBoxTypeSelected == ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_EMAIL) {
                if (!iFormItem.value.isValidEmail()) {
                    errorMessage = context.getString(R.string.invalid_email)
                }
            }
            val valueLength = iFormItem.value?.trim()?.length ?: 0
            if (iFormItem.isRequired && valueLength == 0) {
                errorMessage = context.getString(R.string.required_field)
            } else if (iFormItem.minLength > 0 && valueLength < iFormItem.minLength) {
                errorMessage = context.resources.getQuantityString(
                    R.plurals.text_field_min_validation,
                    iFormItem.minLength, iFormItem.minLength
                )
            } else if (iFormItem.maxLength > iFormItem.minLength && valueLength > iFormItem.maxLength) {
                errorMessage = context.resources.getQuantityString(
                    R.plurals.text_field_max_validation,
                    iFormItem.maxLength, iFormItem.maxLength
                )
            }
        } else if (iFormItem is IFormTextAreaItem) {
            if (iFormItem.isRequired) {
                val valueLength = iFormItem.value?.trim()?.length ?: 0
                if (valueLength == 0) {
                    errorMessage = context.getString(R.string.required_field)
                }
            }
        } else if (iFormItem is IFormRadioButtonsItem) {
            if (iFormItem.isRequired) {
                val anyItemSelected = iFormItem.options.find { it.isChecked } != null
                if (!anyItemSelected) {
                    errorMessage = context.getString(R.string.required_field)
                }
            }
        } else if (iFormItem is IFormCheckboxesItem) {
            if (iFormItem.isRequired) {
                val anyItemSelected = iFormItem.options.find { it.isChecked } != null
                if (!anyItemSelected) {
                    errorMessage = context.getString(R.string.required_field)
                }
            }
        } else if (iFormItem is IFormDateItem) {
            if (iFormItem.isRequired && iFormItem.selectedDate == null) {
                errorMessage = context.getString(R.string.required_field)
            } else if (!iFormItem.dateType.isNullOrBlank()) {
                if (iFormItem.dateType == ServerConstants.DIALOGUE_FORM_DATE_TYPE_TODAY && !DateUtils.isTodayDate(
                        iFormItem.selectedDate
                    )
                ) {
                    errorMessage = context.getString(R.string.today_date_must_be_selected)
                } else if (iFormItem.dateType == ServerConstants.DIALOGUE_FORM_DATE_TYPE_FUTURE && !DateUtils.isFutureOrToday(
                        iFormItem.selectedDate
                    )
                ) {
                    errorMessage = context.getString(R.string.future_date_must_be_selected)
                } else if (iFormItem.dateType == ServerConstants.DIALOGUE_FORM_DATE_TYPE_PAST && !DateUtils.isPastOrToday(
                        iFormItem.selectedDate
                    )
                ) {
                    errorMessage = context.getString(R.string.past_date_must_be_selected)
                }
            }
        } else if (iFormItem is IFormFileItem) {
            if (iFormItem.isRequired && iFormItem.file == null) {
                errorMessage = context.getString(R.string.required_field)
            }
        }
        iFormItem.errorMessage = errorMessage
        return errorMessage.isNullOrBlank()
    }
}