package ai.engagely.openbot.viewmodel

import ai.engagely.openbot.model.constants.AppConstants
import ai.engagely.openbot.model.localstorage.impl.PreferenceDataStore
import ai.engagely.openbot.model.pojos.internal.faqautocomplete.IFaqAutoCompleteItem
import ai.engagely.openbot.model.repositories.FaqAutoCompleteRepository
import ai.engagely.openbot.model.repositories.SessionRepository
import ai.engagely.openbot.model.repositories.impl.AppFaqAutoCompleteRepository
import ai.engagely.openbot.model.repositories.impl.AppSessionRepository
import android.app.Application
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

class FaqAutoCompleteViewModel(application: Application) : NetworkStateAwareViewModel(application) {

    private val faqAutoCompleteRepository: FaqAutoCompleteRepository by lazy {
        AppFaqAutoCompleteRepository(Dispatchers.IO)
    }

    private val sessionRepository: SessionRepository by lazy {
        AppSessionRepository(Dispatchers.Default, Dispatchers.IO, PreferenceDataStore(application))
    }

    private val _faqAutoCompleteItemsLiveData = MutableLiveData<List<IFaqAutoCompleteItem>>()
    val faqAutoCompleteItemsLiveData: LiveData<List<IFaqAutoCompleteItem>> =
        _faqAutoCompleteItemsLiveData

    private val _isLoadingLiveData = MutableLiveData<Boolean>()
    val isLoadingLiveData: LiveData<Boolean> = _isLoadingLiveData

    private var intId: String? = null
    private var sessionId: String? = null

    fun init(intId: String?) {
        this.intId = intId
        viewModelScope.launch {
            sessionId = sessionRepository.getSessionId()
        }
    }

    fun processAutoComplete(text: String) {
        if (text.length >= AppConstants.FAQ_AUTO_COMPLETE_MIN_CHARACTER_COUNT
            && text.endsWith(" ") && text.isNotBlank()
        ) {
            viewModelScope.launch {
                try {
                    _isLoadingLiveData.postValue(true)
                    _faqAutoCompleteItemsLiveData.postValue(
                        faqAutoCompleteRepository.autoComplete(
                            text,
                            intId,
                            sessionId
                        )
                    )
                    _isLoadingLiveData.postValue(false)
                } catch (e: Exception) {
                    _isLoadingLiveData.postValue(false)
                }
            }
        } else if (text.length < AppConstants.FAQ_AUTO_COMPLETE_MIN_CHARACTER_COUNT) {
            _faqAutoCompleteItemsLiveData.postValue(null)
        }
    }
}