package ai.engagely.openbot.model.utils.helpers

import ai.engagely.openbot.model.constants.AppConstants
import ai.engagely.openbot.model.utils.general.LogUtils
import android.content.Context
import android.speech.tts.TextToSpeech
import android.speech.tts.UtteranceProgressListener
import androidx.core.text.parseAsHtml
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import java.util.*

class TextToSpeechHelper(private val onSpeechCompletedListener: OnSpeechCompletedListener) :
    DefaultLifecycleObserver, TextToSpeech.OnInitListener {

    private lateinit var textToSpeechEngine: TextToSpeech

    fun init(context: Context) {
        textToSpeechEngine = TextToSpeech(context, this)
    }

    override fun onPause(owner: LifecycleOwner) {
        if (!::textToSpeechEngine.isInitialized) return

        textToSpeechEngine.stop()

        super.onPause(owner)
    }

    override fun onDestroy(owner: LifecycleOwner) {
        if (!::textToSpeechEngine.isInitialized) return

        textToSpeechEngine.shutdown()

        super.onDestroy(owner)
    }

    fun startSpeech(message: String, isHtml: Boolean, flushExisting: Boolean, utteranceId: String) {
        try {
            if (flushExisting) stopSpeaking()

            if (message.isEmpty()) return

            speak(
                if (isHtml) try {
                    message.parseAsHtml().trimEnd().toString()
                } catch (e: Exception) {
                    LogUtils.logException(e)
                    message
                } else message,
                flushExisting,
                utteranceId
            )
            speakSilence(AppConstants.TEXT_TO_SPEECH_DELAY_BETWEEN_SPEECHES, false)
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
    }

    private fun speak(message: String?, flushExisting: Boolean, utteranceId: String) {
        if (!::textToSpeechEngine.isInitialized) return

        message?.let {
            if (it.isBlank()) return
            LogUtils.log("Speaking $it")

            textToSpeechEngine.speak(
                it.trim(),
                if (flushExisting) TextToSpeech.QUEUE_FLUSH else TextToSpeech.QUEUE_ADD,
                null,
                utteranceId
            )
        }
    }

    private fun speakSilence(durationInMillis: Long, flushExisting: Boolean) {
        if (!::textToSpeechEngine.isInitialized) return

        textToSpeechEngine.playSilentUtterance(
            durationInMillis,
            if (flushExisting) TextToSpeech.QUEUE_FLUSH else TextToSpeech.QUEUE_ADD,
            Calendar.getInstance().timeInMillis.toString()
        )
    }

    fun stopSpeaking() {
        if (!::textToSpeechEngine.isInitialized) return

        if (textToSpeechEngine.isSpeaking) {
            textToSpeechEngine.stop()
        }
    }

    override fun onInit(status: Int) {
        if (status == TextToSpeech.SUCCESS) {
            val result = textToSpeechEngine.setLanguage(Locale.UK)
            if (result == TextToSpeech.LANG_MISSING_DATA || result == TextToSpeech.LANG_NOT_SUPPORTED) {
                return
            }
            textToSpeechEngine.setOnUtteranceProgressListener(object : UtteranceProgressListener() {
                override fun onStart(utteranceId: String?) {
                    onSpeechCompletedListener.onSpeechStarted(utteranceId)
                }

                override fun onDone(utteranceId: String?) {
                    onSpeechCompletedListener.onSpeechCompleted(utteranceId)
                }

                override fun onError(utteranceId: String?) {
                }

                override fun onError(utteranceId: String?, errorCode: Int) {
                    super.onError(utteranceId, errorCode)
                }
            })
        }
    }

    interface OnSpeechCompletedListener {
        fun onSpeechCompleted(utteranceId: String?)

        fun onSpeechStarted(utteranceId: String?)
    }
}