package ai.engagely.openbot.viewmodel

import ai.engagely.openbot.model.utils.general.LogUtils
import ai.engagely.openbot.model.utils.general.NetworkUtils
import android.app.Application
import android.content.Context.CONNECTIVITY_SERVICE
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.net.NetworkRequest
import android.os.Build
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData

open class NetworkStateAwareViewModel(application: Application) : AndroidViewModel(application) {

    private val _networkStateLiveData = MutableLiveData<NetworkState?>()
    val networkStateLiveData: LiveData<NetworkState?> = _networkStateLiveData
    private var networkStateChangeListener: NetworkStateChangeListener? = null

    private var currentNetworkState: NetworkState = NetworkState.UNKNOWN

    private val networkCallback = object : ConnectivityManager.NetworkCallback() {
        override fun onAvailable(network: Network) {
            super.onAvailable(network)
            notifyNetworkStateChanged(NetworkState.CONNECTED)
        }

        override fun onCapabilitiesChanged(
            network: Network,
            networkCapabilities: NetworkCapabilities
        ) {
            super.onCapabilitiesChanged(network, networkCapabilities)
            val hasWifi = networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
            val hasCellular =
                networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
            val hasEthernet =
                networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)
            notifyNetworkStateChanged(
                if (hasCellular || hasWifi || hasEthernet)
                    NetworkState.CONNECTED
                else
                    NetworkState.DISCONNECTED
            )
        }

        override fun onLost(network: Network) {
            super.onLost(network)
            notifyNetworkStateChanged(NetworkState.DISCONNECTED)
        }
    }

    private fun notifyNetworkStateChanged(networkState: NetworkState) {
        if (networkState == currentNetworkState) {
            return
        }
        currentNetworkState = networkState
        _networkStateLiveData.postValue(networkState)
        networkStateChangeListener?.onNetworkStateChanged(networkState)
    }

    init {
        try {
            val networkRequestBuilder = NetworkRequest.Builder()
                .addCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
                .addTransportType(NetworkCapabilities.TRANSPORT_WIFI)
                .addTransportType(NetworkCapabilities.TRANSPORT_CELLULAR)
                .addTransportType(NetworkCapabilities.TRANSPORT_ETHERNET)

            getConnectivityManager(application)?.requestNetwork(
                networkRequestBuilder.build(),
                networkCallback
            )
        } catch (e: Exception) {
            LogUtils.logException(e)
        }

        val isConnectedToInternet = NetworkUtils.isInternetAvailable(application)
        notifyNetworkStateChanged(
            if (isConnectedToInternet) NetworkState.CONNECTED
            else NetworkState.DISCONNECTED
        )
    }

    private fun getConnectivityManager(application: Application): ConnectivityManager? {
        val connectivityManager = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            application.getSystemService(ConnectivityManager::class.java)
        } else {
            application.getSystemService(CONNECTIVITY_SERVICE) as ConnectivityManager
        }
        return connectivityManager
    }

    enum class NetworkState {
        CONNECTED, DISCONNECTED, UNKNOWN
    }

    override fun onCleared() {
        super.onCleared()
        try {
            getConnectivityManager(getApplication())?.unregisterNetworkCallback(networkCallback)
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
    }

    fun setNetworkStateChangeListener(networkStateChangeListener: NetworkStateChangeListener) {
        this.networkStateChangeListener = networkStateChangeListener
    }

    interface NetworkStateChangeListener {
        fun onNetworkStateChanged(networkState: NetworkState)
    }
}