package ai.engagely.openbot.viewmodel

import ai.engagely.openbot.R
import ai.engagely.openbot.model.constants.AppConstants
import ai.engagely.openbot.model.network.VideoCache
import ai.engagely.openbot.model.utils.exts.showToast
import ai.engagely.openbot.model.utils.general.LogUtils
import android.app.Application
import android.app.DownloadManager
import android.app.DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED
import android.content.Context
import android.database.ContentObserver
import android.media.AudioManager
import android.net.Uri
import android.os.Build
import android.os.Environment
import android.os.Handler
import android.os.Looper
import androidx.lifecycle.*
import com.google.android.exoplayer2.ExoPlayer
import com.google.android.exoplayer2.MediaItem
import com.google.android.exoplayer2.Player
import com.google.android.exoplayer2.source.ProgressiveMediaSource
import com.google.android.exoplayer2.upstream.DataSource
import com.google.android.exoplayer2.upstream.DefaultHttpDataSource
import com.google.android.exoplayer2.upstream.HttpDataSource
import com.google.android.exoplayer2.upstream.cache.CacheDataSource
import com.google.android.exoplayer2.video.VideoSize

class VideoPlayerViewModel(application: Application) : AndroidViewModel(application),
    DefaultLifecycleObserver {

    var player: ExoPlayer? = null
        private set

    private var contentPosition = 0L
    private var playWhenReady = true

    private var url: String? = null

    var videoWidth: Int = 0
        private set
    var videoHeight: Int = 0
        private set

    private val httpDataSourceFactory: HttpDataSource.Factory =
        DefaultHttpDataSource.Factory().setAllowCrossProtocolRedirects(true)
    private val cacheDataSourceFactory: DataSource.Factory = CacheDataSource.Factory()
        .setCache(VideoCache.getInstance(application).getSimpleCache())
        .setUpstreamDataSourceFactory(httpDataSourceFactory)

    private val _playerLiveData = MutableLiveData<Player>()
    val playerLiveData: LiveData<Player> = _playerLiveData

    private val _volumeMutedLiveData = MutableLiveData<Boolean?>()
    val volumeMutedLiveData: LiveData<Boolean?> = _volumeMutedLiveData

    private var lastVolume: Int? = null

    private val _playbackSpeedLiveData = MutableLiveData<Float>()
    val playbackSpeedLiveData: LiveData<Float> = _playbackSpeedLiveData
    private var playbackSpeed: Float = 1f

    private val settingsContentObserver =
        object : ContentObserver(Handler(Looper.getMainLooper())) {
            override fun onChange(selfChange: Boolean) {
                super.onChange(selfChange)
                syncWithSystemMediaVolumeLevel()
            }
        }

    private val _pictureInPictureModeLiveData = MutableLiveData<Boolean>()
    val pictureInPictureModeLiveData: LiveData<Boolean> = _pictureInPictureModeLiveData

    private val _isPlayingLiveData = MutableLiveData<Boolean>()
    val isPlayingLiveData: LiveData<Boolean> = _isPlayingLiveData

    private val playbackListener = object : Player.Listener {
        override fun onPlaybackStateChanged(playbackState: Int) {
            super.onPlaybackStateChanged(playbackState)
            if (playbackState == Player.STATE_ENDED) {
                contentPosition = 0
                playWhenReady = false
            }
        }

        override fun onIsPlayingChanged(isPlaying: Boolean) {
            super.onIsPlayingChanged(isPlaying)
            _isPlayingLiveData.postValue(isPlaying)
        }

        override fun onVideoSizeChanged(videoSize: VideoSize) {
            super.onVideoSizeChanged(videoSize)
            videoWidth = videoSize.width
            videoHeight = videoSize.height
        }
    }

    init {
        getApplication<Application>().applicationContext.contentResolver.registerContentObserver(
            android.provider.Settings.System.CONTENT_URI, true,
            settingsContentObserver
        )
    }

    fun updateInitialData(
        contentPosition: Long,
        playWhenReady: Boolean,
        speed: Float,
        url: String
    ) {
        this.contentPosition = contentPosition
        this.playWhenReady = playWhenReady
        this.url = url
        this.playbackSpeed = speed
    }

    private fun initializePlayer() {
        ExoPlayer.Builder(getApplication())
            .build()
            .also { exoPlayer ->
                player = exoPlayer
                player?.addListener(playbackListener)
                _playerLiveData.postValue(exoPlayer)
                setPlaybackSpeed(playbackSpeed)
                syncWithSystemMediaVolumeLevel()
                setMediaItemAndPrepare(exoPlayer)
            }
    }

    private fun setMediaItemAndPrepare(exoPlayer: ExoPlayer) {
        url?.let {
            val mediaItem = MediaItem.fromUri(it)
            val mediaSource =
                ProgressiveMediaSource.Factory(cacheDataSourceFactory).createMediaSource(mediaItem)
            exoPlayer.setMediaSource(mediaSource)

            exoPlayer.playWhenReady = playWhenReady
            exoPlayer.seekTo(0, contentPosition)
            exoPlayer.prepare()
        }
    }

    override fun onStart(owner: LifecycleOwner) {
        super.onStart(owner)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            initializePlayer()
        }
    }

    override fun onResume(owner: LifecycleOwner) {
        super.onResume(owner)
        if ((Build.VERSION.SDK_INT < Build.VERSION_CODES.N || player == null)) {
            initializePlayer()
        }
    }

    override fun onPause(owner: LifecycleOwner) {
        super.onPause(owner)
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.N) {
            releasePlayer()
        }
    }

    override fun onStop(owner: LifecycleOwner) {
        super.onStop(owner)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            releasePlayer()
        }
    }

    private fun releasePlayer() {
        player ?: return

        contentPosition = if (isPlayingFinished()) {
            0
        } else {
            player?.contentPosition ?: 0
        }
        playWhenReady = player?.playWhenReady ?: true
        player?.removeListener(playbackListener)
        player?.release()
        player = null
        _playerLiveData.postValue(null)
    }

    fun startPlaying(url: String?) {
        url ?: return

        this.url = url
        player?.let {
            setMediaItemAndPrepare(it)
        }
    }

    fun stopPlaying() {
        player?.pause()
    }

    fun getCurrentPosition(): Long {
        if (isPlayingFinished()) {
            contentPosition = 0
            return 0
        }
        val currentPosition = player?.contentPosition ?: 0
        return if (currentPosition > AppConstants.VIDEO_SEEK_BACK_DURATION_FOR_NEXT_SCREEN_IN_MILLIS) {
            currentPosition - AppConstants.VIDEO_SEEK_BACK_DURATION_FOR_NEXT_SCREEN_IN_MILLIS
        } else {
            0
        }
    }

    fun getSavedCurrentPosition(): Long = contentPosition

    private fun isPlayingFinished(): Boolean {
        val currentPosition = player?.contentPosition ?: 0
        val duration = player?.contentDuration ?: 0
        return if (currentPosition >= duration) {
            true
        } else currentPosition > (duration - AppConstants.VIDEO_END_POSITION_CORRECTION_IN_MILLIS) //100 milliseconds is the end buffer
    }

    fun getUrl(): String? = url

    fun getPlayWhenReady(): Boolean = player?.isPlaying ?: false

    fun getPlaybackSpeed(): Float = playbackSpeed

    fun toggleVolume() {
        try {
            val audioManager =
                getApplication<Application>().getSystemService(Context.AUDIO_SERVICE) as AudioManager
            val currentVolume = audioManager.getStreamVolume(AudioManager.STREAM_MUSIC)
            if (currentVolume > 0) {
                audioManager.setStreamVolume(AudioManager.STREAM_MUSIC, 0, 0)
            } else {
                val newVolume: Int = lastVolume
                    ?: (audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC) * 0.8).toInt()
                audioManager.setStreamVolume(AudioManager.STREAM_MUSIC, newVolume, 0)
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
        syncWithSystemMediaVolumeLevel()
    }

    private fun syncWithSystemMediaVolumeLevel() {
        try {
            val audioManager =
                getApplication<Application>().getSystemService(Context.AUDIO_SERVICE) as AudioManager
            val currentVolume = audioManager.getStreamVolume(AudioManager.STREAM_MUSIC)
            _volumeMutedLiveData.postValue(currentVolume == 0)
            if (currentVolume > 0) {
                lastVolume = currentVolume
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
    }

    fun setPlaybackSpeed(speed: Float) {
        if (speed < 0 || speed > 2) {
            return
        }
        player?.let {
            it.setPlaybackSpeed(speed)
            playbackSpeed = speed
            _playbackSpeedLiveData.postValue(speed)
        }
    }

    override fun onCleared() {
        super.onCleared()
        releasePlayer()
        getApplication<Application>().applicationContext.contentResolver.unregisterContentObserver(
            settingsContentObserver
        )
    }

    fun downloadVideo() {
        url?.let {
            val fileName = it.substring(it.lastIndexOf('/') + 1)
            val request: DownloadManager.Request = DownloadManager.Request(Uri.parse(it))
                .setDestinationInExternalPublicDir(Environment.DIRECTORY_DOWNLOADS, fileName)
                .setTitle(fileName)
                .setDescription(getApplication<Application>().getString(R.string.downloading))
                .setAllowedOverMetered(true)
                .setAllowedOverRoaming(true)
                .setNotificationVisibility(VISIBILITY_VISIBLE_NOTIFY_COMPLETED)

            (getApplication<Application>().getSystemService(Context.DOWNLOAD_SERVICE) as? DownloadManager)?.enqueue(
                request
            )
            showToast(
                getApplication<Application>().getString(
                    R.string.downloading_file_started,
                    fileName
                )
            )
        }
    }

    fun setPictureInPictureMode(isInPictureInPictureMode: Boolean) {
        _pictureInPictureModeLiveData.postValue(isInPictureInPictureMode)
    }

    fun updateCurrentData(currentPosition: Long, playbackSpeed: Float) {
        player?.let {
            it.seekTo(currentPosition)
            setPlaybackSpeed(playbackSpeed)
        }
    }

    fun pausePlayer() {
        player?.pause()
    }

    fun playPlayer() {
        player?.let {
            if (isPlayingFinished()) {
                it.seekTo(0)
                contentPosition = 0
                playWhenReady = false
            }
            it.play()
        }
    }

    companion object {
        const val CACHE_SIZE: Long = 100 * 1024 * 1024 //100MB
    }
}