package ai.engagely.openbot.model.utils.converters

import ai.engagely.openbot.model.constants.AppConstants
import ai.engagely.openbot.model.constants.ServerConstants
import ai.engagely.openbot.model.network.ApiConstants
import ai.engagely.openbot.model.pojos.external.apiresponses.chat.ChatResponse
import ai.engagely.openbot.model.pojos.external.apiresponses.chat.LiveChatResponse
import ai.engagely.openbot.model.pojos.external.apiresponses.history.*
import ai.engagely.openbot.model.pojos.external.apiresponses.history.Response
import ai.engagely.openbot.model.pojos.external.apiresponses.livechat.*
import ai.engagely.openbot.model.pojos.internal.botsettings.IOptionsStyle
import ai.engagely.openbot.model.pojos.internal.chat.*
import ai.engagely.openbot.model.pojos.internal.history.*
import ai.engagely.openbot.model.pojos.internal.livechat.*
import ai.engagely.openbot.model.utils.exts.obtainNonBlankOrNull
import ai.engagely.openbot.model.utils.general.DateUtils
import ai.engagely.openbot.model.utils.general.LogUtils
import ai.engagely.openbot.model.utils.general.NumberUtils
import ai.engagely.openbot.model.utils.general.ViewUtils
import android.net.Uri
import com.google.gson.Gson
import com.google.gson.internal.LinkedTreeMap
import com.google.gson.reflect.TypeToken
import java.lang.reflect.Type
import java.util.*

class ChatResponseConverter {

    fun convertChatResponseToChatItems(
        chatResponse: ChatResponse?,
        chatResponseString: String?,
        sessionId: String?,
        languageCode: String
    ): List<IChatItem>? {
        val messageDate = Date()
        if (ServerConstants.CHAT_ITEM_TYPE_TEXT == chatResponse?.responseType) {
            return convertTextMessageResponseToChatItem(
                chatResponse = chatResponse,
                chatResponseString = chatResponseString,
                date = messageDate,
                languageCode
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_PROCESS_TREE == chatResponse?.responseType && chatResponse.response is LinkedTreeMap<*, *>) {
            return convertButtonItemsToChatItem(
                response = chatResponse.response,
                date = messageDate,
                resBy = ServerConstants.CHAT_ITEM_RESPONSE_BY_BOT,
                sessionId = sessionId,
                isHistoryItem = false,
                isVoiceQuery = ApiConstants.CHAT_QUERY_TYPE_VOICE == chatResponse.queryType
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_CONFLICTS == chatResponse?.responseType && chatResponse.response is LinkedTreeMap<*, *>) {
            return convertConflictItemToChatItem(
                response = chatResponse.response,
                messageDate = messageDate,
                isHistoryItem = false
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_FAQ == chatResponse?.responseType && chatResponse.response is List<*>) {
            return convertFaqItemToChatItem(
                response = chatResponse.response,
                date = messageDate,
                isHistoryItem = false,
                isVoiceQuery = ApiConstants.CHAT_QUERY_TYPE_VOICE == chatResponse.queryType
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_TABULAR_INFO == chatResponse?.responseType && chatResponse.response is LinkedTreeMap<*, *>) {
            return convertTabularInfoItemToChatItem(
                response = chatResponse.response,
                date = messageDate,
                isHistoryItem = false,
                isVoiceQuery = ApiConstants.CHAT_QUERY_TYPE_VOICE == chatResponse.queryType
            )
        }
        return null
    }

    fun isTabularInfoItems(chatResponse: ChatResponse?): Boolean {
        return ServerConstants.CHAT_ITEM_TYPE_TABULAR_INFO == chatResponse?.responseType && chatResponse.response is LinkedTreeMap<*, *>
    }

    fun convertToTabularItemChildDropDownItems(
        chatResponse: ChatResponse?,
        heading: String
    ): List<ITabularDropDownItem>? {
        if (ServerConstants.CHAT_ITEM_TYPE_TABULAR_INFO == chatResponse?.responseType && chatResponse.response is LinkedTreeMap<*, *>) {
            return getTabularDropDownItemsForKey(heading, chatResponse.response)
        }
        return null
    }

    fun convertChatResponseToJourney(chatResponse: ChatResponse?): List<String> {
        val journeys = ArrayList<String>()
        if (ServerConstants.CHAT_ITEM_TYPE_PROCESS_TREE == chatResponse?.responseType && chatResponse.response is LinkedTreeMap<*, *>) {
            val responseJson = Gson().toJson(chatResponse.response)
            val responseObject = Gson().fromJson(responseJson, Response::class.java)
            responseObject.dialogues?.forEach { dialogue ->
                if (dialogue.isNotEmpty()) {
                    dialogue.forEach { innerDialogue ->
                        parseDialogueJourneyIfPresent(innerDialogue)?.let { journeyName ->
                            journeys.add(journeyName)
                        }
                    }
                }
            }
        } else if (ServerConstants.CHAT_ITEM_TYPE_FAQ == chatResponse?.responseType && chatResponse.response is List<*>) {
            convertToFaqResponse(chatResponse.response)?.forEach { faqResponse ->
                parseFaqJourney(faqResponse)?.let { journeyName ->
                    journeys.add(journeyName)
                }
            }
        }
        return journeys
    }

    fun convertHistoryItemToChatItem(
        chatDataItem: ChatData,
        sessionId: String?,
    ): List<IChatItem>? {
        val date = DateUtils.convertFromUtc3ToLocalDate(chatDataItem.resTime)
        if (ServerConstants.CHAT_ITEM_TYPE_PROCESS_TREE == chatDataItem.responseType && chatDataItem.response is LinkedTreeMap<*, *>) {
            return convertButtonItemsToChatItem(
                response = chatDataItem.response,
                date = date,
                resBy = chatDataItem.resBy,
                sessionId = sessionId,
                isHistoryItem = true,
                isVoiceQuery = false
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_TEXT == chatDataItem.responseType) {
            return convertHistoryMessageToChatItem(
                chatDataItem = chatDataItem,
                date = date
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_CONFLICTS == chatDataItem.responseType && chatDataItem.response is LinkedTreeMap<*, *>) {
            return convertConflictItemToChatItem(
                response = chatDataItem.response,
                messageDate = date,
                isHistoryItem = true
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_FAQ == chatDataItem.responseType && chatDataItem.response is List<*>) {
            return convertFaqItemToChatItem(
                response = chatDataItem.response,
                date = date,
                isHistoryItem = true,
                isVoiceQuery = false
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_TABULAR_INFO == chatDataItem.responseType && chatDataItem.response is LinkedTreeMap<*, *>) {
            return convertTabularInfoItemToChatItem(
                response = chatDataItem.response,
                date = date,
                isHistoryItem = true,
                isVoiceQuery = false
            )
        } else if (ServerConstants.CHAT_ITEM_TYPE_LIVE_CHAT_CONNECTED == chatDataItem.responseType
            || ServerConstants.CHAT_ITEM_TYPE_LIVE_CHAT_DISCONNECTED == chatDataItem.responseType
        ) {
            return ArrayList<IChatItem>().apply {
                convertToLiveChatStatusItem(chatDataItem.response, null)?.let {
                    add(it)
                }
            }.ifEmpty { null }
        } else if (ServerConstants.CHAT_ITEM_TYPE_LIVE_CHAT_MEDIA == chatDataItem.responseType) {
            return ArrayList<IChatItem>().apply {
                val isUserResponse =
                    ServerConstants.LIVE_CHAT_RESPONSE_BY_USER == chatDataItem.resBy
                convertToLiveChatMediaItem(
                    response = if (isUserResponse) chatDataItem.userInput else chatDataItem.response,
                    isUserResponse = isUserResponse,
                    date = date,
                    isHistoryItem = true
                )?.let {
                    add(it)
                }
            }.ifEmpty { null }
        } else if (ServerConstants.CHAT_ITEM_TYPE_CSAT_RESPONSE == chatDataItem.responseType && chatDataItem.response is List<*>) {
            return convertHistoryItemToFeedbackItem(chatDataItem.response)
        }
        return null
    }

    private fun convertHistoryItemToFeedbackItem(response: List<*>): ArrayList<IChatItem>? {
        val feedbackResponseItems = ArrayList<IChatItem>()
        response.forEach { r ->
            val responseItem = Gson().fromJson(Gson().toJson(r), ResponseXX::class.java)
            if (ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_MEDIA == responseItem.responseType
                && ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_CSAT_MEDIA == responseItem.csatResponse
            ) {
                convertCsatMediaToChatItems(responseItem.response, feedbackResponseItems)
            } else if (ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_TEXT == responseItem.responseType
                && ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_CSAT_TEXT == responseItem.csatResponse
            ) {
                (responseItem.response as? String)?.let { response ->
                    feedbackResponseItems.add(
                        IMessageItem(
                            message = response,
                            isHtml = true,
                            date = Date(),
                            hideTime = true,
                            isBotItem = true,
                        )
                    )
                }
            }
        }
        return feedbackResponseItems.ifEmpty { null }
    }

    private fun convertCsatMediaToChatItems(
        response: Any?,
        feedbackResponseItems: ArrayList<IChatItem>
    ) {
        try {
            val mediaResponse =
                Gson().fromJson(
                    Gson().toJson(response),
                    MediaResponse::class.java
                )
            if (mediaResponse.mediaType == ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_MEDIA_IMAGE) {
                feedbackResponseItems.add(
                    IImageItem(
                        imageUrl = mediaResponse?.mediaUrl,
                        date = Date(),
                        isBotItem = true,
                        hideDate = true
                    )
                )
            } else if (mediaResponse.mediaType == ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_MEDIA_VIDEO) {
                mediaResponse?.mediaUrl?.let { url ->
                    feedbackResponseItems.add(
                        IVideoItem(
                            title = mediaResponse.mediaName,
                            description = null,
                            url = url,
                            fileSize = null,
                            date = Date(),
                            isBotItem = true,
                            isHistoryItem = false,
                            hideDetails = true
                        )
                    )
                }
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
    }

    fun convertFeedbackSubmitResponseToChatItems(
        liveChatData: ILiveChatData,
        additionalFeedbackDetails: UserFeedbackResponse
    ): Any? {
        if (additionalFeedbackDetails.endCsat == false
            && additionalFeedbackDetails.type == ServerConstants.LIVE_CHAT_FEEDBACK_TYPE_ADDITION_QUESTION
        ) {
            additionalFeedbackDetails.message?.find { message ->
                ServerConstants.LIVE_CHAT_FEEDBACK_TYPE_ADDITION_QUESTION == message.additionalQuestion
                        && ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_TEXT == message.responseType
            }?.let { additionalQuestion ->
                return ILiveChatFeedbackAdditionalQuestion(
                    additionalQuestion = additionalQuestion.response as? String,
                    isPositive = additionalFeedbackDetails.isPositive,
                    additionalAnswer = null,
                    messages = additionalFeedbackDetails.message,
                    type = additionalFeedbackDetails.type
                )
            }
        } else if (additionalFeedbackDetails.endCsat == true
            && additionalFeedbackDetails.type == ServerConstants.LIVE_CHAT_FEEDBACK_TYPE_CSAT_RESPONSE
        ) {
            val feedbackResponseItems = ArrayList<IChatItem>()
            additionalFeedbackDetails.message?.forEach { message ->
                if (ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_TEXT == message.responseType
                    && ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_CSAT_TEXT == message.csatResponse
                ) {
                    (message.response as? String)?.let { response ->
                        feedbackResponseItems.add(
                            IMessageItem(
                                message = response,
                                isHtml = true,
                                date = Date(),
                                hideTime = true,
                                isBotItem = true,
                            )
                        )
                    }
                } else if (ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_MEDIA == message.responseType
                    && ServerConstants.LIVE_CHAT_FEEDBACK_RESPONSE_TYPE_CSAT_MEDIA == message.csatResponse
                ) {
                    convertCsatMediaToChatItems(message.response, feedbackResponseItems)
                }
            }
            return ILiveChatDataContainer(
                items = feedbackResponseItems,
                data = liveChatData,
                status = ILiveChatDataContainer.ILiveChatStatus.DISCONNECTED,
                feedbackRequest = null
            )
        }
        return null
    }

    private fun convertButtonItemsToChatItem(
        response: LinkedTreeMap<*, *>,
        date: Date?,
        resBy: String?,
        sessionId: String?,
        isHistoryItem: Boolean,
        isVoiceQuery: Boolean,
    ): List<IChatItem>? {
        try {
            val chatItems = ArrayList<IChatItem>()
            val responseJson = Gson().toJson(response)
            val responseObject = Gson().fromJson(responseJson, Response::class.java)

            //Parse dialogue items
            parseDialogueItemsIfPresent(
                responseObject,
                chatItems,
                date,
                sessionId,
                isHistoryItem,
                isVoiceQuery
            )

            //Parse buttons
            parseButtonsIfPresent(responseObject, chatItems, resBy, date, isHistoryItem)
            return chatItems.ifEmpty { null }
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
        return null
    }

    private fun parseDialogueItemsIfPresent(
        responseObject: Response,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        sessionId: String?,
        isHistoryItem: Boolean,
        isVoiceQuery: Boolean,
    ) {
        responseObject.dialogues?.forEach { dialogue ->
            if (dialogue.isNotEmpty()) {
                dialogue.forEach { innerDialogue ->
                    parseDialogueImageIfPresent(innerDialogue, chatItems, date, isHistoryItem)
                    parseDialogueMessagesIfPresent(
                        innerDialogue,
                        chatItems,
                        date,
                        isHistoryItem,
                        isVoiceQuery
                    )
                    parseDialogueCarousalImagesIfPresent(
                        innerDialogue,
                        chatItems,
                        date,
                        isHistoryItem
                    )
                    parseDialogueFormIfPresent(
                        innerDialogue = innerDialogue,
                        inputId = responseObject.inputId,
                        journeyId = responseObject.journeyId,
                        languageCode = responseObject.languageCode,
                        messageType = ServerConstants.CHAT_ITEM_TYPE_PROCESS_TREE,
                        messageId = responseObject.msgId,
                        nodeId = responseObject.nodeId,
                        sessionId = sessionId,
                        journeyName = responseObject.journeyName,
                        chatItems = chatItems,
                        date = date,
                        isHistoryItem = isHistoryItem,
                    )
                    parseDialogueFileIfPresent(innerDialogue, chatItems, date, isHistoryItem)
                    parseDialogueUrlsIfPresent(innerDialogue, chatItems, date, isHistoryItem)
                    parseDialogueVideoIfPresent(innerDialogue, chatItems, date, isHistoryItem)
                    parseDialogueAdvancedUrlIfPresent(innerDialogue, chatItems, date, isHistoryItem)
                }
            }
        }
    }

    private fun parseDialogueAdvancedUrlIfPresent(
        innerDialogue: Dialogue,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        historyItem: Boolean
    ) {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_ADVANCED_URL == innerDialogue.respondType && innerDialogue.responds is LinkedTreeMap<*, *>) {
            val advancedUrlResponseString = Gson().toJson(innerDialogue.responds)
            val advancedUrlResponse =
                Gson().fromJson(advancedUrlResponseString, AdvancedUrlRespond::class.java)
            convertToAdvancedUrlItem(
                advancedUrlResponse,
                date,
                historyItem
            )?.let { chatItem ->
                chatItems.add(chatItem)
            }
        }
    }

    private fun convertToAdvancedUrlItem(
        advancedUrlResponse: AdvancedUrlRespond,
        date: Date?,
        historyItem: Boolean
    ): IChatItem? {
        if (!advancedUrlResponse.uRLLink.isNullOrBlank() && !advancedUrlResponse.uRLText.isNullOrBlank()) {
            return IAdvancedUrlItem(
                urlLink = advancedUrlResponse.uRLLink,
                urlText = advancedUrlResponse.uRLText,
                date = date,
                isHistoryItem = historyItem,
                isBotItem = true
            )
        }
        return null
    }

    private fun parseDialogueVideoIfPresent(
        innerDialogue: Dialogue,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        historyItem: Boolean
    ) {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_VIDEO == innerDialogue.respondType && innerDialogue.responds is List<*>) {
            innerDialogue.responds.forEach {
                if (it is LinkedTreeMap<*, *>) {
                    val videoResponseString = Gson().toJson(it)
                    val videoResponse =
                        Gson().fromJson(videoResponseString, VideoResponse::class.java)
                    convertToVideoItem(
                        videoResponse,
                        ApiConstants.JOURNEY_DIALOG_VIDEO_FILE_BASE_URL,
                        date,
                        historyItem
                    )?.let { videoItem ->
                        chatItems.add(videoItem)
                    }
                }
            }
        }
    }

    private fun convertToVideoItem(
        videoResponse: VideoResponse,
        videoFileBaseUrl: String,
        date: Date?,
        historyItem: Boolean
    ): IChatItem? {
        if (ServerConstants.VIDEO_TYPE_FILE == videoResponse.videoType && !videoResponse.name.isNullOrBlank()) {
            return IVideoItem(
                title = videoResponse.title,
                description = videoResponse.description,
                fileSize = videoResponse.fileSize,
                url = videoFileBaseUrl + videoResponse.name,
                date = date,
                isBotItem = true,
                isHistoryItem = historyItem
            )
        } else if (ServerConstants.VIDEO_TYPE_URL == videoResponse.videoType && !videoResponse.filePath.isNullOrBlank()) {
            return IYouTubeVideoItem(
                title = videoResponse.title,
                description = videoResponse.description,
                url = videoResponse.filePath,
                date = date,
                isBotItem = true,
                isHistoryItem = historyItem
            )
        }
        return null
    }

    private fun parseDialogueJourneyIfPresent(innerDialogue: Dialogue): String? {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_JOURNEY == innerDialogue.respondType && innerDialogue.responds is String) {
            innerDialogue.responds.let {
                return it
            }
        }
        return null
    }

    private fun parseDialogueFileIfPresent(
        innerDialogue: Dialogue,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        historyItem: Boolean
    ) {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_FILE == innerDialogue.respondType && innerDialogue.responds is List<*>) {
            innerDialogue.responds.forEach {
                if (it is LinkedTreeMap<*, *>) {
                    val fileResponseString = Gson().toJson(it)
                    val fileResponse = Gson().fromJson(fileResponseString, FileResponse::class.java)
                    convertToFileItem(
                        fileResponse,
                        ApiConstants.JOURNEY_DIALOG_FILE_BASE_URL,
                        date,
                        historyItem
                    )?.let { fileItem ->
                        chatItems.add(fileItem)
                    }
                }
            }
        }
    }

    private fun convertToFileItem(
        fileResponse: FileResponse,
        fileBaseUrl: String,
        date: Date?,
        historyItem: Boolean
    ): IChatItem? {
        if (!fileResponse.name.isNullOrBlank() && !fileResponse.fileFormat.isNullOrBlank() && fileResponse.fileSize != null) {
            return IFileItem(
                fileUrl = fileBaseUrl + fileResponse.name,
                fileSize = fileResponse.fileSize.toLong(),
                fileName = fileResponse.name,
                date = date,
                isBotItem = true,
                isHistoryItem = historyItem
            )
        }
        return null
    }

    private fun parseDialogueUrlsIfPresent(
        innerDialogue: Dialogue,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        historyItem: Boolean
    ) {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_URL == innerDialogue.respondType && innerDialogue.responds is List<*>) {
            convertToUrlsItem(innerDialogue.responds, date, historyItem)?.let {
                chatItems.add(it)
            }
        }
    }

    private fun convertToUrlsItem(
        responds: List<*>,
        date: Date?,
        historyItem: Boolean
    ): IChatItem? {
        val urlList = ArrayList<String>()
        responds.forEach {
            if (it is String) {
                urlList.add(it)
            }
        }
        if (urlList.isNotEmpty()) {
            return IUrlsItem(
                urls = urlList,
                date = date,
                isBotItem = true,
                isHistoryItem = historyItem
            )
        }
        return null
    }

    private fun parseButtonsIfPresent(
        responseObject: Response?,
        chatItems: ArrayList<IChatItem>,
        resBy: String?,
        date: Date?,
        isHistoryItem: Boolean
    ) {
        if (ServerConstants.CHAT_ITEM_DISPLAY_AS_BUTTON == responseObject?.fields?.displayAs) {
            val options =
                responseObject.fields.fields?.mapNotNull { field ->
                    (field.translatedText
                        ?: field.text)?.obtainNonBlankOrNull()
                        ?.let {
                            IChatItemOption(
                                name = it,
                                isNew = false,
                                optionObject = IField(
                                    translatedText = field.translatedText,
                                    id = field.id,
                                    text = field.text,
                                    type = field.type,
                                    intent = field.intent,
                                    entity = field.entity,
                                    desc = field.desc,
                                    languageSelection = true == field.languageSelection,
                                    languageCode = field.languageCode,
                                    inputId = responseObject.inputId,
                                    journeyId = responseObject.journeyId,
                                    journeyName = responseObject.journeyName,
                                    nodeId = responseObject.nodeId,
                                    messageId = responseObject.msgId
                                )
                            )
                        }
                }
            if (options?.isNotEmpty() == true) {
                chatItems.add(
                    IMultipleOptionsItem(
                        options = options,
                        optionsStyle = IOptionsStyle.BUTTON,
                        optionsHeader = null,
                        date = date,
                        isBotItem = ServerConstants.CHAT_ITEM_RESPONSE_BY_BOT == resBy,
                        isHistoryItem = isHistoryItem
                    )
                )
            }
        }
    }

    private fun parseDialogueCarousalImagesIfPresent(
        innerDialogue: Dialogue,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        isHistoryItem: Boolean
    ) {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_CAROUSEL == innerDialogue.respondType && innerDialogue.responds is List<*>) {
            innerDialogue.responds.forEach {
                convertToCarousalItem(it, date, isHistoryItem)?.let { chatItem ->
                    chatItems.add(chatItem)
                }
            }
        }
    }

    private fun convertToCarousalItem(
        carousalResponse: Any?,
        date: Date?,
        isHistoryItem: Boolean,
    ): IChatItem? {
        if (carousalResponse is LinkedTreeMap<*, *>) {
            val carousalDatString = Gson().toJson(carousalResponse)
            val carousalData = Gson().fromJson(
                carousalDatString,
                CarousalImagesData::class.java
            )

            val carousals =
                carousalData.carouselData?.map { carousalItem ->
                    val imageData =
                        ViewUtils.convertFromBase64ToBitmap(carousalItem.imageData)
                    val buttonData =
                        carousalItem.buttonData?.map { buttonItem ->
                            val variables =
                                buttonItem.variables?.map { variable ->
                                    ICarousalButtonVariable(
                                        mapTo = variable.mapTo,
                                        variable = variable.variable
                                    )
                                }
                            ICarousalButtonData(
                                buttonName = buttonItem.buttonName,
                                buttonType = buttonItem.buttonType,
                                data = buttonItem.data,
                                variables = variables
                            )
                        }
                    ICarousalData(
                        imageName = carousalItem.imageName,
                        imageData = imageData,
                        imageDetails = carousalItem.imageDetails,
                        buttonData = buttonData
                    )
                }
            val carousalDataContainer = ICarousalDataContainer(
                name = carousalData?.name,
                carousalData = carousals,
                action = carousalData.action
            )
            return IDialogueCarousalItem(
                carousalDataContainer = carousalDataContainer,
                date = date,
                isBotItem = true,
                isHistoryItem = isHistoryItem
            )
        }
        return null
    }

    private fun parseDialogueMessagesIfPresent(
        innerDialogue: Dialogue,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        isHistoryItem: Boolean,
        isVoiceQuery: Boolean
    ) {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_TEXT == innerDialogue.respondType && innerDialogue.responds is List<*>) {
            val messages = ArrayList<String>()
            innerDialogue.responds.forEach {
                if (it is String) {
                    messages.add(it)
                }
            }
            if (messages.isNotEmpty()) {
                chatItems.add(
                    IMessagesItem(
                        messages = messages,
                        date = date,
                        isHtml = true,
                        isBotItem = true,
                        isHistoryItem = isHistoryItem,
                        isVoiceMessage = isVoiceQuery,
                    )
                )
            }
        }
    }

    private fun parseDialogueFormIfPresent(
        innerDialogue: Dialogue,
        inputId: String?,
        journeyId: String?,
        languageCode: String?,
        messageType: String?,
        messageId: String?,
        nodeId: Int?,
        sessionId: String?,
        journeyName: String?,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        isHistoryItem: Boolean
    ) {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_FORM == innerDialogue.respondType) {
            if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_FORM_TYPE_NORMAL == innerDialogue.formType && innerDialogue.responds is List<*>) {
                parseForm(
                    responds = innerDialogue.responds,
                    innerDialogue = innerDialogue,
                    isHistoryItem = isHistoryItem,
                    chatItems = chatItems,
                    date = date,
                    inputId = inputId,
                    journeyId = journeyId,
                    languageCode = languageCode,
                    messageType = messageType,
                    messageId = messageId,
                    nodeId = nodeId,
                    sessionId = sessionId,
                    journeyName = journeyName,
                    submitType = IChatFormItem.SubmitType.API
                )
            } else if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_FORM_TYPE_CUSTOM == innerDialogue.formType && innerDialogue.responds is LinkedTreeMap<*, *>) {
                val respondString = Gson().toJson(innerDialogue.responds)
                val customFormRespond =
                    Gson().fromJson(respondString, CustomFormRespond::class.java)
                customFormRespond.customFormData?.obtainNonBlankOrNull()?.let {
                    chatItems.add(
                        IChatCustomFormItem(
                            htmlData = it,
                            widthFraction = NumberUtils.convertPercentageStringToFraction(
                                customFormRespond.customFormAttribute?.width
                            ),
                            showBubble = customFormRespond.customFormAttribute?.showBox ?: true,
                            date = date,
                            isBotItem = true,
                            isHistoryItem = isHistoryItem,
                        )
                    )
                }
            } else if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_FORM_TYPE_CONVERSATIONAL == innerDialogue.formType && innerDialogue.responds is LinkedTreeMap<*, *>) {
                val conversationalFormQuestionString = Gson().toJson(innerDialogue.responds)
                val conversationalFormQuestion = Gson().fromJson(
                    conversationalFormQuestionString,
                    ConversationalFormQuestion::class.java
                )
                if (conversationalFormQuestion.responseType == ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_FORM_TYPE_CONVERSATIONAL_TEXT) {
                    (conversationalFormQuestion.response as? String)?.let { message ->
                        chatItems.add(
                            IMessageItem(
                                message = message,
                                isHtml = true,
                                date = date,
                                isBotItem = true
                            )
                        )
                    }
                }
            }
        }
    }

    private fun parseForm(
        responds: List<*>,
        innerDialogue: Dialogue? = null,
        isHistoryItem: Boolean = false,
        chatItems: ArrayList<IChatItem>,
        date: Date? = null,
        inputId: String? = null,
        journeyId: String? = null,
        languageCode: String? = null,
        messageType: String? = null,
        messageId: String? = null,
        nodeId: Int? = null,
        sessionId: String? = null,
        journeyName: String? = null,
        submitType: IChatFormItem.SubmitType,
        extras: Any? = null
    ) {
        val formItemsContainers = ArrayList<IFormItemsContainer>()

        val formResponds = ArrayList<FormRespond>()
        responds.forEach {
            if (it is LinkedTreeMap<*, *>) {
                val respondJson = Gson().toJson(it)
                val formItem = Gson().fromJson(respondJson, FormRespond::class.java)
                formResponds.add(formItem)
            }
        }
        formResponds.forEach { formRespond ->
            val formItems = ArrayList<IFormItem>()
            formRespond.children?.forEach { child ->
                val isRequired =
                    if (child.isRequired is Boolean) child.isRequired else false
                val widthFraction =
                    NumberUtils.convertPercentageStringToFraction(child.width)

                parseDialogueFormTextField(
                    child,
                    isRequired,
                    widthFraction,
                    innerDialogue
                )?.let { formItems.add(it) }

                parseDialogueFormTextArea(
                    child,
                    isRequired,
                    widthFraction,
                    innerDialogue
                )?.let { formItems.add(it) }

                parseDialogueFormRadioButtonsItem(
                    child,
                    innerDialogue,
                    isRequired,
                    widthFraction
                )?.let { formItems.add(it) }

                parseDialogueFormCheckboxesItem(
                    child,
                    innerDialogue,
                    isRequired,
                    widthFraction
                )?.let { formItems.add(it) }

                parseDialogueFormDateItem(
                    child,
                    isRequired,
                    widthFraction,
                    innerDialogue
                )?.let { formItems.add(it) }

                parseDialogueDropDownItem(
                    child,
                    innerDialogue,
                    isRequired,
                    widthFraction
                )?.let { formItems.add(it) }

                parseDialogueFormFileItem(
                    isRequired,
                    child,
                    widthFraction,
                    innerDialogue
                )?.let { formItems.add(it) }

                parseDialogueFormButtonItem(
                    innerDialogue,
                    isHistoryItem,
                    isRequired,
                    widthFraction,
                    child,
                    extras
                )?.let { formItems.add(it) }
            }
            formItemsContainers.add(IFormItemsContainer((formItems)))
        }

        chatItems.add(
            IChatFormItem(
                formItemContainers = formItemsContainers,
                date = date,
                isBotItem = true,
                isHistoryItem = isHistoryItem,
                inputId = inputId,
                journeyId = journeyId,
                languageCode = languageCode,
                messageType = messageType,
                messageId = messageId,
                nodeId = nodeId,
                onSubmit = innerDialogue?.onSubmit?.enVal,
                respondName = innerDialogue?.respondName,
                respondType = "${innerDialogue?.formType}_${innerDialogue?.respondType}_Upload",
                sessionId = sessionId,
                userInput = journeyName,
                submitType = submitType
            )
        )
    }

    private fun parseDialogueFormButtonItem(
        innerDialogue: Dialogue?,
        isHistoryItem: Boolean,
        isRequired: Boolean,
        widthFraction: Float,
        child: Children,
        extras: Any? = null
    ): IFormButtonItem? {
        if (ServerConstants.DIALOGUE_FORM_TYPE_BUTTON == child.type) {
            if (innerDialogue != null) {
                if (false == innerDialogue.issubmit || !isHistoryItem) {
                    innerDialogue.onSubmit?.transVal?.let { transVal ->
                        var submitName: String? = ""
                        if (transVal is LinkedTreeMap<*, *>) {
                            val transValJson = Gson().toJson(transVal)
                            val transValObject = Gson().fromJson(transValJson, TransVal::class.java)
                            submitName = transValObject.text
                        } else if (transVal is String) {
                            submitName = transVal
                        }
                        return IFormButtonItem(
                            isRequired = isRequired,
                            labelName = submitName,
                            widthFraction = widthFraction,
                            inputContent = child.inputcontent,
                            fieldId = child.fieldid,
                            variable = child.variable,
                            disableAfterClick = child.disableAfterClick
                        )
                    }
                }
            } else {
                return IFormButtonItem(
                    isRequired = isRequired,
                    labelName = child.labelname,
                    widthFraction = widthFraction,
                    inputContent = child.inputcontent,
                    fieldId = child.fieldid,
                    variable = child.variable,
                    extras = extras,
                    disableAfterClick = child.disableAfterClick
                )
            }
        }
        return null
    }

    private fun parseDialogueFormFileItem(
        isRequired: Boolean,
        child: Children,
        widthFraction: Float,
        innerDialogue: Dialogue?
    ): IFormFileItem? {
        if (ServerConstants.DIALOGUE_FORM_TYPE_FILE == child.type) {
            return IFormFileItem(
                isRequired = isRequired,
                labelName = child.labelname,
                widthFraction = widthFraction,
                inputContent = child.inputcontent,
                fieldId = child.fieldid,
                variable = child.variable
            ).apply {
                getSelectedArrayForVariable(
                    innerDialogue,
                    child.variable
                )?.let { selectedArray ->
                    try {
                        if (selectedArray.isNotEmpty()) {
                            val firstItem = selectedArray.first()
                            this.fileOriginalName = firstItem
                            val remoteFileUri =
                                Uri.parse(ApiConstants.JOURNEY_DIALOG_UPLOADED_FILE_BASE_URL + firstItem)
                            this.fileUri = remoteFileUri
                        }
                    } catch (ignore: Exception) {
                    }
                }
            }
        }
        return null
    }

    private fun parseDialogueDropDownItem(
        child: Children,
        innerDialogue: Dialogue?,
        isRequired: Boolean,
        widthFraction: Float
    ): IFormDropDownItem? {
        if (ServerConstants.DIALOGUE_FORM_TYPE_SELECT == child.type) {
            val dropDownItem = ArrayList<IFormItemCheckableOption>()
            child.options?.forEach { option ->
                dropDownItem.add(
                    IFormItemCheckableOption(
                        label = option.label,
                        optionId = option.optionid,
                        fieldId = option.fieldid,
                        isChecked = option.checked ?: false,
                        value = option.value
                    ).apply {
                        getSelectedValueForVariable(
                            innerDialogue,
                            child.variable
                        )?.let { enteredValue ->
                            if (enteredValue == option.value) {
                                isChecked = true
                            }
                        }
                    }
                )
            }
            return IFormDropDownItem(
                options = dropDownItem,
                isRequired = isRequired,
                labelName = child.labelname,
                widthFraction = widthFraction,
                inputContent = child.inputcontent,
                fieldId = child.fieldid,
                variable = child.variable,
                selectedOptionPosition = dropDownItem.find { it.isChecked }
                    ?.let {
                        dropDownItem.indexOf(it)
                    } ?: 0
            )
        }
        return null
    }

    private fun parseDialogueFormDateItem(
        child: Children,
        isRequired: Boolean,
        widthFraction: Float,
        innerDialogue: Dialogue?
    ): IFormDateItem? {
        if (ServerConstants.DIALOGUE_FORM_TYPE_DATE == child.type) {
            return IFormDateItem(
                selectedDate = DateUtils.convertFromServerDate(child.value),
                dateType = child.datetype,
                placeHolder = child.placeholder,
                isRequired = isRequired,
                labelName = child.labelname,
                widthFraction = widthFraction,
                inputContent = child.inputcontent,
                fieldId = child.fieldid,
                variable = child.variable
            ).apply {
                getSelectedValueForVariable(
                    innerDialogue,
                    child.variable
                )?.let { enteredValue ->
                    DateUtils.convertFromServerDate(enteredValue)
                        ?.let { parsedDate ->
                            selectedDate = parsedDate
                        }
                }
            }
        }
        return null
    }

    private fun parseDialogueFormCheckboxesItem(
        child: Children,
        innerDialogue: Dialogue?,
        isRequired: Boolean,
        widthFraction: Float
    ): IFormCheckboxesItem? {
        if (ServerConstants.DIALOGUE_FORM_TYPE_CHECKBOX == child.type) {
            val checkBoxes = ArrayList<IFormItemCheckableOption>()
            child.options?.forEach { option ->
                checkBoxes.add(
                    IFormItemCheckableOption(
                        label = option.label,
                        optionId = option.optionid,
                        fieldId = option.fieldid,
                        isChecked = option.checked ?: false,
                        value = option.value
                    ).apply {
                        getSelectedValueForVariable(
                            innerDialogue,
                            child.variable
                        )?.let { enteredValue ->
                            option.label?.let { optionLabel ->
                                if (enteredValue.contains(optionLabel)) {
                                    isChecked = true
                                }
                            }
                        }
                    }
                )
            }
            return IFormCheckboxesItem(
                options = checkBoxes,
                isRequired = isRequired,
                labelName = child.labelname,
                widthFraction = widthFraction,
                inputContent = child.inputcontent,
                fieldId = child.fieldid,
                variable = child.variable
            )
        }
        return null
    }

    private fun parseDialogueFormRadioButtonsItem(
        child: Children,
        innerDialogue: Dialogue?,
        isRequired: Boolean,
        widthFraction: Float
    ): IFormRadioButtonsItem? {
        if (ServerConstants.DIALOGUE_FORM_TYPE_RADIO == child.type) {
            val radioButtons = ArrayList<IFormItemCheckableOption>()
            child.options?.forEach { option ->
                radioButtons.add(
                    IFormItemCheckableOption(
                        label = option.label,
                        optionId = option.optionid,
                        fieldId = option.fieldid,
                        isChecked = option.checked ?: false,
                        value = option.value
                    ).apply {
                        getSelectedValueForVariable(
                            innerDialogue,
                            child.variable
                        )?.let { enteredValue ->
                            if (enteredValue == value) {
                                isChecked = true
                            }
                        }
                    }
                )
            }
            return IFormRadioButtonsItem(
                options = radioButtons,
                isRequired = isRequired,
                labelName = child.labelname,
                widthFraction = widthFraction,
                inputContent = child.inputcontent,
                fieldId = child.fieldid,
                variable = child.variable
            )
        }
        return null
    }

    private fun parseDialogueFormTextArea(
        child: Children,
        isRequired: Boolean,
        widthFraction: Float,
        innerDialogue: Dialogue?
    ): IFormTextAreaItem? {
        if (ServerConstants.DIALOGUE_FORM_TYPE_TEXT_AREA == child.type) {
            return IFormTextAreaItem(
                maxLength = NumberUtils.parseInt(child.maxlength),
                minLength = NumberUtils.parseInt(child.minlength),
                rows = NumberUtils.parseIntFromAny(child.rows) ?: 2,
                placeHolder = child.placeholder,
                value = child.value,
                textBoxTypeSelected = child.textboxTypeSelected,
                isRequired = isRequired,
                labelName = child.labelname,
                widthFraction = widthFraction,
                inputContent = child.inputcontent,
                fieldId = child.fieldid,
                variable = child.variable
            ).apply {
                getSelectedValueForVariable(
                    innerDialogue,
                    child.variable
                )?.let { enteredValue ->
                    value = enteredValue
                }
            }
        }
        return null
    }

    private fun parseDialogueFormTextField(
        child: Children,
        isRequired: Boolean,
        widthFraction: Float,
        innerDialogue: Dialogue?
    ): IFormTextItem? {
        if (ServerConstants.DIALOGUE_FORM_TYPE_TEXT == child.type) {
            return IFormTextItem(
                maxLength = NumberUtils.parseInt(child.maxlength),
                minLength = NumberUtils.parseInt(child.minlength),
                placeHolder = child.placeholder,
                value = child.value,
                textBoxTypeSelected = child.textboxTypeSelected,
                prefixValue = AppConstants.FORM_ITEM_DEFAULT_PHONE_COUNTRY_CODE,
                isRequired = isRequired,
                labelName = child.labelname,
                widthFraction = widthFraction,
                inputContent = child.inputcontent,
                fieldId = child.fieldid,
                variable = child.variable
            ).apply {
                getSelectedValueForVariable(
                    innerDialogue,
                    child.variable
                )?.let { enteredValue ->
                    if (ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_PHONE_NUMBER == child.textboxTypeSelected) {
                        val phoneNumberParts = enteredValue.split(" ")
                        if (phoneNumberParts.size > 1) {
                            prefixValue = phoneNumberParts.first()
                            value = enteredValue.substring(
                                enteredValue.indexOf(phoneNumberParts[1])
                            )
                        }
                    } else {
                        value = enteredValue
                    }
                }
            }
        }
        return null
    }

    private fun getSelectedValueForVariable(
        innerDialogue: Dialogue?,
        variable: String?
    ): String? {
        if (variable == null) return null
        innerDialogue?.selectedFieldsVal?.forEach { selectedFieldsVal ->
            if (selectedFieldsVal is LinkedTreeMap<*, *>) {
                if (selectedFieldsVal.containsKey(variable) && selectedFieldsVal[variable] is String) {
                    val selectedVal = selectedFieldsVal[variable] as String
                    if (selectedVal.isNotBlank()) {
                        return selectedVal
                    }
                }
            }
        }
        return null
    }

    private fun getSelectedArrayForVariable(
        innerDialogue: Dialogue?,
        variable: String?
    ): List<String>? {
        if (variable == null) return null
        innerDialogue?.selectedFieldsVal?.forEach { selectedFieldsVal ->
            if (selectedFieldsVal is LinkedTreeMap<*, *>) {
                if (selectedFieldsVal.containsKey(variable) && selectedFieldsVal[variable] is List<*>) {
                    val selectedArray = ArrayList<String>()
                    (selectedFieldsVal[variable] as List<*>).forEach { value ->
                        if (value is String) {
                            selectedArray.add(value)
                        }
                    }
                    if (selectedArray.isNotEmpty()) {
                        return selectedArray
                    }
                }
            }
        }
        return null
    }

    private fun parseDialogueImageIfPresent(
        innerDialogue: Dialogue,
        chatItems: ArrayList<IChatItem>,
        date: Date?,
        isHistoryItem: Boolean
    ) {
        if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_IMAGE == innerDialogue.respondType && innerDialogue.responds is String) {
            chatItems.add(
                IImageItem(
                    imageUrl = ApiConstants.JOURNEY_DIALOG_IMAGE_BASE_URL + innerDialogue.responds,
                    date = date,
                    isBotItem = true,
                    isHistoryItem = isHistoryItem
                )
            )
        }
    }

    private fun convertTextMessageResponseToChatItem(
        chatResponse: ChatResponse?,
        chatResponseString: String?,
        date: Date?,
        languageCode: String
    ): List<IChatItem>? {
        val chatItems = ArrayList<IChatItem>()
        parseTextMessages(chatResponse, chatResponseString, date, languageCode, chatItems)
        return chatItems.ifEmpty { null }
    }

    fun convertChatResponseToLiveChatItems(
        liveChatResponse: LiveChatResponse?,
        liveChatForm: ILiveChatForm,
        queryType: String?
    ): List<IChatItem>? {
        val date = Date()
        val chatItems = ArrayList<IChatItem>()
        chatItems.add(
            IMessageItem(
                message = liveChatResponse?.livechatMsg ?: "",
                isHtml = true,
                date = date,
                isBotItem = true,
                isVoiceMessage = ApiConstants.CHAT_QUERY_TYPE_VOICE == queryType,
                hideTime = true
            )
        )
        chatItems.add(
            IMultipleOptionsItem(
                options = ArrayList<IChatItemOption>().apply {
                    add(
                        IChatItemOption(
                            name = liveChatResponse?.buttonTrue ?: "",
                            isNew = false,
                            optionObject = liveChatForm
                        )
                    )
                    add(
                        IChatItemOption(
                            name = liveChatResponse?.buttonFalse ?: "",
                            isNew = false,
                            optionObject = AppConstants.ACTION_LIVE_CHAT_BUTTON_FALSE
                        )
                    )
                },
                optionsStyle = IOptionsStyle.BUTTON,
                optionsHeader = null
            )
        )
        return chatItems.ifEmpty { null }
    }

    fun parseLiveChatDecisionResponseIfPresent(
        chatResponse: ChatResponse?
    ): LiveChatResponse? {
        if (chatResponse?.processLivechat == true
            && chatResponse.livechatResponse?.livechatMsg?.isNotBlank() == true
            && chatResponse.livechatResponse.buttonTrue?.isNotBlank() == true
            && chatResponse.livechatResponse.buttonFalse?.isNotBlank() == true
        ) {
            return chatResponse.livechatResponse
        }
        return null
    }

    fun parseDialogueLiveChatDecisionItem(
        chatResponse: ChatResponse?
    ): LiveChatResponse? {
        try {
            chatResponse?.response?.let { response ->
                val responseJson = Gson().toJson(response)
                val responseObject = Gson().fromJson(responseJson, Response::class.java)

                responseObject.dialogues?.forEach { dialogue ->
                    if (dialogue.isNotEmpty()) {
                        dialogue.forEach { innerDialogue ->
                            if (ServerConstants.JOURNEY_DIALOGUE_RESPOND_TYPE_SEND_TO_LIVE == innerDialogue.respondType) {
                                val liveChatResponseJson = Gson().toJson(innerDialogue.responds)
                                val liveChatResponseObject =
                                    Gson().fromJson(
                                        liveChatResponseJson,
                                        DialogueLiveChatResponse::class.java
                                    )
                                if (liveChatResponseObject.livechatResponse?.livechatMsg?.isNotBlank() == true
                                    && liveChatResponseObject.livechatResponse.buttonTrue?.isNotBlank() == true
                                    && liveChatResponseObject.livechatResponse.buttonFalse?.isNotBlank() == true
                                ) {
                                    return liveChatResponseObject.livechatResponse
                                }
                            }
                        }
                    }
                }
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
        return null
    }

    private fun parseTextMessages(
        chatResponse: ChatResponse?,
        chatResponseString: String?,
        date: Date?,
        languageCode: String,
        chatItems: ArrayList<IChatItem>
    ) {
        try {
            if (chatResponse?.response is String) {
                var responseText: String? = chatResponse.response

                val keyForLanguageSpecificResponse =
                    ApiConstants.LANGUAGE_SPECIFIC_RESPONSE_KEY_PREFIX + languageCode
                val mapType: Type = object : TypeToken<Map<String?, Any?>?>() {}.type
                val mapResponse = Gson().fromJson<Map<String?, Any?>>(
                    chatResponseString,
                    mapType
                )
                if (mapResponse.containsKey(keyForLanguageSpecificResponse) && mapResponse[keyForLanguageSpecificResponse] is String) {
                    val responseTextInLocaleLanguage =
                        mapResponse[keyForLanguageSpecificResponse] as String
                    responseText = responseTextInLocaleLanguage
                }
                responseText?.obtainNonBlankOrNull()?.let {
                    chatItems.add(
                        IMessageItem(
                            message = it,
                            date = date,
                            isHtml = true,
                            isBotItem = true,
                            isHistoryItem = false,
                            isVoiceMessage = ApiConstants.CHAT_QUERY_TYPE_VOICE == chatResponse.queryType
                        )
                    )
                }
            } else if (chatResponse?.response is List<*>) {
                chatResponse.response.forEach { response ->
                    if (response is String && response.isNotBlank()) {
                        chatItems.add(
                            IMessageItem(
                                message = response,
                                date = date,
                                isHtml = true,
                                isBotItem = true,
                                isHistoryItem = false,
                                isVoiceMessage = ApiConstants.CHAT_QUERY_TYPE_VOICE == chatResponse.queryType
                            )
                        )
                    }
                }
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
    }

    private fun convertHistoryMessageToChatItem(
        chatDataItem: ChatData,
        date: Date?
    ): List<IChatItem> {
        val chatItems = ArrayList<IChatItem>()
        if (ServerConstants.CHAT_ITEM_RESPONSE_BY_BOT == chatDataItem.resBy) {
            if (chatDataItem.response is String && chatDataItem.response.isNotBlank()) {
                chatItems.add(
                    IMessageItem(
                        message = chatDataItem.response,
                        date = date,
                        isHtml = true,
                        isLoading = false,
                        isBotItem = true,
                        isHistoryItem = true
                    )
                )
            } else if (chatDataItem.response is List<*>) {
                chatDataItem.response.forEach { response ->
                    if (response is String && response.isNotBlank()) {
                        chatItems.add(
                            IMessageItem(
                                message = response,
                                date = date,
                                isHtml = true,
                                isLoading = false,
                                isBotItem = true,
                                isHistoryItem = true
                            )
                        )
                    }
                }
            }
        } else if (ServerConstants.CHAT_ITEM_RESPONSE_BY_USER == chatDataItem.resBy) {
            (chatDataItem.userInput as? String)?.obtainNonBlankOrNull()?.let {
                chatItems.add(
                    IMessageItem(
                        message = it,
                        date = date,
                        isHtml = true,
                        isLoading = false,
                        isBotItem = false,
                        isHistoryItem = true
                    )
                )
            }
        }
        return chatItems
    }

    private fun convertConflictItemToChatItem(
        response: Any,
        messageDate: Date?,
        isHistoryItem: Boolean
    ): ArrayList<IChatItem> {
        val chatItems = ArrayList<IChatItem>()
        val responseString = Gson().toJson(response)
        val conflictResponse = Gson().fromJson(responseString, ConflictResponse::class.java)
        if (conflictResponse.conflictMessage?.isNotBlank() == true && !conflictResponse.faqList.isNullOrEmpty()) {
            chatItems.add(
                IChatConflictItem(
                    message = conflictResponse.conflictMessage,
                    faqList = conflictResponse.faqList.map { faq ->
                        IChatConflictChildItem(
                            question = faq.question,
                            journeyId = faq.journeyID,
                            faqId = faq.fAQID
                        )
                    },
                    date = messageDate,
                    isBotItem = true,
                    isHistoryItem = isHistoryItem,
                )
            )
        }
        return chatItems
    }

    private fun convertFaqItemToChatItem(
        response: List<*>,
        date: Date?,
        isHistoryItem: Boolean,
        isVoiceQuery: Boolean
    ): List<IChatItem>? {
        val chatItems = ArrayList<IChatItem>()
        convertToFaqResponse(response)?.forEach { faqResponse ->
            parseFaqMessageItem(
                faqResponse,
                date,
                isHistoryItem,
                isVoiceQuery
            )?.let { chatItems.add(it) }
            parseFaqImage(faqResponse, date, isHistoryItem)?.let { chatItems.add(it) }
            parseFaqVideo(faqResponse, date, isHistoryItem)?.let { chatItems.add(it) }
            parseFaqFile(faqResponse, date, isHistoryItem)?.let { chatItems.add(it) }
            parseFaqUrls(faqResponse, date, isHistoryItem)?.let { chatItems.add(it) }
            parseFaqCarousal(faqResponse, date, isHistoryItem)?.let { chatItems.add(it) }
        }
        return chatItems.ifEmpty { null }
    }

    private fun convertToFaqResponse(response: List<*>): List<FaqResponse>? {
        val faqResponses = ArrayList<FaqResponse>()
        try {
            response.forEach { outerResponse ->
                if (outerResponse is List<*>) {
                    outerResponse.forEach { innerResponse ->
                        val innerResponseString = Gson().toJson(innerResponse)
                        faqResponses.add(
                            Gson().fromJson(
                                innerResponseString,
                                FaqResponse::class.java
                            )
                        )
                    }
                }
            }
        } catch (ignore: Exception) {
        }
        return faqResponses.ifEmpty { null }
    }

    private fun parseFaqImage(
        faqResponse: FaqResponse,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (faqResponse.respondWith == ServerConstants.FAQ_RESPOND_TYPE_IMAGE && faqResponse.responseMsg is LinkedTreeMap<*, *>) {
            val responseString = Gson().toJson(faqResponse.responseMsg)
            val imageResponse =
                Gson().fromJson(responseString, FileResponse::class.java)
            if (!imageResponse.name.isNullOrBlank()) {
                return IImageItem(
                    imageUrl = ApiConstants.FAQ_IMAGE_BASE_URL + imageResponse.name,
                    date = date,
                    isBotItem = true,
                    isHistoryItem = isHistoryItem
                )
            }
        }
        return null
    }

    private fun parseFaqVideo(
        faqResponse: FaqResponse,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (faqResponse.respondWith == ServerConstants.FAQ_RESPOND_TYPE_VIDEO && faqResponse.responseMsg is LinkedTreeMap<*, *>) {
            val responseString = Gson().toJson(faqResponse.responseMsg)
            val videoResponse =
                Gson().fromJson(responseString, VideoResponse::class.java)
            return convertToVideoItem(
                videoResponse,
                ApiConstants.FAQ_VIDEO_BASE_URL,
                date,
                isHistoryItem
            )
        }
        return null
    }

    private fun parseFaqMessageItem(
        faqResponse: FaqResponse?,
        date: Date?,
        isHistoryItem: Boolean,
        isVoiceQuery: Boolean
    ): IChatItem? {
        if (faqResponse?.respondWith == ServerConstants.FAQ_RESPOND_TYPE_TEXT && faqResponse.responseMsg is List<*>) {
            val messages = ArrayList<String>()

            faqResponse.responseMsg.forEach { message ->
                if (message is String) {
                    messages.add(message)
                }
            }

            return IMessagesItem(
                messages = messages,
                date = date,
                isHtml = true,
                isLoading = false,
                isBotItem = true,
                isHistoryItem = isHistoryItem,
                isVoiceMessage = isVoiceQuery
            )
        }
        return null
    }

    private fun parseFaqFile(
        faqResponse: FaqResponse,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (faqResponse.respondWith == ServerConstants.FAQ_RESPOND_TYPE_FILE && faqResponse.responseMsg is LinkedTreeMap<*, *>) {
            val responseString = Gson().toJson(faqResponse.responseMsg)
            val fileResponse =
                Gson().fromJson(responseString, FileResponse::class.java)
            return convertToFileItem(
                fileResponse,
                ApiConstants.FAQ_FILE_BASE_URL,
                date,
                isHistoryItem
            )
        }
        return null
    }

    private fun parseFaqUrls(
        faqResponse: FaqResponse,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (faqResponse.respondWith == ServerConstants.FAQ_RESPOND_TYPE_URL && faqResponse.responseMsg is List<*>) {
            return convertToUrlsItem(faqResponse.responseMsg, date, isHistoryItem)
        }
        return null
    }

    private fun parseFaqCarousal(
        faqResponse: FaqResponse,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (faqResponse.respondWith == ServerConstants.FAQ_RESPOND_TYPE_CAROUSAL && faqResponse.responseMsg is LinkedTreeMap<*, *>) {
            return convertToCarousalItem(faqResponse.responseMsg, date, isHistoryItem)
        }
        return null
    }

    private fun parseFaqJourney(faqResponse: FaqResponse): String? {
        if (faqResponse.respondWith == ServerConstants.FAQ_RESPOND_TYPE_JOURNEY && faqResponse.responseMsg is String) {
            return faqResponse.responseMsg
        }
        return null
    }

    private fun convertTabularInfoItemToChatItem(
        response: LinkedTreeMap<*, *>,
        date: Date?,
        isHistoryItem: Boolean,
        isVoiceQuery: Boolean
    ): List<IChatItem>? {
        val chatItems = ArrayList<IChatItem>()
        val responseString = Gson().toJson(response)
        val tabularInfoResponse = Gson().fromJson(responseString, TabularInfoResponse::class.java)

        val dialogues = ArrayList<Dialogue>()
        tabularInfoResponse.dialogues?.forEach { innerDialogues ->
            innerDialogues.forEach { dialogue ->
                dialogues.add(dialogue)
            }
        }

        //Parse the dialogues before the main content
        dialogues.filter { dialogue ->
            dialogue.respondPosition == ServerConstants.TABULAR_INFO_DIALOGUE_RESPOND_POSITION_BEFORE
        }.forEach { dialogue ->
            parseTabularInfoDialogueItems(dialogue, date, isHistoryItem, chatItems, isVoiceQuery)
        }

        if (!tabularInfoResponse.headings.isNullOrEmpty()
            && tabularInfoResponse.headings.size == tabularInfoResponse.enHeadings?.size
        ) {
            chatItems.add(
                ITabularInfoItem(
                    responseParams = ITabularInfoResponseParams(
                        search = tabularInfoResponse.responseParams?.search,
                        searchLocation = tabularInfoResponse.responseParams?.searchLocation,
                        select = tabularInfoResponse.responseParams?.select,
                        submit = tabularInfoResponse.responseParams?.submit
                    ),
                    childItems = tabularInfoResponse.enHeadings.mapIndexed { index, item ->
                        ITabularInfoChildItem(
                            heading = tabularInfoResponse.headings[index],
                            enHeading = item,
                            items = getTabularDropDownItemsForKey(item, response)
                        )
                    },
                    date = date,
                    isHistoryItem = isHistoryItem,
                    inputId = tabularInfoResponse.inputId,
                    isMapEnabled = tabularInfoResponse.isMapEnabled ?: false,
                    journeyId = tabularInfoResponse.journeyId,
                    journeyName = tabularInfoResponse.jrName,
                    languageCode = tabularInfoResponse.languageCode,
                    messageType = tabularInfoResponse.messageType,
                    msgId = tabularInfoResponse.msgId,
                    responseId = tabularInfoResponse.responseId,
                    latitude = if (tabularInfoResponse.isMapEnabled == true) NumberUtils.parseDouble(
                        tabularInfoResponse.latitude
                    ) else null,
                    longitude = if (tabularInfoResponse.isMapEnabled == true) NumberUtils.parseDouble(
                        tabularInfoResponse.longitude
                    ) else null
                )
            )
        }


        //Parse the dialogues after the main content
        dialogues.filter { dialogue ->
            dialogue.respondPosition == ServerConstants.TABULAR_INFO_DIALOGUE_RESPOND_POSITION_AFTER
        }.forEach { dialogue ->
            parseTabularInfoDialogueItems(dialogue, date, isHistoryItem, chatItems, isVoiceQuery)
        }

        return chatItems.ifEmpty { null }
    }

    private fun getTabularDropDownItemsForKey(
        key: String,
        response: LinkedTreeMap<*, *>
    ): List<ITabularDropDownItem> {
        val dropDownItems = ArrayList<ITabularDropDownItem>()
        dropDownItems.add(ITabularDropDownItem(null, null))
        if (response.containsKey(key)) {
            val items = response[key]
            if (items is List<*>) {
                dropDownItems.addAll(items.map { item ->
                    val itemString = Gson().toJson(item)
                    val tabularDropDownItem =
                        Gson().fromJson(itemString, TabularDopDownItem::class.java)
                    ITabularDropDownItem(
                        enVal = tabularDropDownItem.enVal,
                        transVal = tabularDropDownItem.transVal
                    )
                })
            }
        }
        return dropDownItems
    }

    private fun parseTabularInfoDialogueItems(
        dialogue: Dialogue,
        date: Date?,
        isHistoryItem: Boolean,
        chatItems: ArrayList<IChatItem>,
        isVoiceQuery: Boolean
    ) {
        parseTabularDialogueMsg(
            dialogue,
            date,
            isHistoryItem,
            isVoiceQuery
        )?.let { chatItems.add(it) }
        parseTabularDialogueImage(dialogue, date, isHistoryItem)?.let { chatItems.add(it) }
        parseTabularDialogueVideo(dialogue, date, isHistoryItem)?.let { chatItems.add(it) }
        parseTabularDialogueFile(dialogue, date, isHistoryItem)?.let { chatItems.add(it) }
        parseTabularDialogueUrl(dialogue, date, isHistoryItem)?.let { chatItems.add(it) }
        parseTabularDialogueCarousal(
            dialogue,
            date,
            isHistoryItem
        )?.let { chatItems.add(it) }
    }

    private fun parseTabularDialogueMsg(
        dialogue: Dialogue,
        date: Date?,
        isHistoryItem: Boolean,
        isVoiceQuery: Boolean
    ): IChatItem? {
        if (dialogue.respondWith == ServerConstants.TABULAR_INFO_DIALOGUE_RESPOND_TYPE_TEXT && dialogue.responseMsg is String) {
            return IMessagesItem(
                messages = ArrayList<String>().apply {
                    add(dialogue.responseMsg)
                },
                isHtml = true,
                date = date,
                isBotItem = true,
                isHistoryItem = isHistoryItem,
                isVoiceMessage = isVoiceQuery
            )
        }
        return null
    }

    private fun parseTabularDialogueImage(
        dialogue: Dialogue,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (dialogue.respondWith == ServerConstants.TABULAR_INFO_DIALOGUE_RESPOND_TYPE_IMAGE && dialogue.responseMsg is LinkedTreeMap<*, *>) {
            val responseString = Gson().toJson(dialogue.responseMsg)
            val imageResponse =
                Gson().fromJson(responseString, FileResponse::class.java)
            if (!imageResponse.name.isNullOrBlank()) {
                return IImageItem(
                    imageUrl = ApiConstants.TABULAR_DIALOGUE_IMAGE_BASE_URL + imageResponse.name,
                    date = date,
                    isBotItem = true,
                    isHistoryItem = isHistoryItem
                )
            }
        }
        return null
    }

    private fun parseTabularDialogueVideo(
        dialogue: Dialogue,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (dialogue.respondWith == ServerConstants.TABULAR_INFO_DIALOGUE_RESPOND_TYPE_VIDEO && dialogue.responseMsg is LinkedTreeMap<*, *>) {
            val videoResponseString = Gson().toJson(dialogue.responseMsg)
            val videoResponse =
                Gson().fromJson(videoResponseString, VideoResponse::class.java)
            convertToVideoItem(
                videoResponse,
                ApiConstants.TABULAR_DIALOGUE_VIDEO_BASE_URL,
                date,
                isHistoryItem
            )?.let { videoItem ->
                return videoItem
            }
        }
        return null
    }

    private fun parseTabularDialogueFile(
        dialogue: Dialogue,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (dialogue.respondWith == ServerConstants.TABULAR_INFO_DIALOGUE_RESPOND_TYPE_FILE && dialogue.responseMsg is LinkedTreeMap<*, *>) {
            val fileResponseString = Gson().toJson(dialogue.responseMsg)
            val fileResponse = Gson().fromJson(fileResponseString, FileResponse::class.java)
            convertToFileItem(
                fileResponse,
                ApiConstants.TABULAR_DIALOGUE_FILE_BASE_URL,
                date,
                isHistoryItem
            )?.let { fileItem ->
                return fileItem
            }
        }
        return null
    }

    private fun parseTabularDialogueUrl(
        dialogue: Dialogue,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (dialogue.respondWith == ServerConstants.TABULAR_INFO_DIALOGUE_RESPOND_TYPE_URL && dialogue.responseMsg is String) {
            return IUrlsItem(
                urls = ArrayList<String>().apply { add(dialogue.responseMsg) },
                date = date,
                isBotItem = true,
                isHistoryItem = isHistoryItem
            )
        }
        return null
    }

    private fun parseTabularDialogueCarousal(
        dialogue: Dialogue,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        if (dialogue.respondWith == ServerConstants.TABULAR_INFO_DIALOGUE_RESPOND_TYPE_CAROUSAL && dialogue.responseMsg is LinkedTreeMap<*, *>) {
            return convertToCarousalItem(
                carousalResponse = dialogue.responseMsg,
                date = date,
                isHistoryItem = isHistoryItem
            )
        }
        return null
    }

    fun convertLiveChatFormToChatItems(liveChatForm: ILiveChatForm): List<IChatItem>? {
        if (liveChatForm.formData is List<*>) {
            val chatItems = ArrayList<IChatItem>()
            parseForm(
                responds = liveChatForm.formData,
                chatItems = chatItems,
                submitType = IChatFormItem.SubmitType.SOCKET,
                extras = liveChatForm
            )
            return chatItems
        }
        return null
    }

    fun convertNewUserReplyToChatItems(
        response: String,
        sessionId: String
    ): ILiveChatDataContainer {
        val newChatItems = ArrayList<IChatItem>()
        var liveChatData: ILiveChatData? = null
        var liveChatStatus: ILiveChatDataContainer.ILiveChatStatus =
            ILiveChatDataContainer.ILiveChatStatus.IDLE
        var liveChatFeedbackRequest: ILiveChatFeedbackRequest? = null
        var sendJoinMeHereRequest = false
        try {
            val addTeamList: Boolean
            val userReply = Gson().fromJson(response, AgentConnectResponse::class.java)
            if (sessionId == userReply.sessionId || sessionId == userReply.data?.sessionId) {
                val date = DateUtils.convertFromUtc2ToLocalDate(userReply.responseTime)
                val isUserResponse =
                    ServerConstants.LIVE_CHAT_RESPONSE_BY_CLIENT == userReply.responseBy
                if (ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENTS_READY == userReply.responseType) {
                    (userReply.response as? String)?.let { message ->
                        newChatItems.add(
                            IMessageItem(
                                message = message,
                                isBotItem = !isUserResponse,
                                date = date
                            )
                        )
                    }
                    liveChatData = convertToLiveChatData(userReply)
                    addTeamList = true
                    liveChatStatus = ILiveChatDataContainer.ILiveChatStatus.READY
                } else if (ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENTS_QUEUED == userReply.responseType) {
                    convertToLiveChatStatusItem(userReply.response, userReply.responseType)?.let {
                        newChatItems.add(it)
                    }
                    liveChatData = convertToLiveChatData(userReply)
                    addTeamList = false
                    liveChatStatus = ILiveChatDataContainer.ILiveChatStatus.QUEUED
                } else if (ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENT_CONNECTED == userReply.responseType) {
                    convertToLiveChatStatusItem(userReply.response, userReply.responseType)?.let {
                        newChatItems.add(it)
                    }
                    liveChatData = convertToLiveChatData(userReply)
                    addTeamList = false
                    liveChatStatus = ILiveChatDataContainer.ILiveChatStatus.CONNECTED
                } else if (ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENTS_UNAVAILABLE == userReply.responseType) {
                    convertToLiveChatStatusItem(userReply.response, userReply.responseType)?.let {
                        newChatItems.add(it)
                    }
                    addTeamList = false
                    liveChatStatus = ILiveChatDataContainer.ILiveChatStatus.UNAVAILABLE
                } else if (ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENT_DISCONNECTED == userReply.responseType) {
                    convertToLiveChatStatusItem(userReply.response, userReply.responseType)?.let {
                        newChatItems.add(it)
                    }
                    liveChatData = convertToLiveChatData(userReply)
                    addTeamList = false
                    liveChatStatus = ILiveChatDataContainer.ILiveChatStatus.DISCONNECTED

                    if (userReply.lcCsat == true) {
                        liveChatFeedbackRequest = ILiveChatFeedbackRequest(
                            surveyMessage = userReply.surveyMessage,
                            ratings = HashMap<Int, ILiveChatRatingItem>().apply {
                                userReply.ratingsList?.forEach { ratingItem ->
                                    ratingItem.indexNo?.let { index ->
                                        put(
                                            index,
                                            ILiveChatRatingItem(
                                                isRatingPositive = ratingItem.isRatingPositive
                                                    ?: false,
                                                rating = ratingItem.rating
                                            )
                                        )
                                    }
                                }
                            },
                            liveChatData = liveChatData
                        )
                    }

                } else if (ServerConstants.LIVE_CHAT_RESPONSE_TYPE_MEDIA == userReply.responseType) {
                    convertToLiveChatMediaItem(
                        response = userReply.response,
                        isUserResponse = isUserResponse,
                        date = date,
                        isHistoryItem = false
                    )?.let {
                        newChatItems.add(it)
                    }

                    addTeamList = false
                } else if (ServerConstants.LIVE_CHAT_RESPONSE_TYPE_CHAT_TRANSFER == userReply.responseType) {
                    convertToLiveChatStatusItem(userReply.response, userReply.responseType)?.let {
                        newChatItems.add(it)
                    }
                    liveChatData = convertToLiveChatData(
                        userReply = userReply,
                        agentDetails = userReply.agentDetails
                    )
                    addTeamList = false
                    liveChatStatus = ILiveChatDataContainer.ILiveChatStatus.CONNECTED
                    sendJoinMeHereRequest = true
                } else {
                    (userReply.response as? String)?.let { message ->
                        newChatItems.add(
                            IMessageItem(
                                message = message,
                                isBotItem = !isUserResponse,
                                date = date
                            )
                        )
                    }
                    addTeamList = false
                }
                if (!userReply.data?.teamList.isNullOrEmpty() && addTeamList) {
                    newChatItems.add(
                        IMultipleOptionsItem(
                            options = ArrayList<IChatItemOption>().apply {
                                userReply.data?.teamList?.forEach { team ->
                                    team.traslatedTeam?.let { translatedTeam ->
                                        add(
                                            IChatItemOption(
                                                name = translatedTeam,
                                                isNew = false,
                                                optionObject = ILiveChatTeam(
                                                    team = team.team,
                                                    translatedTeam = team.traslatedTeam,
                                                    session = ILiveChatSession(
                                                        currentSessionId = userReply.data.currentSessionId,
                                                        displayOutput = userReply.data.displayOutput,
                                                        fetchApi = userReply.data.fetchApi,
                                                        formData = userReply.data.formData,
                                                        settingId = userReply.data.settingId,
                                                        userName = userReply.data.userName,
                                                        socketId = userReply.data.socketId
                                                    ),
                                                    teamList = userReply.data.teamList.map { team ->
                                                        ILiveChatTeam(
                                                            team = team.team,
                                                            translatedTeam = team.traslatedTeam
                                                                ?: "",
                                                            session = null,
                                                            teamList = null,
                                                            userProperty = null
                                                        )
                                                    },
                                                    userProperty = userReply.data.userProperty
                                                )
                                            )
                                        )
                                    }
                                }
                            },
                            optionsStyle = IOptionsStyle.BUTTON,
                            optionsHeader = null
                        )
                    )
                }
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
        return ILiveChatDataContainer(
            items = newChatItems.ifEmpty { null },
            data = liveChatData,
            status = liveChatStatus,
            feedbackRequest = liveChatFeedbackRequest,
            sendJoinMeHereRequest = sendJoinMeHereRequest
        )
    }

    private fun convertToLiveChatMediaItem(
        response: Any?,
        isUserResponse: Boolean,
        date: Date?,
        isHistoryItem: Boolean
    ): IChatItem? {
        val mediaResponse =
            Gson().fromJson(Gson().toJson(response), MediaResponse::class.java)
        mediaResponse.mediaUrl?.let { url ->
            if (ServerConstants.LIVE_CHAT_MEDIA_TYPE_IMAGE == mediaResponse.mediaType) {
                return if (isUserResponse) {
                    IImageWithTextItem(
                        imageUrl = mediaResponse?.mediaUrl,
                        imageText = mediaResponse?.mediaText,
                        date = date,
                        isBotItem = false
                    )
                } else {
                    IImageItem(
                        imageUrl = mediaResponse?.mediaUrl,
                        date = date,
                        isBotItem = true
                    )
                }
            } else if (ServerConstants.LIVE_CHAT_MEDIA_TYPE_VIDEO == mediaResponse.mediaType) {
                return IVideoItem(
                    title = null,
                    description = null,
                    videoText = mediaResponse.mediaText,
                    url = url,
                    fileSize = null,
                    date = date,
                    isBotItem = !isUserResponse,
                    isHistoryItem = false
                )
            } else if (ServerConstants.LIVE_CHAT_MEDIA_TYPE_FILE == mediaResponse.mediaType) {
                return IFileItem(
                    fileUrl = url,
                    fileSize = null,
                    fileSizeString = mediaResponse.mediaSize,
                    fileName = mediaResponse.mediaName ?: "",
                    fileText = mediaResponse.mediaText,
                    date = date,
                    isBotItem = !isUserResponse,
                    isHistoryItem = isHistoryItem,
                )
            }
        }
        return null
    }

    private fun convertToLiveChatStatusItem(
        response: Any?,
        responseType: String?
    ): IChatItem? {
        (response as? String)?.let { message ->
            return ILiveChatStatusItem(
                statusMessage = message,
                status = when (responseType) {
                    ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENTS_READY -> ILiveChatDataContainer.ILiveChatStatus.READY
                    ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENTS_QUEUED -> ILiveChatDataContainer.ILiveChatStatus.QUEUED
                    ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENT_CONNECTED -> ILiveChatDataContainer.ILiveChatStatus.CONNECTED
                    ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENTS_UNAVAILABLE -> ILiveChatDataContainer.ILiveChatStatus.UNAVAILABLE
                    ServerConstants.LIVE_CHAT_RESPONSE_TYPE_AGENT_DISCONNECTED -> ILiveChatDataContainer.ILiveChatStatus.DISCONNECTED
                    ServerConstants.LIVE_CHAT_RESPONSE_TYPE_CHAT_TRANSFER -> ILiveChatDataContainer.ILiveChatStatus.CONNECTED
                    else -> ILiveChatDataContainer.ILiveChatStatus.IDLE
                }
            )
        }
        return null
    }

    private fun convertToLiveChatData(
        userReply: AgentConnectResponse,
        agentDetails: AgentDetails? = null,
    ): ILiveChatData? {
        var liveChatData: ILiveChatData?
        userReply.let {
            liveChatData = ILiveChatData(
                agentEmailId = agentDetails?.agentEmailId ?: it.agentDetails?.agentEmailId
                ?: it.agentEmailId,
                agentIntId = agentDetails?.agentIntId ?: it.agentDetails?.agentIntId
                ?: it.agentIntId,
                agentIntName = agentDetails?.agentIntName ?: it.agentDetails?.agentIntName
                ?: it.agentName,
                agentLangCode = agentDetails?.agentLangCode ?: it.agentDetails?.agentLangCode
                ?: it.agentLangCode,
                agentName = agentDetails?.agentName ?: it.agentDetails?.agentName,
                agentSessionId = agentDetails?.agentSessionId ?: it.agentDetails?.agentSessionId,
                agentTeam = agentDetails?.agentTeam ?: it.agentDetails?.agentTeam,
                availableChats = agentDetails?.availableChats ?: it.agentDetails?.availableChats,
                botId = agentDetails?.botId ?: it.agentDetails?.botId ?: it.data?.botId
                ?: it.userIntId,
                channel = agentDetails?.channel ?: it.data?.channel ?: it.channel,
                chattingWith = agentDetails?.chattingWith ?: it.agentDetails?.chattingWith,
                currentChatCount = agentDetails?.currentChatCount
                    ?: it.agentDetails?.currentChatCount,
                imageName = agentDetails?.imageName ?: it.agentDetails?.imageName,
                isAvailable = agentDetails?.isAvailable ?: it.agentDetails?.isAvailable,
                maxChatCount = agentDetails?.maxChatCount ?: it.agentDetails?.maxChatCount,
                onBreak = agentDetails?.onBreak ?: it.agentDetails?.onBreak,
                role = agentDetails?.role ?: it.agentDetails?.role,
                roomId = agentDetails?.roomId ?: it.agentDetails?.roomId ?: it.roomId,
                shortMessage = agentDetails?.shortMessage ?: it.agentDetails?.shortMessage,
                userIntId = agentDetails?.userIntId ?: it.data?.userIntId ?: it.userIntId,
                userIntName = agentDetails?.userIntName ?: it.data?.userIntName ?: it.userIntName,
                userTeam = agentDetails?.agentTeam ?: it.agentDetails?.agentTeam,
                currentSessionId = it.data?.currentSessionId ?: it.currentSessionId,
                sessionId = it.data?.sessionId ?: it.sessionId,
                userId = agentDetails?.userId ?: it.data?.userId ?: it.userId,
                userName = it.data?.userName ?: it.userName,
                userLangCode = agentDetails?.userLangCode ?: it.userLangCode,
                userSelectedTeam = it.data?.userTeam ?: it.userTeam,
                resolved = it.resolved,
                unreadEmailCount = it.agentDetails?.unreadEmailCount,
                totalEmailCount = it.agentDetails?.totalEmailCount
            )
        }
        return liveChatData
    }

    fun convertCustomerInfoInputToChatItem(
        liveChatForm: ILiveChatForm
    ): IChatItem? {
        try {
            return IMessagesItem(
                messages = ArrayList<Any>().apply {
                    if (liveChatForm.displayField is List<*> && liveChatForm.displayField.isNotEmpty()) {
                        if (liveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_DEFAULT_FORM) {
                            liveChatForm.displayField.forEach { displayFieldAny ->
                                (displayFieldAny as? String)?.let { displayField ->
                                    val inputType = when (displayField) {
                                        ServerConstants.LIVE_CHAT_DISPLAY_FIELD_NAME -> ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_TEXT
                                        ServerConstants.LIVE_CHAT_DISPLAY_FIELD_EMAIL -> ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_EMAIL
                                        else -> ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_TEXT
                                    }
                                    add(
                                        IFormTextItem(
                                            textBoxTypeSelected = inputType,
                                            isRequired = true,
                                            labelName = displayField,
                                            prefixValue = AppConstants.FORM_ITEM_DEFAULT_PHONE_COUNTRY_CODE
                                        )
                                    )
                                }

                            }
                        } else if (liveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_ID
                            || liveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_PHONE_NUMBER
                            || liveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_EMAIL_ID
                        ) {
                            (liveChatForm.displayField.first() as? String)?.let { displayField ->
                                val inputType = when (liveChatForm.inputType) {
                                    ServerConstants.LIVE_CHAT_INPUT_TYPE_ID -> ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_TEXT
                                    ServerConstants.LIVE_CHAT_INPUT_TYPE_PHONE_NUMBER -> ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_PHONE_NUMBER
                                    ServerConstants.LIVE_CHAT_INPUT_TYPE_EMAIL_ID -> ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_EMAIL
                                    else -> ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_TEXT
                                }
                                add(
                                    IFormTextItem(
                                        textBoxTypeSelected = inputType,
                                        isRequired = true,
                                        labelName = displayField,
                                        prefixValue = AppConstants.FORM_ITEM_DEFAULT_PHONE_COUNTRY_CODE
                                    )
                                )
                            }
                        }
                    }
                },
                hideTime = true,
                messagesStyle = IMessagesItem.IMessagesStyle.FORM,
                extras = liveChatForm
            )
        } catch (e: Exception) {
            LogUtils.logException(e)
            return null
        }
    }

    fun convertFieldMessageToChatItem(liveChatForm: ILiveChatForm): IChatItem? {
        (liveChatForm.field as? String)?.let { field ->
            return IMessageItem(
                message = field,
                hideTime = true,
                isBotItem = true
            )
        }
        return null
    }
}