package ai.engagely.openbot.view.activities

import ai.engagely.openbot.R
import ai.engagely.openbot.databinding.ActivityBotChatBinding
import ai.engagely.openbot.model.constants.AppConstants
import ai.engagely.openbot.model.constants.ServerConstants
import ai.engagely.openbot.model.constants.ViewConstants
import ai.engagely.openbot.model.pojos.internal.botsettings.*
import ai.engagely.openbot.model.pojos.internal.chat.*
import ai.engagely.openbot.model.pojos.internal.common.IOpenUrlData
import ai.engagely.openbot.model.pojos.internal.common.ISendJourneyData
import ai.engagely.openbot.model.pojos.internal.common.IViewImageData
import ai.engagely.openbot.model.pojos.internal.faqautocomplete.IFaqAutoCompleteItem
import ai.engagely.openbot.model.pojos.internal.history.*
import ai.engagely.openbot.model.pojos.internal.livechat.*
import ai.engagely.openbot.model.pojos.internal.location.ILocation
import ai.engagely.openbot.model.utils.exts.*
import ai.engagely.openbot.model.utils.general.*
import ai.engagely.openbot.model.utils.helpers.BotViewHelper
import ai.engagely.openbot.model.utils.helpers.DebouncingTextChangeListener
import ai.engagely.openbot.model.utils.helpers.TakePictureWithUriReturnContract
import ai.engagely.openbot.model.utils.helpers.TakeVideoWithUriReturnContract
import ai.engagely.openbot.model.utils.validators.FormItemValidator
import ai.engagely.openbot.view.adapters.ChatItemsRecyclerViewAdapter
import ai.engagely.openbot.view.adapters.viewholders.TabularInfoWithMapHolder
import ai.engagely.openbot.view.customviews.FormDateVIew
import ai.engagely.openbot.view.customviews.FormFileUploadView
import ai.engagely.openbot.view.customviews.QuickMenuView
import ai.engagely.openbot.view.dialogfragments.BottomQuickMenuGridDialogFragment
import ai.engagely.openbot.view.dialogfragments.BottomQuickMenuListDialogFragment
import ai.engagely.openbot.view.dialogfragments.DatePickerFragment
import ai.engagely.openbot.viewmodel.BotChatViewModel
import ai.engagely.openbot.viewmodel.FaqAutoCompleteViewModel
import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.speech.RecognizerIntent
import android.view.*
import android.view.View.GONE
import android.view.View.VISIBLE
import android.widget.PopupMenu
import androidx.activity.result.contract.ActivityResultContracts
import androidx.core.content.ContextCompat
import androidx.core.view.GravityCompat
import androidx.drawerlayout.widget.DrawerLayout
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.lifecycleScope
import androidx.recyclerview.widget.LinearLayoutManager
import com.google.android.material.shape.CornerFamily
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.util.*


class BotChatActivity : BaseActivity(), View.OnClickListener,
    ChatItemsRecyclerViewAdapter.SpeechListener {

    private var updateListJob: Job? = null
    private lateinit var binding: ActivityBotChatBinding
    private var scrollToEndRunnable: Runnable? = null

    private val botChatViewModel by lazy {
        ViewModelProvider(this)[BotChatViewModel::class.java]
    }
    private val faqAutoCompleteViewModel by lazy {
        ViewModelProvider(this)[FaqAutoCompleteViewModel::class.java]
    }

    private val quickMenuView: QuickMenuView? by lazy {
        binding.nvQuickMenu.getHeaderView(0) as QuickMenuView
    }

    private var tempFormFileItem: IFormFileItem? = null
    private var tempFormFileUploadView: FormFileUploadView? = null
    private val filePickerResult =
        registerForActivityResult(ActivityResultContracts.StartActivityForResult()) {
            it?.let { activityResult ->
                when (activityResult.resultCode) {
                    Activity.RESULT_OK -> {
                        activityResult.data?.data?.let { fileUri ->
                            botChatViewModel.setSelectedFile(
                                fileUri,
                                contentResolver,
                                tempFormFileItem,
                                tempFormFileUploadView
                            )
                            tempFormFileItem = null
                            tempFormFileUploadView = null
                        }
                    }
                }
            }
        }

    private val speechToTextResult =
        registerForActivityResult(ActivityResultContracts.StartActivityForResult()) {
            it?.let { activityResult ->
                try {
                    when (activityResult.resultCode) {
                        Activity.RESULT_OK -> {
                            activityResult.data?.getStringArrayListExtra(RecognizerIntent.EXTRA_RESULTS)
                                ?.let { extraResults ->
                                    val recognizedText = extraResults[0]
                                    botChatViewModel.setSpeechText(recognizedText)
                                }
                        }
                    }
                } catch (e: Exception) {
                    LogUtils.logException(e)
                    showToast(getString(R.string.failed_to_detect_speech))
                }
            }
        }

    private val pickFileResult =
        registerForActivityResult(ActivityResultContracts.GetContent()) {
            it?.let { imageUri ->
                botChatViewModel.setLiveChatMedia(imageUri, contentResolver)
            }
        }

    private val pickImageCameraResult =
        registerForActivityResult(TakePictureWithUriReturnContract()) { (isSuccess, imageUri) ->
            if (isSuccess) {
                botChatViewModel.setLiveChatMedia(imageUri, contentResolver)
            }
        }

    private val pickVideoCameraResult =
        registerForActivityResult(TakeVideoWithUriReturnContract()) { (isSuccess, imageUri) ->
            if (isSuccess) {
                botChatViewModel.setLiveChatMedia(imageUri, contentResolver)
            }
        }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityBotChatBinding.inflate(layoutInflater)
        setContentView(binding.root)
        setupView()
        initChat()
    }

    private fun setupView() {
        processInitialData()
        setupChat()
        setupActionBar()
        setupTopJourney()
        setupThemeColor()
        setupCurrentLanguage()
        setupQuickMenu()
        setupNetworkStatus()
        setupBottomBar()
        setupLiveChatFeedback()
    }

    private fun setupLiveChatFeedback() {
        botChatViewModel.userFeedbackLiveData.observe(this) {
            if (it != null) {
                hideKeyboard()
            }
            binding.userFeedbackView.setData(it, this)
        }
    }

    private fun initChat() {
        botChatViewModel.initChat()
    }

    private fun setupNetworkStatus() {
        botChatViewModel.networkStateLiveData.observe(this) {
            binding.vNetworkStatus.setData(it)
        }
    }

    private fun setupQuickMenu() {
        botChatViewModel.quickMenuLiveData.observe(this) {
            if (it?.isNullOrEmpty() == true) {
                return@observe
            }
            quickMenuView?.setData(it, this)
        }
    }

    private fun setupCurrentLanguage() {
        binding.dropDownContainer.setOnClickListener { clickedView ->
            botChatViewModel.allLanguagesData?.let { languageData ->
                val popupMenu = PopupMenu(this, clickedView, Gravity.END)
                languageData.forEachIndexed { index, iLanguage ->
                    popupMenu.menu.add(Menu.NONE, index, index, iLanguage.languageName)
                }
                popupMenu.show()
                popupMenu.setOnMenuItemClickListener {
                    botChatViewModel.processLanguageSelection(it.itemId, it.title)
                    true
                }
            }
        }

        botChatViewModel.currentLanguageLiveData.observe(this) {
            binding.dropDownContainer.visibility = if (it == null) GONE else VISIBLE
            binding.dropDownTitle.text = it?.languageCode ?: ""
        }

        botChatViewModel.isLanguageLoadingLiveData.observe(this) {
            binding.rlMainLoader.visibility = if (it == true) VISIBLE else GONE
        }
    }

    private fun setupTopJourney() {
        botChatViewModel.greetingMessageLiveData.observe(this) {
            binding.vGreetingMessageView.setData(it)
        }
    }

    private fun setupThemeColor() {
        botChatViewModel.themeColorLiveData.observe(this) {
            val parsedColor =
                ViewUtils.parseColorSafely(it, ViewConstants.VIEW_DEFAULT_COLOR_STRING)

            //Set to Actionbar
            supportActionBar?.setBackgroundDrawable(ColorDrawable(parsedColor))
            window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
            window.statusBarColor = parsedColor

            //Set to Greeting message view
            binding.vGreetingMessageView.setBgStripColor(parsedColor)
        }

        botChatViewModel.bgColorLiveData.observe(this) {
            binding.mainBg.setBackgroundColor(
                ViewUtils.parseColorSafely(it, ViewConstants.VIEW_DEFAULT_COLOR_STRING)
            )
        }

        botChatViewModel.headerTextColorLiveData.observe(this) {
            val parsedColor =
                ViewUtils.parseColorSafely(it, ViewConstants.VIEW_DEFAULT_COLOR_STRING)
            binding.tbTitle.setTextColor(parsedColor)
            binding.dropDownTitle.setTextColor(parsedColor)
            binding.ivMenuDown.setColorFilter(parsedColor)
        }
    }

    private fun setupActionBar() {
        setSupportActionBar(binding.toolbar)

        //ActionBar title
        botChatViewModel.actionBarTitleLiveData.observe(this) {
            binding.tbTitle.text = it ?: ""
        }

        //Actionbar logo
        botChatViewModel.actionBarLogoLiveData.observe(this) {
            it?.let { bitmap ->
                val chatLogoSize = resources.getDimension(R.dimen.bot_logo_size_on_actionbar)
                ImageUtils.loadImage(this, bitmap, binding.botLogo)
                val cornerSize = (chatLogoSize * 0.5).toFloat()
                binding.botLogo.shapeAppearanceModel = binding.botLogo.shapeAppearanceModel
                    .toBuilder()
                    .setTopLeftCorner(CornerFamily.ROUNDED, cornerSize)
                    .setTopRightCorner(CornerFamily.ROUNDED, cornerSize)
                    .setBottomLeftCorner(CornerFamily.ROUNDED, cornerSize)
                    .setBottomRightCorner(CornerFamily.ROUNDED, cornerSize)
                    .build()
            }
        }
    }

    private fun setupChat() {
        //Setup chat recyclerview
        binding.rvChats.layoutManager =
            LinearLayoutManager(this, LinearLayoutManager.VERTICAL, false)
        binding.rvChats.setRecyclerListener { holder ->
            if (holder is TabularInfoWithMapHolder) {
                holder.clearView()
            }
        }
        createOrGetAdapter()

        //Observe and set chat data
        botChatViewModel.chatItemsLiveData.observe(this) { chatItemsContainer ->
            updateListJob?.cancel()
            updateListJob = lifecycleScope.launch {
                delay(300)
                createOrGetAdapter().updateChatItems(chatItemsContainer.chatItems)
                if (chatItemsContainer.scrollToEnd) {
                    scrollChatToEnd()
                }
            }
        }
        botChatViewModel.isHistoryLoadingLiveData.observe(this) { isLoading ->
            if (isLoading == true) {
                binding.etMessage.isEnabled = false
                binding.etMessage.hint = getString(R.string.loading)
            } else {
                binding.etMessage.isEnabled = true
                binding.etMessage.hint = getString(R.string.chat_bottom_bar_hint)
            }
        }

        //Observe and set colours
        botChatViewModel.botConfigLiveData.observe(this) { botChatWindowConfig ->

            val botButtonColor = ViewUtils.parseColorSafely(
                botChatWindowConfig.botButtonColor, ViewConstants.VIEW_DEFAULT_COLOR_STRING
            )
            val buttonChatBubbleColor = ViewUtils.parseColorSafely(
                botChatWindowConfig.botChatBoxColor, ViewConstants.VIEW_DEFAULT_COLOR_STRING
            )

            //Chat list configs
            createOrGetAdapter().updateBothChatConfig(
                botChatButtonColor = botButtonColor,
                botChatBubbleColor = buttonChatBubbleColor,
                botChatTextColor = ViewUtils.parseColorSafely(
                    botChatWindowConfig.botTextColor, ViewConstants.VIEW_DEFAULT_COLOR_STRING_DARK
                ),
                userChatBubbleColor = ViewUtils.parseColorSafely(
                    botChatWindowConfig.userChatBoxColor, ViewConstants.VIEW_DEFAULT_COLOR_STRING
                ),
                userChatTextColor = ViewUtils.parseColorSafely(
                    botChatWindowConfig.userTextColor, ViewConstants.VIEW_DEFAULT_COLOR_STRING_DARK
                ),
                themeColor = ViewUtils.parseColorSafely(
                    botChatWindowConfig.botThemColor, ViewConstants.VIEW_DEFAULT_COLOR_STRING_DARK
                ),
                linkColor = ViewUtils.parseColorSafely(
                    botChatWindowConfig.linkColor, ViewConstants.VIEW_DEFAULT_COLOR_STRING_DARK
                ),
                statusGreenColor = ContextCompat.getColor(this, R.color.status_green),
                statusBlueColor = ContextCompat.getColor(this, R.color.status_blue),
                statusRedColor = ContextCompat.getColor(this, R.color.status_red),
                speechListener = if (botChatWindowConfig?.enableSpeech == true) this else null
            )

            binding.btEndChat.setBackgroundColor(botButtonColor)

            binding.userFeedbackView.setColor(buttonChatBubbleColor)
        }
    }

    private fun processInitialData() {
        val botConfigData: IBotChatWindowConfig? =
            intent.getParcelableExtra(IBotChatWindowConfig.UNIQUE_NAME)
        botChatViewModel.setInitialData(botConfigData, intent.extras, lifecycle)
        faqAutoCompleteViewModel.init(botConfigData?.intId)
    }

    private fun scrollChatToEnd() {
        if (scrollToEndRunnable != null) {
            binding.rvChats.removeCallbacks(scrollToEndRunnable)
        }
        scrollToEndRunnable = Runnable {
            smoothScrollToEnd()
        }
        binding.rvChats.postDelayed(scrollToEndRunnable, 100)
    }

    private fun smoothScrollToEnd() {
        val targetPositionToScroll = createOrGetAdapter().itemCount - 1
        if (targetPositionToScroll > 0) {
            binding.rvChats.smoothScrollToPosition(targetPositionToScroll)
        }
    }

    private fun setupBottomBar() {
        setupBottomBarMenu()
        setupFaqAutoComplete()
    }

    private fun setupBottomBarMenu() {
        binding.bottomMenuBar.inflateMenu(R.menu.chat_bottom_bar_menu)
        binding.bottomMenuBar.setOnMenuItemClickListener {
            it?.let { menuItem ->
                when (menuItem.itemId) {
                    R.id.menu_send -> {
                        sendMessage()
                    }
                    R.id.menu_mic -> {
                        //Open speak now flow
                        hideKeyboard()
                        showSpeechToTextPopup()
                    }
                    R.id.menu_quick_menu -> {
                        //Open quick menu
                        hideKeyboard()
                        handleQuickMenuClick()
                    }
                }
            }
            return@setOnMenuItemClickListener false
        }
        setDrawerEnabled(false)

        binding.bottomMenuBar.menu?.findItem(R.id.menu_quick_menu)?.isVisible =
            botChatViewModel.hasQuickMenu()

        binding.bottomMenuBar.menu?.findItem(R.id.menu_mic)?.isVisible =
            botChatViewModel.isMicEnabled()

        botChatViewModel.liveChatStatusLiveData.observe(this) {
            binding.btEndChat.tag = it
            binding.btEndChat.setOnClickListener(this)

            binding.ibAttachment.setOnClickListener { clickedView ->
                openAttachmentPicker(clickedView)
            }
            if (ILiveChatDataContainer.ILiveChatStatus.CONNECTED == it
            ) {
                setLiveChatControls(
                    showLiveChatButtons = true,
                    showAttachment = true,
                    enableEditMessage = true,
                    requestEditMessageFocus = true
                )
            } else if (ILiveChatDataContainer.ILiveChatStatus.READY == it
                || ILiveChatDataContainer.ILiveChatStatus.QUEUED == it
            ) {
                setLiveChatControls(
                    showLiveChatButtons = true,
                    showAttachment = false,
                    enableEditMessage = false,
                    requestEditMessageFocus = false
                )
            } else {
                setLiveChatControls(
                    showLiveChatButtons = false,
                    showAttachment = false,
                    enableEditMessage = true,
                    requestEditMessageFocus = true
                )
            }
        }

        botChatViewModel.liveChatMediaLiveData.observe(this) {
            binding.uploadPhotoOverlayView.setData(
                it,
                ViewUtils.parseColorSafely(
                    botChatViewModel.themeColorLiveData.value,
                    ViewConstants.VIEW_DEFAULT_COLOR_STRING
                ),
                this
            )
        }
    }

    private fun openAttachmentPicker(clickedView: View) {
        val popupMenu = PopupMenu(this, clickedView, Gravity.START)
        popupMenu.menu.add(Menu.NONE, 0, 0, getString(R.string.capture_a_photo))
        popupMenu.menu.add(Menu.NONE, 1, 1, getString(R.string.record_a_video))
        popupMenu.menu.add(Menu.NONE, 2, 2, getString(R.string.select_a_file))
        popupMenu.show()
        popupMenu.setOnMenuItemClickListener {
            when (it?.itemId) {
                0 -> {
                    lifecycleScope.launch {
                        FileUtils.getTmpFileUri(this@BotChatActivity, "tmp_image_file", ".png")
                            .let { tempFileUri ->
                                pickImageCameraResult.launch(tempFileUri)
                            }
                    }
                }
                1 -> {
                    lifecycleScope.launch {
                        FileUtils.getTmpFileUri(this@BotChatActivity, "tmp_video_file", ".mp4")
                            .let { tempFileUri ->
                                pickVideoCameraResult.launch(tempFileUri)
                            }
                    }
                }
                2 -> {
                    pickFileResult.launch("*/*")
                }
            }
            true
        }
    }

    private fun setLiveChatControls(
        showLiveChatButtons: Boolean,
        showAttachment: Boolean,
        enableEditMessage: Boolean,
        requestEditMessageFocus: Boolean
    ) {
        binding.btEndChat.visibility = if (showLiveChatButtons) VISIBLE else GONE
        binding.ibAttachment.visibility = if (showAttachment) VISIBLE else GONE
        binding.bottomMenuBar.menu?.findItem(R.id.menu_quick_menu)?.isVisible =
            if (showLiveChatButtons) false else botChatViewModel.hasQuickMenu()
        binding.bottomMenuBar.menu?.findItem(R.id.menu_mic)?.isVisible =
            if (showLiveChatButtons) false else botChatViewModel.isMicEnabled()
        binding.etMessage.isEnabled = enableEditMessage

        if (requestEditMessageFocus) {
            binding.etMessage.requestFocus()
        }
    }

    private fun showSpeechToTextPopup() {
        try {
            botChatViewModel.requestSpeechStop {
                speechToTextResult.launch(Intent(RecognizerIntent.ACTION_RECOGNIZE_SPEECH).apply {
                    putExtra(
                        RecognizerIntent.EXTRA_LANGUAGE_MODEL,
                        RecognizerIntent.LANGUAGE_MODEL_FREE_FORM
                    )
                    putExtra(RecognizerIntent.EXTRA_LANGUAGE, Locale.getDefault())
                    putExtra(
                        RecognizerIntent.EXTRA_PROMPT,
                        getString(R.string.speak_now)
                    )
                })
            }
        } catch (e: ActivityNotFoundException) {
            e.printStackTrace()
            showToast(getString(R.string.your_device_does_not_support_stt))
        }
    }

    private fun setupFaqAutoComplete() {
        if (botChatViewModel.isAutoCompleteEnabled()) {
            binding.etMessage.addTextChangedListener(
                DebouncingTextChangeListener(
                    lifecycle = this.lifecycle,
                    delayInMillis = AppConstants.FAQ_AUTO_COMPLETE_DEBOUNCE_DELAY,
                    onDebouncingTextChange = {
                        faqAutoCompleteViewModel.processAutoComplete(it)
                    },
                    onTextChange = {
                        botChatViewModel.requestSpeechStop()
                    },
                )
            )
            faqAutoCompleteViewModel.faqAutoCompleteItemsLiveData.observe(this) {
                binding.vFaqAutoComplete.setData(it, this)
            }
            faqAutoCompleteViewModel.isLoadingLiveData.observe(this) {
                binding.vFaqAutoComplete.setIsLoading(it == true)
            }
        }
    }

    private fun handleQuickMenuClick() {
        if (botChatViewModel.hasQuickMenu()) {
            val quickMenuPosition = botChatViewModel.getQuickMenuPosition()
            when {
                IQuickMenuPosition.BOTTOM == quickMenuPosition -> {
                    botChatViewModel.quickMenuLiveData.value?.let { quickMenuList ->
                        if (quickMenuList.isNotEmpty()) {
                            BottomQuickMenuGridDialogFragment.getInstance(
                                quickMenuList
                            ).show(supportFragmentManager, "quick_menu_grid_dialog")
                        }
                    }
                }
                IQuickMenuPosition.LIST == quickMenuPosition -> {
                    botChatViewModel.quickMenuLiveData.value?.let { quickMenuList ->
                        if (quickMenuList.isNotEmpty()) {
                            BottomQuickMenuListDialogFragment.getInstance(
                                quickMenuList
                            ).show(supportFragmentManager, "quick_menu_list_dialog")
                        }
                    }
                }
                else -> {
                    binding.dlChat.openDrawer(GravityCompat.END)
                }
            }
        }
    }

    private fun sendMessage() {
        if (NetworkUtils.isInternetAvailable(this)) {
            scrollChatToEnd()
            botChatViewModel.sendMessageFromUserTextInput(binding.etMessage.consumeText())
        } else {
            showToast(resources.getString(R.string.no_internet))
        }
    }

    private fun createOrGetAdapter(): ChatItemsRecyclerViewAdapter {
        return if (binding.rvChats.adapter is ChatItemsRecyclerViewAdapter) {
            binding.rvChats.adapter as ChatItemsRecyclerViewAdapter
        } else {
            val adapter = ChatItemsRecyclerViewAdapter(this, getScreenWidth())
            binding.rvChats.adapter = adapter
            return adapter
        }
    }

    override fun onCreateOptionsMenu(menu: Menu?): Boolean {
        menuInflater.inflate(R.menu.bot_toolbar_menu, menu)

        botChatViewModel.headerTextColorLiveData.value?.let {
            val parsedColor =
                ViewUtils.parseColorSafely(it, ViewConstants.VIEW_DEFAULT_COLOR_STRING)
            menu?.findItem(R.id.menu_close)?.icon?.overrideWithCustomColor(parsedColor)
            menu?.findItem(R.id.menu_minimize)?.icon?.overrideWithCustomColor(parsedColor)
        }
        return true
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        when (item.itemId) {
            R.id.menu_minimize -> {
                onBackPressed()
                return true
            }
            R.id.menu_close -> {
                setBotConfigResult(BotViewHelper.RESULT_SESSION_UPDATE_REQUESTED)
                botChatViewModel.endLiveChat()
                finish()
                return true
            }
        }
        return super.onOptionsItemSelected(item)
    }

    private fun setBotConfigResult(resultCode: Int) {
        Intent().apply {
            putExtra(
                IBotChatWindowConfig.UNIQUE_NAME,
                botChatViewModel.getUpdatedBotConfigData()
            )
            setResult(resultCode, this)
        }
    }

    override fun onBackPressed() {
        setBotConfigResult(BotViewHelper.RESULT_BOT_CONFIG_UPDATED)
        super.onBackPressed()
    }

    private fun setDrawerEnabled(enabled: Boolean) {
        val lockMode: Int =
            if (enabled) DrawerLayout.LOCK_MODE_UNLOCKED else DrawerLayout.LOCK_MODE_LOCKED_CLOSED
        binding.dlChat.setDrawerLockMode(lockMode)
    }

    override fun onClick(view: View?) {
        botChatViewModel.requestSpeechStop {
            lifecycleScope.launch(Dispatchers.Main) {
                when (view?.tag) {
                    is IGreetingCarousalImage -> {
                        handleCarousalImageClick(view)
                    }
                    is ITopJr -> {
                        handleTopJourneyItemClick(view)
                    }
                    is IQuickMenuItem -> {
                        handleQuickMenuItemClick(view)
                    }
                    is IField -> {
                        handleChatButtonItemClick(view)
                    }
                    is IMessageItem -> {
                        botChatViewModel.processPendingMessage(view.tag)
                    }
                    is ICarousalButtonData -> {
                        handleCarousalButtonClick(view)
                    }
                    is IFormDateItem -> {
                        handleFormDateClick(view)
                    }
                    is IFormFileItem -> {
                        handleFormFileItemClick(view)
                    }
                    is IFormButtonItem -> {
                        handleFormSubmitButtonClick(view)
                    }
                    is IFileItem -> {
                        handleFileItemClick(view)
                    }
                    is IUrlsItem -> {
                        handleUrlItemClick(view)
                    }
                    is IVideoItem -> {
                        handleVideoItemClick(view)
                    }
                    is IYouTubeVideoItem -> {
                        handleYouTubeVideoItemClick(view)
                    }
                    is IChatConflictChildItem -> {
                        handleFaqConflictItemClick(view)
                    }
                    is IFaqAutoCompleteItem -> {
                        handleFaqAutoCompleteItemClick(view)
                    }
                    is ITabularDropDownItem -> {
                        handleTabularInfoChildDropDownSelection(view)
                    }
                    is IAdvancedUrlItem -> {
                        handleAdvancedUrlClick(view)
                    }
                    is String -> {
                        handleActionClick(view)
                    }
                    is ILiveChatForm -> {
                        handleLiveChatClick(view)
                    }
                    is IMessagesItem -> {
                        requestForLocation(view)
                    }
                    is ILiveChatTeam -> {
                        handleLiveChatTeamClick(view)
                    }
                    is ILiveChatDataContainer.ILiveChatStatus -> {
                        handleEndChatClick(view)
                    }
                    is ILiveChatMediaData -> {
                        handleLiveChatPhotoSendClick(view)
                    }
                    is IViewImageData -> {
                        handleViewImageData(view)
                    }
                    is ILiveChatFeedbackRequest -> {
                        handleLiveChatFeedbackSubmitClick(view)
                    }
                    is ISendJourneyData -> {
                        handleSendJourneyDataClick(view)
                    }
                    is IOpenUrlData -> {
                        handleOpenUrlClick(view)
                    }
                }
            }
        }
    }

    private fun handleSendJourneyDataClick(view: View) {
        (view.tag as? ISendJourneyData)?.let {
            botChatViewModel.sendCarousalJourneyMessage(
                message = it.journeyName,
                isFromProcessTree = true
            )
        }
    }

    private fun handleOpenUrlClick(view: View) {
        (view.tag as? IOpenUrlData)?.let {
            openBrowser(it.url, getString(R.string.browser_open_failed))
        }
    }

    private fun handleLiveChatFeedbackSubmitClick(view: View) {
        (view.tag as? ILiveChatFeedbackRequest).let {
            botChatViewModel.handleLiveChatFeedbackSubmitClick(it)
        }
    }

    private fun handleViewImageData(view: View) {
        (view.tag as? IViewImageData)?.let {
            binding.showPhotoOverlayView.setData(it)
        }
    }

    private fun handleLiveChatPhotoSendClick(view: View) {
        (view.tag as? ILiveChatMediaData)?.let {
            hideKeyboard()
            botChatViewModel.sendLiveChatPhotoAndMessage(
                it,
                binding.uploadPhotoOverlayView.getMessage()
            )
        }
    }

    private fun handleEndChatClick(view: View) {
        (view.tag as? ILiveChatDataContainer.ILiveChatStatus)?.let {
            botChatViewModel.endLiveChat()
        }
    }

    private fun handleLiveChatTeamClick(view: View) {
        (view.tag as? ILiveChatTeam)?.let { iLiveChatTeam ->
            botChatViewModel.processLiveChatTeamClick(
                iLiveChatTeam,
                getScreenWidth(),
                getScreenHeight(),
                isLandscapeOrientation()
            )
        }
    }

    private fun handleMessagesActionClick(view: View, location: ILocation?) {
        (view.tag as? IMessagesItem)?.let { iMessagesItem ->
            botChatViewModel.processMessagesActionClick(
                iMessagesItem, getScreenWidth(),
                getScreenHeight(),
                isLandscapeOrientation(),
                location
            )
        }
    }

    private fun handleLiveChatClick(view: View) {
        (view.tag as? ILiveChatForm)?.let { liveChatForm ->
            if (ServerConstants.LIVE_CHAT_INPUT_TYPE_NO_FORM == liveChatForm.inputType) {
                requestForLocation(view)
            } else {
                processLiveChatTrueClick(view, null)
            }
        }
    }

    private fun processLiveChatTrueClick(view: View, location: ILocation?) {
        (view.tag as? ILiveChatForm)?.let { liveChatForm ->
            (view.getTag(R.id.tag_item_uuid) as? String)?.let { uuid ->
                botChatViewModel.processLiveChatTrueClick(
                    uuid,
                    liveChatForm,
                    getScreenWidth(),
                    getScreenHeight(),
                    isLandscapeOrientation(),
                    location
                )
            }
        }
    }

    private fun handleActionClick(view: View) {
        (view.tag as? String)?.let {
            if (it == AppConstants.ACTION_LIVE_CHAT_BUTTON_FALSE) {
                (view.getTag(R.id.tag_item_uuid) as? String)?.let { uuid ->
                    botChatViewModel.processLiveChatFalseClick(uuid)
                }
            }
        }
    }

    private fun handleAdvancedUrlClick(view: View) {
        (view.tag as? IAdvancedUrlItem)?.let {
            openBrowser(it.urlLink, getString(R.string.failed_to_open_link))
        }
    }

    private fun handleTabularInfoChildDropDownSelection(view: View) {
        (view.tag as? ITabularDropDownItem)?.let {
            val tabularInfoItem = view.getTag(R.id.tabular_item) as? ITabularInfoItem
            val tabularInfoItemChildItem =
                view.getTag(R.id.tabular_item_child) as? ITabularInfoChildItem
            botChatViewModel.processTabularItemChildDropDownSelection(
                tabularInfoItem,
                tabularInfoItemChildItem
            )
        }
    }

    private fun handleFaqAutoCompleteItemClick(view: View) {
        (view.tag as? IFaqAutoCompleteItem)?.let {
            val sent = botChatViewModel.sendFaqAutoCompleteItem(it)
            if (sent) {
                binding.etMessage.consumeText()
            }
        }
    }

    private fun handleFaqConflictItemClick(view: View) {
        val iChatConflictItem = view.tag as? IChatConflictChildItem
        iChatConflictItem?.let {
            botChatViewModel.sendFaqConflictChildClick(it)
        }
    }

    private fun handleYouTubeVideoItemClick(view: View) {
        val iYouTubeVideoItem = view.tag as IYouTubeVideoItem
        UrlUtils.getVideoId(iYouTubeVideoItem.url)?.let { videoId ->
            Intent(this, YouTubePlayerActivity::class.java).apply {
                putExtra(YouTubePlayerActivity.VIDEO_ID_TAG, videoId)
                putExtra(YouTubePlayerActivity.VIDEO_TITLE_TAG, iYouTubeVideoItem.title)
                putExtra(
                    YouTubePlayerActivity.VIDEO_SUBTITLE_TAG,
                    iYouTubeVideoItem.description
                )
            }.also {
                startActivity(it)
            }
        }
    }

    private fun handleVideoItemClick(view: View) {
        val iVideoItem = view.tag as IVideoItem
        Intent(this, ExoPlayerActivity::class.java).apply {
            putExtra(ExoPlayerActivity.VIDEO_URL_TAG, iVideoItem.url)
            putExtra(ExoPlayerActivity.VIDEO_TITLE_TAG, iVideoItem.title)
            putExtra(ExoPlayerActivity.VIDEO_SUBTITLE_TAG, iVideoItem.description)
        }.also {
            startActivity(it)
        }
    }

    private fun handleUrlItemClick(view: View) {
        view.getTag(R.id.tag_item_url)?.also { url ->
            if (url is String) {
                openBrowser(url, getString(R.string.failed_to_open_link))
            }
        }
    }

    private fun handleFileItemClick(view: View) {
        val fileItem = view.tag as IFileItem
        //Assuming that all the files will be non media files.
        openBrowser(fileItem.fileUrl, getString(R.string.failed_to_open_file))
    }

    private fun handleFormSubmitButtonClick(view: View) {
        val buttonItem = view.tag as IFormButtonItem
        val chatFormItem = view.getTag(R.id.tag_form_item) as IChatFormItem
        val formContainsValidData = FormItemValidator.validate(this, chatFormItem)
        buttonItem.isLoading = formContainsValidData
        createOrGetAdapter().notifyDataSetChanged()

        if (formContainsValidData) {
            if (chatFormItem.submitType == IChatFormItem.SubmitType.SOCKET && buttonItem.extras is ILiveChatForm) {
                requestForLocation(view)
            } else {
                botChatViewModel.submitForm(
                    chatFormItem,
                    buttonItem,
                    createOrGetAdapter(),
                )
            }
        }
    }

    private fun handleFormFileItemClick(view: View) {
        val formFileItem = view.tag as IFormFileItem
        if (view.id == R.id.tvFileName) {
            formFileItem.fileUri?.let { fileUri ->
                openDeviceFile(fileUri)
                return
            }
        }

        val formFileView = view.getTag(R.id.tag_form_view) as FormFileUploadView
        tempFormFileUploadView = formFileView
        tempFormFileItem = formFileItem

        val fileChooserIntent = Intent(Intent.ACTION_OPEN_DOCUMENT).apply {
            type = "*/*"
            addCategory(Intent.CATEGORY_OPENABLE)
        }
        filePickerResult.launch(fileChooserIntent)
    }

    private fun handleFormDateClick(view: View) {
        val dateItem = view.tag as IFormDateItem
        val dateView = view.getTag(R.id.tag_form_date) as FormDateVIew

        DatePickerFragment(
            selectedDate = dateItem.selectedDate,
            onDateSelected = { date ->
                dateItem.selectedDate = date
                FormItemValidator.validateFormItem(dateItem, this)
                dateView.setData(dateItem, this)
            }).show(supportFragmentManager, "date_picker_tag")
    }

    private fun handleCarousalButtonClick(view: View) {
        val buttonData = view.tag as ICarousalButtonData
        if (ServerConstants.DIALOGUE_BUTTON_TYPE_URL == buttonData.buttonType && buttonData.data != null) {
            openBrowser(buttonData.data, getString(R.string.browser_open_failed))
        } else {
            botChatViewModel.processCarousalButtonClick(buttonData)
        }
    }

    private fun handleChatButtonItemClick(view: View) {
        val iField = view.tag as IField
        botChatViewModel.sendButtonClick(iField)
    }

    private fun handleQuickMenuItemClick(view: View) {
        closeQuickMenuIfOpen()
        val quickMenuItem = view.tag as IQuickMenuItem
        botChatViewModel.sendQuickMenuMessage(quickMenuItem)
    }

    private fun handleTopJourneyItemClick(view: View) {
        val topJr = view.tag as ITopJr
        botChatViewModel.sendTopJourneyMessage(topJr)
    }

    private fun handleCarousalImageClick(view: View) {
        val clickedImageData = view.tag as IGreetingCarousalImage
        clickedImageData.clickData.obtainNonBlankOrNull()?.let {
            if (clickedImageData.clickEnabled) {
                if (ServerConstants.GREETING_CAROUSAL_CLICK_TYPE_JOURNEY == clickedImageData.clickType) {
                    botChatViewModel.sendCarousalJourneyMessage(it)
                } else if (ServerConstants.GREETING_CAROUSAL_CLICK_TYPE_URL == clickedImageData.clickType) {
                    openBrowser(it, getString(R.string.browser_open_failed))
                }
            }
        }
    }

    private fun closeQuickMenuIfOpen() {
        if (binding.dlChat.isDrawerOpen(GravityCompat.END)) {
            binding.dlChat.closeDrawer(GravityCompat.END)
        }
    }

    override fun onSpeechRequested(chatItem: IChatItem) {
        botChatViewModel.requestSpeechStart(chatItem, true)
    }

    override fun onLocationFetched(
        latitude: Double,
        longitude: Double,
        locationDependantObject: Any?
    ) {
        processLocationDependantObject(locationDependantObject, ILocation(latitude, longitude))
    }

    override fun onLocationNotFetched(locationDependantObject: Any?) {
        processLocationDependantObject(locationDependantObject, null)
    }

    private fun processLocationDependantObject(
        locationDependantObject: Any?,
        location: ILocation?
    ) {
        (locationDependantObject as? View)?.let { view ->
            handleLiveChatFormSubmission(view, location)
            handleMessagesActionClick(view, location)
            processLiveChatTrueClick(view, location)
        }
    }

    private fun handleLiveChatFormSubmission(
        view: View,
        location: ILocation?
    ) {
        (view.tag as? IFormButtonItem)?.let {
            val buttonItem = view.tag as IFormButtonItem
            val chatFormItem = view.getTag(R.id.tag_form_item) as IChatFormItem
            botChatViewModel.submitLiveChatForm(
                chatFormItem,
                buttonItem,
                createOrGetAdapter(),
                getScreenWidth(),
                getScreenHeight(),
                isLandscapeOrientation(),
                location
            )
        }
    }
}