package ai.engagely.openbot.viewmodel

import ai.engagely.openbot.model.network.VideoCache
import android.app.Application
import android.os.Build
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.ProcessLifecycleOwner
import com.google.android.exoplayer2.ExoPlayer
import com.google.android.exoplayer2.MediaItem
import com.google.android.exoplayer2.source.ProgressiveMediaSource
import com.google.android.exoplayer2.ui.PlayerView
import com.google.android.exoplayer2.upstream.DataSource
import com.google.android.exoplayer2.upstream.DefaultHttpDataSource
import com.google.android.exoplayer2.upstream.HttpDataSource
import com.google.android.exoplayer2.upstream.cache.CacheDataSource

class VideoPlayerViewModel(application: Application) : AndroidViewModel(application),
    DefaultLifecycleObserver {

    private var player: ExoPlayer? = null
    private var contentPosition = 0L
    private var playWhenReady = true

    private var url: String? = null

    var httpDataSourceFactory: HttpDataSource.Factory =
        DefaultHttpDataSource.Factory().setAllowCrossProtocolRedirects(true)
    var cacheDataSourceFactory: DataSource.Factory = CacheDataSource.Factory()
        .setCache(VideoCache.getInstance(application).getSimpleCache())
        .setUpstreamDataSourceFactory(httpDataSourceFactory)

    init {
        ProcessLifecycleOwner.get().lifecycle.addObserver(this)
    }

    fun updateInitialData(contentPosition: Long, playWhenReady: Boolean, url: String) {
        this.contentPosition = contentPosition
        this.playWhenReady = playWhenReady
        this.url = url
    }

    private fun initializePlayer() {
        ExoPlayer.Builder(getApplication())
            .build()
            .also { exoPlayer ->
                player = exoPlayer
            }
    }

    private fun setMediaItemAndPrepare(exoPlayer: ExoPlayer) {
        url?.let {
            val mediaItem = MediaItem.fromUri(it)
            val mediaSource =
                ProgressiveMediaSource.Factory(cacheDataSourceFactory).createMediaSource(mediaItem)
            exoPlayer.setMediaSource(mediaSource)

            exoPlayer.playWhenReady = playWhenReady
            exoPlayer.seekTo(0, contentPosition)
            exoPlayer.prepare()
        }
    }

    override fun onStart(owner: LifecycleOwner) {
        super.onStart(owner)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            initializePlayer()
        }
    }

    override fun onResume(owner: LifecycleOwner) {
        super.onResume(owner)
        if ((Build.VERSION.SDK_INT < Build.VERSION_CODES.N || player == null)) {
            initializePlayer()
        }
    }

    override fun onPause(owner: LifecycleOwner) {
        super.onPause(owner)
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.N) {
            releasePlayer()
        }
    }

    override fun onStop(owner: LifecycleOwner) {
        super.onStop(owner)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            releasePlayer()
        }
    }

    private fun releasePlayer() {
        player ?: return

        contentPosition = player?.contentPosition ?: 0
        playWhenReady = player?.playWhenReady ?: true
        player?.release()
        player = null
    }

    override fun onCleared() {
        super.onCleared()
        releasePlayer()
        ProcessLifecycleOwner.get().lifecycle.removeObserver(this)
    }

    fun startPlaying(playerView: PlayerView?, url: String?) {
        playerView ?: url ?: return

        playerView?.player = player
        this.url = url
        player?.let {
            setMediaItemAndPrepare(it)
        }
    }

    fun stopPlayingAndSaveState() {
        player?.pause()
    }

    fun getCurrentPosition(): Long = player?.contentPosition ?: 0

    fun getUrl(): String? = url

    fun getPlayWhenReady(): Boolean = player?.isPlaying ?: false

    fun resumePlaying(playerView: PlayerView?) {
        startPlaying(playerView, url)
    }

    companion object {
        const val CACHE_SIZE: Long = 100 * 1024 * 1024 //100MB
    }
}