package ai.engagely.openbot.view.activities

import ai.engagely.openbot.R
import ai.engagely.openbot.model.pojos.internal.location.ILocationFetchStatus
import ai.engagely.openbot.model.utils.exts.showToast
import ai.engagely.openbot.model.utils.general.LogUtils
import android.Manifest
import android.annotation.SuppressLint
import android.content.Intent
import android.content.IntentSender
import android.content.pm.ActivityInfo
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.net.Uri
import android.os.Looper
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.WindowInsetsControllerCompat
import com.google.android.gms.common.api.ResolvableApiException
import com.google.android.gms.location.*
import com.google.android.gms.tasks.Task

abstract class BaseActivity : AppCompatActivity() {

    private var fusedLocationClient: FusedLocationProviderClient? = null
    private var locationCallback: LocationCallback? = null
    private var locationDependantObject: Any? = null

    /**
     * To open a browser
     */
    fun openBrowser(url: String, errorMessage: String) {
        try {
            startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(url)))
        } catch (e: Exception) {
            LogUtils.logException(e)
            showToast(errorMessage)
        }
    }

    /**
     * To open the file selected from device using Uri
     */
    fun openDeviceFile(fileUri: Uri) {
        try {
            Intent(Intent.ACTION_VIEW).apply {
                data = fileUri
                flags =
                    Intent.FLAG_GRANT_READ_URI_PERMISSION or Intent.FLAG_GRANT_WRITE_URI_PERMISSION
            }.also {
                startActivity(it)
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
            showToast(getString(R.string.failed_to_open_file))
        }
    }

    protected fun hideSystemBars() {
        val windowInsetsController =
            ViewCompat.getWindowInsetsController(window.decorView) ?: return
        // Configure the behavior of the hidden system bars
        windowInsetsController.systemBarsBehavior =
            WindowInsetsControllerCompat.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
        // Hide both the status bar and the navigation bar
        windowInsetsController.hide(WindowInsetsCompat.Type.systemBars())
    }

    protected fun unhideSystemBars() {
        val windowInsetsController =
            ViewCompat.getWindowInsetsController(window.decorView) ?: return
        // Configure the behavior of the hidden system bars
        windowInsetsController.systemBarsBehavior =
            WindowInsetsControllerCompat.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
        // Show both the status bar and the navigation bar
        windowInsetsController.show(WindowInsetsCompat.Type.systemBars())
    }

    protected fun setLandscapeOrientation() {
        if (resources.configuration.orientation != Configuration.ORIENTATION_LANDSCAPE) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
        }
    }

    protected fun setPortraitOrientation() {
        if (resources.configuration.orientation != Configuration.ORIENTATION_PORTRAIT) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        }
    }

    protected fun isLandscapeOrientation(): Boolean {
        return resources.configuration.orientation == Configuration.ORIENTATION_LANDSCAPE
    }

    private fun checkForLocationServices() {
        val locationRequest = createLocationRequest()

        val builder = LocationSettingsRequest.Builder()
            .addLocationRequest(locationRequest)

        val client: SettingsClient = LocationServices.getSettingsClient(this)
        val task: Task<LocationSettingsResponse> = client.checkLocationSettings(builder.build())
        task.addOnSuccessListener {
            // All location settings are satisfied.
            checkForLocationPermission()
        }

        task.addOnFailureListener { exception ->
            if (exception is ResolvableApiException) {
                // Location settings are not satisfied
                try {
                    // Show the dialog by calling startResolutionForResult(),
                    // and check the result in onActivityResult().
                    exception.startResolutionForResult(
                        this,
                        REQUEST_CHECK_SETTINGS
                    )
                } catch (sendEx: IntentSender.SendIntentException) {
                    onLocationServicesResult(ILocationFetchStatus.SERVICE_NOT_ENABLED)
                }
            } else {
                onLocationServicesResult(ILocationFetchStatus.SERVICE_NOT_ENABLED)
            }
        }
    }

    private fun createLocationRequest(): LocationRequest {
        val locationRequest = LocationRequest.create().apply {
            interval = 3000
            fastestInterval = 2000
            priority = LocationRequest.PRIORITY_HIGH_ACCURACY
        }
        return locationRequest
    }

    private fun checkForLocationPermission() {
        if ((ContextCompat.checkSelfPermission(
                this, Manifest.permission.ACCESS_COARSE_LOCATION
            )
                    != PackageManager.PERMISSION_GRANTED) ||
            ContextCompat.checkSelfPermission(
                this, Manifest.permission.ACCESS_FINE_LOCATION
            )
            != PackageManager.PERMISSION_GRANTED
        ) {
            ActivityCompat.requestPermissions(
                this, arrayOf(
                    Manifest.permission.ACCESS_COARSE_LOCATION,
                    Manifest.permission.ACCESS_FINE_LOCATION
                ),
                PERMISSIONS_ACCESS_COARSE_LOCATION
            )
        } else {
            onLocationServicesResult(ILocationFetchStatus.PERMISSION_GRANTED)
        }
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        when (requestCode) {
            REQUEST_CHECK_SETTINGS -> if (resultCode == RESULT_OK) {
                checkForLocationPermission()
            } else {
                onLocationServicesResult(ILocationFetchStatus.SERVICE_NOT_ENABLED)
            }
        }
    }

    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        when (requestCode) {
            PERMISSIONS_ACCESS_COARSE_LOCATION -> if (grantResults.isNotEmpty()) {
                if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    onLocationServicesResult(ILocationFetchStatus.PERMISSION_GRANTED)
                } else {
                    onLocationServicesResult(ILocationFetchStatus.PERMISSION_DENIED)
                }
            } else {
                onLocationServicesResult(ILocationFetchStatus.PERMISSION_DENIED)
            }
        }
    }

    private fun onLocationServicesResult(iLocationFetchStatus: ILocationFetchStatus) {
        when (iLocationFetchStatus) {
            ILocationFetchStatus.PERMISSION_GRANTED -> getLastKnownLocation()
            ILocationFetchStatus.PERMISSION_DENIED -> showToast(getString(R.string.location_permission_denied))
            ILocationFetchStatus.SERVICE_NOT_ENABLED -> showToast(getString(R.string.location_service_not_enabled))
            ILocationFetchStatus.FETCHING_FAILED -> showToast(getString(R.string.current_location_unavailable))
        }
        if (iLocationFetchStatus != ILocationFetchStatus.PERMISSION_GRANTED) {
            onLocationNotFetched(locationDependantObject)
        }
    }

    @SuppressLint("MissingPermission")
    private fun getLastKnownLocation() {
        fusedLocationClient = LocationServices.getFusedLocationProviderClient(this)
        fusedLocationClient?.lastLocation?.addOnCompleteListener {
            if (it.result == null) {
                //Uh oh! :( No last known location
                fetchCurrentLocation()
            } else {
                onLocationFetched(it.result.latitude, it.result.longitude, locationDependantObject)
            }
        }
    }

    /**
     * This will be called when location fetch is successfully completed. Subclasses can override
     * this method to get the location fetch callback.
     */
    protected open fun onLocationFetched(
        latitude: Double,
        longitude: Double,
        locationDependantObject: Any?
    ) {
    }

    /**
     * This will be called when location fetch is failed. Subclasses can override this method
     * to get the location fetch callback.
     */
    protected open fun onLocationNotFetched(locationDependantObject: Any?) {}

    @SuppressLint("MissingPermission")
    private fun fetchCurrentLocation() {
        locationCallback = object : LocationCallback() {
            override fun onLocationResult(locationResult: LocationResult) {
                stopLocationUpdates()
                if (locationResult.locations.isNotEmpty()) {
                    locationResult.locations.first()?.let { location ->
                        onLocationFetched(
                            location.latitude,
                            location.longitude,
                            locationDependantObject
                        )
                    } ?: onLocationServicesResult(ILocationFetchStatus.FETCHING_FAILED)
                } else {
                    onLocationServicesResult(ILocationFetchStatus.FETCHING_FAILED)
                }
            }
        }.also {
            fusedLocationClient?.requestLocationUpdates(
                createLocationRequest(),
                it,
                Looper.getMainLooper()
            )
        }
    }

    private fun stopLocationUpdates() {
        fusedLocationClient?.let { fLocationClient ->
            locationCallback?.let { lCallback ->
                fLocationClient.removeLocationUpdates(lCallback)
            }
        }
        fusedLocationClient = null
        locationCallback = null
    }

    protected fun requestForLocation(dependantObject: Any?) {
        this.locationDependantObject = dependantObject
        checkForLocationServices()
    }

    companion object {
        private const val REQUEST_CHECK_SETTINGS = 10
        private const val PERMISSIONS_ACCESS_COARSE_LOCATION = 11
    }
}