package ai.engagely.openbot.viewmodel

import ai.engagely.openbot.model.localstorage.SimpleKeyValueStorage
import ai.engagely.openbot.model.localstorage.impl.PreferenceDataStore
import ai.engagely.openbot.model.network.ApiConstants
import ai.engagely.openbot.model.repositories.ChatRepository
import ai.engagely.openbot.model.repositories.LiveChatRepository
import ai.engagely.openbot.model.repositories.impl.AppChatRepository
import ai.engagely.openbot.model.repositories.impl.AppLiveChatRepository
import ai.engagely.openbot.model.utils.converters.ChatResponseConverter
import ai.engagely.openbot.model.utils.general.LogUtils
import android.app.Application
import androidx.lifecycle.viewModelScope
import io.socket.client.IO
import io.socket.client.Socket
import io.socket.emitter.Emitter
import io.socket.engineio.client.transports.WebSocket
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withTimeoutOrNull
import java.net.URI
import kotlin.coroutines.resume

open class SocketViewModel(application: Application) : NetworkStateAwareViewModel(application) {

    private val simpleKeyValueStorage: SimpleKeyValueStorage by lazy {
        PreferenceDataStore(application)
    }

    protected val chatRepository: ChatRepository by lazy {
        AppChatRepository(Dispatchers.IO, socket)
    }

    protected val liveChatRepository: LiveChatRepository by lazy {
        AppLiveChatRepository(
            Dispatchers.Default, liveChatSocket, simpleKeyValueStorage,
            ChatResponseConverter()
        )
    }

    private val socket: Socket
    private val liveChatSocket: Socket

    init {
        val (uri: URI, options) = createSocketOptions(ApiConstants.SOCKET_URL)
        socket = IO.socket(uri, options)

        val (liveChatUri: URI, liveChatOptions) = createSocketOptions(
            ApiConstants.LIVE_CHAT_SOCKET_URL,
            ApiConstants.LIVE_CHAT_SOCKET_PATH
        )
        liveChatSocket = IO.socket(liveChatUri, liveChatOptions)
    }

    private fun createSocketOptions(url: String, path: String? = null): Pair<URI, IO.Options> {
        val uri: URI = URI.create(url)
        val options = IO.Options()
        options.transports = arrayOf(WebSocket.NAME)
        options.secure = true
        options.forceNew = true
        options.path = path
        return Pair(uri, options)
    }

    protected fun initSocket() {
        socket.on(Socket.EVENT_ERROR) {
            LogUtils.log("Socket event error")
        }
        socket.on(Socket.EVENT_DISCONNECT) {
            LogUtils.log("Socket disconnected")
        }
        socket.on(Socket.EVENT_CONNECT) {
            LogUtils.log("Socket connected - Id = ${socket.id()}")
            storeSocketId(socket.id())
        }
        socket.on(Socket.EVENT_CONNECTING) {
            LogUtils.log("Socket connecting")
        }
        socket.on(Socket.EVENT_CONNECT_ERROR) {
            LogUtils.log("Socket connect error")
        }
        socket.on(Socket.EVENT_CONNECT_TIMEOUT) {
            LogUtils.log("Socket connect timeout")
        }
        socket.on(Socket.EVENT_RECONNECT) {
            LogUtils.log("Socket reconnect")
        }
        socket.on(Socket.EVENT_RECONNECTING) {
            LogUtils.log("Socket reconnecting")
        }
        socket.on(Socket.EVENT_RECONNECT_ATTEMPT) {
            LogUtils.log("Socket reconnect attempt")
        }
        socket.on(Socket.EVENT_RECONNECT_ERROR) {
            LogUtils.log("Socket reconnect error")
        }
        socket.on(Socket.EVENT_RECONNECT_FAILED) {
            LogUtils.log("Socket reconnect failed")
        }

        liveChatSocket.on(Socket.EVENT_ERROR) {
            LogUtils.log("LCSocket event error")
        }
        liveChatSocket.on(Socket.EVENT_DISCONNECT) {
            LogUtils.log("LCSocket disconnected")
        }
        liveChatSocket.on(Socket.EVENT_CONNECT) {
            LogUtils.log("LCSocket connected - Id = ${liveChatSocket.id()}")
            storeLiveChatSocketId(liveChatSocket.id())
        }
        liveChatSocket.on(Socket.EVENT_CONNECTING) {
            LogUtils.log("LCSocket connecting")
        }
        liveChatSocket.on(Socket.EVENT_CONNECT_ERROR) {
            LogUtils.log("LCSocket connect error")
        }
        liveChatSocket.on(Socket.EVENT_CONNECT_TIMEOUT) {
            LogUtils.log("LCSocket connect timeout")
        }
        liveChatSocket.on(Socket.EVENT_RECONNECT) {
            LogUtils.log("LCSocket reconnect")
        }
        liveChatSocket.on(Socket.EVENT_RECONNECTING) {
            LogUtils.log("LCSocket reconnecting")
        }
        liveChatSocket.on(Socket.EVENT_RECONNECT_ATTEMPT) {
            LogUtils.log("LCSocket reconnect attempt")
        }
        liveChatSocket.on(Socket.EVENT_RECONNECT_ERROR) {
            LogUtils.log("LCSocket reconnect error")
        }
        liveChatSocket.on(Socket.EVENT_RECONNECT_FAILED) {
            LogUtils.log("LCSocket reconnect failed")
        }

        socket.on(ApiConstants.CHANNEL_CHAT) { messageArray ->
            acceptReceivedMessage(ApiConstants.CHANNEL_CHAT, messageArray)
        }

        socket.on(ApiConstants.CHANNEL_TAB_INFO) { messageArray ->
            acceptReceivedMessage(ApiConstants.CHANNEL_TAB_INFO, messageArray)
        }
        liveChatSocket.on(ApiConstants.CHANNEL_USER_REPLY) { messageArray ->
            acceptReceivedMessage(ApiConstants.CHANNEL_USER_REPLY, messageArray)
        }

        liveChatSocket.on(ApiConstants.CHANNEL_LIVECHAT_USER) { messageArray ->
            acceptReceivedMessage(ApiConstants.CHANNEL_LIVECHAT_USER, messageArray)
        }

        liveChatSocket.on(ApiConstants.CHANNEL_LIVE_USER_DISCONNECT_STATUS) { messageArray ->
            acceptReceivedMessage(ApiConstants.CHANNEL_LIVE_USER_DISCONNECT_STATUS, messageArray)
        }
    }

    private fun acceptReceivedMessage(channelName: String, messageArray: Array<Any>) {
        LogUtils.log("Received in $channelName channel ${messageArray.firstOrNull()}")
        messageArray.firstOrNull()?.let {
            onNewSocketResponseReceived(channelName, it.toString())
        }
    }

    //override this function to get the socket response
    open fun onNewSocketResponseReceived(channel: String, response: String) {}

    //Override this to store socket id
    open fun storeSocketId(id: String?) {}

    //Override this to store live chat socket id
    open fun storeLiveChatSocketId(id: String?) {}

    protected suspend fun connectToSocket(): Boolean {
        return initConnection(socket)
    }

    private suspend fun initConnection(socket: Socket): Boolean {
        return withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
            suspendCancellableCoroutine<Boolean> { continuation ->
                val connectCallback = object : Emitter.Listener {
                    override fun call(vararg args: Any?) {
                        LogUtils.log("Socket connected")
                        continuation.resume(true)
                        socket.off(Socket.EVENT_CONNECT, this)
                    }
                }
                socket.on(Socket.EVENT_CONNECT, connectCallback)
                socket.connect()

                continuation.invokeOnCancellation {
                    LogUtils.log("Socket - invokeOnCancellation")
                    socket.off(Socket.EVENT_CONNECT, connectCallback)
                }
            }
        } ?: false
    }

    protected fun isSocketConnected() = socket.connected()

    protected fun isLiveChatSocketConnected() = liveChatSocket.connected()

    protected fun resetSocketConnection() {
        viewModelScope.launch {
            synchronized(this) {
                LogUtils.log("Socket disconnect request")
                socket.disconnect()
                socket.connect()
                LogUtils.log("Socket connect request")
            }
        }
    }

    protected suspend fun resetLiveChatSocketConnection(): Boolean {
        LogUtils.log("LCSocket disconnect request")
        disconnectLiveChatSocket()
        LogUtils.log("LCSocket connect request")
        return connectToLiveChatSocket()
    }

    override fun onCleared() {
        super.onCleared()
        socket.off()
        socket.disconnect()
        liveChatSocket.off()
        liveChatSocket.disconnect()
    }

    protected suspend fun connectToLiveChatSocket(): Boolean {
        return initConnection(liveChatSocket)
    }

    protected fun disconnectLiveChatSocket() {
        liveChatSocket.disconnect()
    }
}