package ai.engagely.openbot.view.customviews

import ai.engagely.openbot.R
import ai.engagely.openbot.databinding.ViewItemChatFormBinding
import ai.engagely.openbot.model.pojos.internal.chat.IChatFormItem
import ai.engagely.openbot.model.pojos.internal.history.*
import android.content.Context
import android.util.AttributeSet
import android.util.TypedValue
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.LinearLayout


class ItemFormView : FrameLayout {

    private var binding: ViewItemChatFormBinding? = null
    private val formCardMargin by lazy {
        resources.getDimension(R.dimen.common_card_margin)
    }
    private val formWidthFraction by lazy {
        val outValue = TypedValue()
        resources.getValue(R.dimen.chat_bubble_width_fraction, outValue, true)
        outValue.float
    }
    var containerWidth: Int = 0

    constructor(context: Context) : super(context) {
        initUi()
    }

    constructor(context: Context, attrs: AttributeSet) : super(context, attrs) {
        initUi()
    }

    constructor(context: Context, attrs: AttributeSet, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        initUi()
    }

    private fun initUi() {
        binding = ViewItemChatFormBinding.inflate(LayoutInflater.from(context), this, true)
    }

    fun calculateDynamicSizes(screenWidth: Int) {
        containerWidth = ((screenWidth * formWidthFraction) - (2 * formCardMargin)).toInt()
    }

    fun setData(
        iChatFormItem: IChatFormItem?,
        themeColor: Int,
        onItemElementClickListener: OnClickListener
    ) {

        binding?.llFormItems?.removeAllViews()
        iChatFormItem?.formItemContainers?.forEach { formItemsContainer ->
            if (formItemsContainer.formsItems.isNotEmpty()) {
                if (formItemsContainer.formsItems.size == 1) {
                    getFormItemView(
                        iChatFormItem,
                        formItemsContainer.formsItems.first(),
                        themeColor,
                        containerWidth,
                        onItemElementClickListener
                    )?.let { itemView ->
                        binding?.llFormItems?.addView(itemView)
                    }
                } else {
                    val itemsContainer = LinearLayout(context)
                    itemsContainer.layoutParams = createLayoutParams()
                    itemsContainer.gravity = Gravity.BOTTOM
                    formItemsContainer.formsItems.forEach { formItem ->
                        getFormItemView(
                            iChatFormItem,
                            formItem,
                            themeColor,
                            containerWidth,
                            onItemElementClickListener
                        )?.let { itemView ->
                            itemsContainer.addView(itemView)
                        }
                    }
                    binding?.llFormItems?.addView(itemsContainer)
                }
            }
        }
        binding?.vTopStrip?.setBackgroundColor(themeColor)
    }

    private fun createLayoutParams(
        width: Int? = null,
        height: Int? = null
    ): ViewGroup.LayoutParams {
        return ViewGroup.LayoutParams(
            width ?: ViewGroup.LayoutParams.MATCH_PARENT,
            height ?: ViewGroup.LayoutParams.WRAP_CONTENT
        )
    }

    private fun getFormItemView(
        iChatFormItem: IChatFormItem,
        iFormItem: IFormItem, themeColor: Int,
        containerWidth: Int,
        onItemElementClickListener: OnClickListener
    ): View? {
        when (iFormItem) {
            is IFormTextItem -> {
                val textFieldView = FormTextFieldView(context)
                applyLayoutParamsIfNeeded(iFormItem, textFieldView, containerWidth)
                textFieldView.setData(iFormItem)
                return textFieldView
            }
            is IFormRadioButtonsItem -> {
                val radioButtonsView = FormRadioButtonsView(context)
                applyLayoutParamsIfNeeded(iFormItem, radioButtonsView, containerWidth)
                radioButtonsView.setData(iFormItem, themeColor)
                return radioButtonsView
            }
            is IFormTextAreaItem -> {
                val textAreaView = FormTextAreaView(context)
                applyLayoutParamsIfNeeded(iFormItem, textAreaView, containerWidth)
                textAreaView.setData(iFormItem)
                return textAreaView
            }
            is IFormCheckboxesItem -> {
                val checkBoxesView = FormCheckBoxesView(context)
                applyLayoutParamsIfNeeded(iFormItem, checkBoxesView, containerWidth)
                checkBoxesView.setData(iFormItem, themeColor)
                return checkBoxesView
            }
            is IFormDateItem -> {
                val dateView = FormDateVIew(context)
                applyLayoutParamsIfNeeded(iFormItem, dateView, containerWidth)
                dateView.setData(iFormItem, onItemElementClickListener)
                return dateView
            }
            is IFormDropDownItem -> {
                val spinnerView = FormSpinnerView(context)
                applyLayoutParamsIfNeeded(iFormItem, spinnerView, containerWidth)
                spinnerView.setData(iFormItem)
                return spinnerView
            }
            is IFormFileItem -> {
                val fileUploadView = FormFileUploadView(context)
                applyLayoutParamsIfNeeded(iFormItem, fileUploadView, containerWidth)
                fileUploadView.setData(iFormItem, onItemElementClickListener)
                return fileUploadView
            }
            is IFormButtonItem -> {
                val buttonView = FormButtonView(context)
                applyLayoutParamsIfNeeded(iFormItem, buttonView, containerWidth)
                buttonView.setData(iFormItem, iChatFormItem, themeColor, onItemElementClickListener)
                return buttonView
            }
        }
        return null
    }

    private fun applyLayoutParamsIfNeeded(
        iFormItem: IFormItem,
        view: View,
        containerWidth: Int
    ) {
        if (iFormItem.widthFraction < 1.0f) {
            view.layoutParams =
                createLayoutParams(width = (containerWidth * iFormItem.widthFraction).toInt())
        }
    }
}