package ai.engagely.openbot.model.repositories.impl

import ai.engagely.openbot.model.constants.ServerConstants
import ai.engagely.openbot.model.network.ApiClient
import ai.engagely.openbot.model.network.ApiConstants
import ai.engagely.openbot.model.network.interfaces.BotSettingsApi
import ai.engagely.openbot.model.pojos.external.apirequests.botsettings.GetBotSettingsRequest
import ai.engagely.openbot.model.pojos.external.apiresponses.botsettings.BotSetting
import ai.engagely.openbot.model.pojos.internal.botsettings.*
import ai.engagely.openbot.model.repositories.BotSettingRepository
import ai.engagely.openbot.model.utils.general.LogUtils
import ai.engagely.openbot.model.utils.general.ViewUtils
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import java.net.HttpURLConnection

class AppBotSettingsRepository(private val dispatcher: CoroutineDispatcher) :
    BotSettingRepository {

    override suspend fun getBotSettings(
        botId: String,
        languageCode: String?
    ): IBotChatWindowConfig? {
        return withContext(dispatcher) {
            try {
                val botApi = ApiClient.getInstance().getClient().create(BotSettingsApi::class.java)
                val botRequest = GetBotSettingsRequest(intId = botId, languageCode = languageCode)
                val response = botApi.getBotSettings(botRequest)
                if (response.isSuccessful && response.code() == HttpURLConnection.HTTP_OK
                    && response.body()?.success == true
                ) {
                    response.body()?.data?.let {
                        val decodedLogo =
                            ViewUtils.convertFromBase64ToBitmap(it.botSetting?.botLogo)
                        val iBotChatWindowConfig = IBotChatWindowConfig(
                            toolbarTitle = it.botSetting?.botName,
                            logo = decodedLogo,
                            defaultILanguage = ILanguage(
                                languageCode = it.botSetting?.defaultLanguage?.languageCode,
                                languageName = it.botSetting?.defaultLanguage?.languageName,
                            ),
                            languageData = it.botSetting?.languageData?.filterNotNull()
                                ?.map { languageDataItem ->
                                    ILanguage(
                                        languageDataItem.languageCode,
                                        languageDataItem.languageName
                                    )
                                },
                            topJrItems = it.topJr?.filterNotNull()?.map { topJrItem ->
                                ITopJr(
                                    intId = topJrItem.intId,
                                    newTag = topJrItem.newTag,
                                    newTopic = topJrItem.newTopic,
                                    responseId = topJrItem.responseId,
                                    topic = topJrItem.topic,
                                    topicNo = topJrItem.topicNo
                                )
                            },
                            quickMenuItems = it.botSetting?.quickMenuList?.filterNotNull()
                                ?.map { quickMenuItem ->
                                    IQuickMenuItem(
                                        journey = quickMenuItem.journey,
                                        responseId = quickMenuItem.responseId,
                                        topic = quickMenuItem.topic,
                                        journeyPosition = quickMenuItem.journeyPositon,
                                        journeyImageName = quickMenuItem.journeyImageName,
                                        journeyImageUrl = ApiConstants.QUICK_MENU_ITEM_IMAGE_BASE_URL + (quickMenuItem.journeyImageName
                                            ?: "")
                                    )
                                },
                            quickMenuPosition = when (it.botSetting?.quickMenuPosition) {
                                ServerConstants.QUICK_MENU_POSITION_BOTTOM -> IQuickMenuPosition.BOTTOM
                                ServerConstants.QUICK_MENU_POSITION_LIST -> IQuickMenuPosition.LIST
                                ServerConstants.QUICK_MENU_POSITION_RIGHT -> IQuickMenuPosition.RIGHT
                                else -> IQuickMenuPosition.RIGHT
                            },
                            botBgColor = it.botSetting?.botBgColor,
                            botButtonColor = it.botSetting?.botButtonColor,
                            botChatBoxColor = it.botSetting?.botChatBoxColor,
                            botTextColor = it.botSetting?.botTextColor,
                            botThemColor = it.botSetting?.botThemeColor,
                            buttonColor = it.botSetting?.button,
                            linkColor = it.botSetting?.link,
                            userChatBoxColor = it.botSetting?.userChatBoxColor,
                            userTextColor = it.botSetting?.userTextColor,
                            headerTextColor = it.botSetting?.headerText,
                            userInputBoxColor = it.botSetting?.userInputBox,
                            botId = botId,
                            intId = it.botSetting?.intId,
                            intName = it.botSetting?.intName,
                            greetingMessage = if (it.botSetting?.uiStyle == ServerConstants.GREETING_STYLE_OLD) {
                                IGreetingMessage(
                                    messages = parseGreetingMessages(it.botSetting),
                                )
                            } else null,
                            greetingMessageWithHeader = if (it.botSetting?.uiStyle == ServerConstants.GREETING_STYLE_NEW) {
                                IGreetingMessageWithHeader(
                                    headerMessage = parseGreetingHeader(it.botSetting),
                                    messages = parseGreetingMessages(it.botSetting),
                                )
                            } else null,
                            carousalGreetingMessage = if (it.botSetting?.uiStyle == ServerConstants.GREETING_STYLE_CAROUSAL) {
                                ICarousalGreetingMessage(
                                    headerMessage = parseGreetingHeader(it.botSetting),
                                    messages = parseGreetingMessages(it.botSetting),
                                    greetingCarousalImages = it.botSetting.greetCarouselImages?.filterNotNull()
                                        ?.map { greetingCarousalImage ->
                                            return@map IGreetingCarousalImage(
                                                clickEnabled = greetingCarousalImage.enableButton
                                                    ?: false,
                                                clickType = greetingCarousalImage.buttonType,
                                                clickData = greetingCarousalImage.buttonData,
                                                imageName = ApiConstants.GREETING_IMAGE_BASE_URL + (greetingCarousalImage.imageName
                                                    ?: "")
                                            )
                                        },
                                    color = it.botSetting.headerText,
                                )
                            } else null,
                            optionsStyle = when (it.botSetting?.topJourneyStyle) {
                                ServerConstants.TOP_JOURNEY_STYLE_BUTTON -> IOptionsStyle.BUTTON
                                ServerConstants.TOP_JOURNEY_STYLE_LIST -> IOptionsStyle.LIST
                                else -> IOptionsStyle.BUTTON
                            },
                            topJourneyHeader = it.botSetting?.listHeader as? String,
                            enableGreetingJourney = it.botSetting?.enableGreetingJourney ?: false,
                            greetingJourney = it.botSetting?.greetingJourney,
                            autoCompleteEnabled = it.botSetting?.autocompleteStatus ?: false,
                            showWelcomeMessage = "show" == it.botSetting?.welcomeMsgMobile,
                            botImageShape = if ("Tipped" == it.botSetting?.iconShape)
                                IBotChatWindowConfig.BotImageShape.TRIPPED
                            else IBotChatWindowConfig.BotImageShape.ROUNDED,
                            botImageSize = if ("small" == it.botSetting?.botIconInMobile)
                                IBotChatWindowConfig.BotImageSize.SMALL
                            else IBotChatWindowConfig.BotImageSize.STANDARD,
                            enableMic = it.botSetting?.enableMic == true,
                            enableSpeech = it.botSetting?.enableSpeech == true,
                            iBotFeedbackRequest = it.botSetting?.feedbackCSATSetting?.let { feedbackCSARSetting ->
                                if (feedbackCSARSetting.showSurvey == true) {
                                    val showAfterSeconds =
                                        if (ServerConstants.FEEDBACK_CSAT_ON_IDLE_TIME == feedbackCSARSetting.csatOn) {
                                            ((feedbackCSARSetting.csatDetails as? Double)?.toInt())
                                                ?: 0
                                        } else 0
                                    val showAfterMessages =
                                        if (ServerConstants.FEEDBACK_CSAT_ON_AFTER_MESSAGE == feedbackCSARSetting.csatOn) {
                                            ((feedbackCSARSetting.csatDetails as? Double)?.toInt())
                                                ?: 0
                                        } else 0
                                    if (showAfterSeconds > 0 || showAfterMessages > 0) {
                                        IBotFeedbackRequest(
                                            surveyMessage = feedbackCSARSetting.csatQuestions,
                                            showAfterSeconds = showAfterSeconds,
                                            showAfterMessages = showAfterMessages
                                        )
                                    } else {
                                        null
                                    }
                                } else {
                                    null
                                }
                            }
                        )
                        return@withContext iBotChatWindowConfig
                    }
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            return@withContext null
        }
    }

    private fun parseGreetingHeader(botSetting: BotSetting) =
        if (botSetting.greetingHeader is String)
            botSetting.greetingHeader else null

    private fun parseGreetingMessages(botSetting: BotSetting) =
        (botSetting.greetingMessage?.filterNotNull()
            ?.map { message ->
                IGreetingMessageItem(message, null)
            }
            ?: emptyList())
}