package ai.engagely.openbot.model.repositories.impl

import ai.engagely.openbot.model.constants.ServerConstants
import ai.engagely.openbot.model.network.ApiClient
import ai.engagely.openbot.model.network.interfaces.HistoryApi
import ai.engagely.openbot.model.pojos.external.apirequests.history.GetChatHistoryRequest
import ai.engagely.openbot.model.pojos.external.apirequests.history.Location
import ai.engagely.openbot.model.pojos.external.apiresponses.history.GetChatHistoryResponse
import ai.engagely.openbot.model.pojos.internal.chat.IChatItem
import ai.engagely.openbot.model.pojos.internal.history.IHistoryContainer
import ai.engagely.openbot.model.pojos.internal.livechat.ILiveChatData
import ai.engagely.openbot.model.pojos.internal.livechat.ILiveChatFeedbackRequest
import ai.engagely.openbot.model.pojos.internal.livechat.ILiveChatRatingItem
import ai.engagely.openbot.model.repositories.ChatHistoryRepository
import ai.engagely.openbot.model.repositories.LiveChatRepository
import ai.engagely.openbot.model.utils.converters.ChatResponseConverter
import ai.engagely.openbot.model.utils.exts.toMap
import ai.engagely.openbot.model.utils.general.LogUtils
import android.os.Bundle
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import retrofit2.Response
import java.net.HttpURLConnection

class AppChatHistoryRepository(
    private val dispatcher: CoroutineDispatcher,
    private val chatResponseConverter: ChatResponseConverter,
    private val liveChatRepository: LiveChatRepository
) : ChatHistoryRepository {

    override suspend fun getChatHistory(
        userId: String?,
        sessionId: String?,
        intId: String?,
        extras: Bundle?
    ): IHistoryContainer? {
        return withContext(dispatcher) {
            try {
                val location = Location(latitude = 0, longitude = 0)
                val request = GetChatHistoryRequest(
                    userId = userId,
                    queryParams = extras.toMap(),
                    sessionId = sessionId,
                    intId = intId,
                    location = location
                )
                val response = ApiClient.getInstance().getClient().create(HistoryApi::class.java)
                    .getChatHistory(request)
                if (response.isSuccessful && response.code() == HttpURLConnection.HTTP_OK
                    && response.body()?.success == true
                ) {
                    val chatItems = ArrayList<IChatItem>()
                    response.body()?.chatData?.forEach { chatDataItem ->
                        chatResponseConverter.convertHistoryItemToChatItem(
                            chatDataItem = chatDataItem,
                            sessionId = sessionId
                        )?.let {
                            chatItems.addAll(it)
                        }
                    }

                    val feedbackRequest = parseFeedback(response)

                    val liveChatData = syncLiveChatData(response)

                    val isBotFeedbackShown = isBotFeedbackShown(response)

                    return@withContext IHistoryContainer(
                        historyItems = chatItems,
                        feedbackRequest = feedbackRequest,
                        liveChatItem = liveChatData,
                        isBotFeedbackShown = isBotFeedbackShown
                    )
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            return@withContext null
        }
    }

    private fun isBotFeedbackShown(
        response: Response<GetChatHistoryResponse>,
    ): Boolean {
        response.body()?.let { getChatHistoryResponse ->
            if (getChatHistoryResponse.feedback == true) {
                return true
            }
        }
        return false
    }

    private suspend fun syncLiveChatData(response: Response<GetChatHistoryResponse>): ILiveChatData? {
        var liveChatData = liveChatRepository.readLiveChatDetails()
        response.body()?.chatData?.let { chatDataItems ->
            if (chatDataItems.size >= 2) {
                val lastItem = chatDataItems.last()
                val lastButOneItem = chatDataItems[chatDataItems.size - 2]

                if (ServerConstants.CHAT_ITEM_TYPE_LIVE_CHAT_DISCONNECTED == lastItem.responseType
                    || ServerConstants.CHAT_ITEM_TYPE_LIVE_CHAT_DISCONNECTED == lastButOneItem.responseType
                ) {
                    liveChatData = null
                    liveChatRepository.saveLiveChatDetails(null)
                }
            }
        }
        return liveChatData
    }

    private suspend fun parseFeedback(response: Response<GetChatHistoryResponse>): ILiveChatFeedbackRequest? {
        response.body()?.chatData?.let { chatDataItems ->
            if (chatDataItems.size >= 2) {
                liveChatRepository.readLiveChatDetails()?.let { liveChatData ->
                    chatDataItems.last().endChatResult?.let { endChatResult ->
                        return ILiveChatFeedbackRequest(
                            surveyMessage = endChatResult.surveyMessage,
                            ratings = HashMap<Int, ILiveChatRatingItem>().apply {
                                endChatResult.ratingsList?.forEach { ratingItem ->
                                    ratingItem.indexNo?.let { index ->
                                        put(
                                            index,
                                            ILiveChatRatingItem(
                                                isRatingPositive = ratingItem.isRatingPositive
                                                    ?: false,
                                                rating = ratingItem.rating
                                            )
                                        )
                                    }
                                }
                            },
                            liveChatData = liveChatData
                        )
                    }
                }
            }
        }
        return null
    }
}