package ai.engagely.openbot.model.repositories.impl

import ai.engagely.openbot.model.constants.ServerConstants
import ai.engagely.openbot.model.network.ApiClient
import ai.engagely.openbot.model.network.interfaces.FormApi
import ai.engagely.openbot.model.pojos.external.apirequests.form.FormSubmitRequest
import ai.engagely.openbot.model.pojos.internal.chat.IChatFormItem
import ai.engagely.openbot.model.pojos.internal.history.*
import ai.engagely.openbot.model.repositories.FormRepository
import ai.engagely.openbot.model.utils.general.DateUtils
import ai.engagely.openbot.model.utils.general.LogUtils
import ai.engagely.openbot.model.utils.general.RetrofitUtils
import com.google.gson.Gson
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import okhttp3.MultipartBody
import okhttp3.RequestBody

class AppFormRepository(
    private val dispatcher: CoroutineDispatcher,
) : FormRepository {

    override suspend fun submitForm(
        iChatFormItem: IChatFormItem,
        botId: String?,
        inputId: String?,
        journeyId: String?,
        languageCode: String?,
        messageType: String?,
        msgId: String?,
        nodeId: Int?,
        onSubmit: String?,
        pageUrl: String?,
        respondName: String?,
        respondType: String?,
        sessionId: String?,
        sessionObj: Any?,
        socketId: String?,
        userInput: String?,
        webSocketId: String?,
    ): FormRepository.Failure? {
        return withContext(dispatcher) {
            try {
                val partValues: HashMap<String, RequestBody> = HashMap()
                val partFiles: ArrayList<MultipartBody.Part> = ArrayList()

                val formFields = ArrayList<String>()
                iChatFormItem.formItemContainers.forEach { formItemsContainer ->
                    formItemsContainer.formsItems.forEach { formItem ->
                        when (formItem) {
                            is IFormFileItem -> {
                                formItem.file?.let { selectedFile ->
                                    formItem.mimeType?.let { mimeType ->
                                        formItem.variable?.let { variable ->
                                            partFiles.add(
                                                RetrofitUtils.prepareFilePart(
                                                    variable,
                                                    selectedFile,
                                                    mimeType
                                                )
                                            )
                                        }
                                    }
                                }
                            }
                            is IFormTextItem -> {
                                formItem.value?.let { value ->
                                    formItem.variable?.let { variable ->
                                        val valueBuilder = StringBuilder()
                                        if ((formItem.prefixValue?.isNotBlank() == true)
                                            && formItem.textBoxTypeSelected == ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_PHONE_NUMBER
                                        ) {
                                            valueBuilder.append(formItem.prefixValue)
                                            valueBuilder.append(" ")
                                        }
                                        valueBuilder.append(value)
                                        partValues[variable] =
                                            RetrofitUtils.createPartFromString(valueBuilder.toString())
                                        formFields.add(variable)
                                    }
                                }
                            }
                            is IFormTextAreaItem -> {
                                formItem.value?.let { value ->
                                    formItem.variable?.let { variable ->
                                        partValues[variable] =
                                            RetrofitUtils.createPartFromString(value)
                                        formFields.add(variable)
                                    }
                                }
                            }
                            is IFormCheckboxesItem -> {
                                val value = StringBuilder()
                                formItem.options.forEach {
                                    if (it.isChecked) {
                                        if (value.isNotBlank()) {
                                            value.append(",")
                                        }
                                        value.append(it.label)
                                    }
                                }
                                formItem.variable?.let { variable ->
                                    partValues[variable] =
                                        RetrofitUtils.createPartFromString(value.toString())
                                    formFields.add(variable)
                                }
                            }
                            is IFormRadioButtonsItem -> {
                                val value = StringBuilder()
                                formItem.options.forEach {
                                    if (it.isChecked) {
                                        if (value.isNotBlank()) {
                                            value.append(",")
                                        }
                                        value.append(it.value)
                                    }
                                }
                                formItem.variable?.let { variable ->
                                    partValues[variable] =
                                        RetrofitUtils.createPartFromString(value.toString())
                                    formFields.add(variable)
                                }
                            }
                            is IFormDateItem -> {
                                formItem.selectedDate?.let { selectedDate ->
                                    formItem.variable?.let { variable ->
                                        partValues[variable] = RetrofitUtils.createPartFromString(
                                            DateUtils.convertToServerDateFormat(selectedDate)
                                        )
                                        formFields.add(variable)
                                    }
                                }
                            }
                            is IFormDropDownItem -> {
                                formItem.variable?.let { variable ->
                                    formItem.options[formItem.selectedOptionPosition].value?.let { value ->
                                        partValues[variable] =
                                            RetrofitUtils.createPartFromString(value)
                                        formFields.add(variable)
                                    }
                                }
                            }
                        }
                    }
                }

                val formSubmitRequest = FormSubmitRequest(
                    botId = botId,
                    inputId = inputId,
                    journeyId = journeyId,
                    languageCode = languageCode,
                    messageType = messageType,
                    msgId = msgId,
                    nodeId = nodeId,
                    onSubmit = onSubmit,
                    pageUrl = pageUrl,
                    respondName = respondName,
                    respondType = respondType,
                    sessionId = sessionId,
                    sessionObj = sessionObj,
                    socketId = socketId,
                    userInput = userInput,
                    websocketid = webSocketId
                )
                partValues["dataToSend"] =
                    RetrofitUtils.createPartFromString(Gson().toJson(formSubmitRequest))
                partValues["formFields"] =
                    RetrofitUtils.createPartFromString(Gson().toJson(formFields))

                val response = ApiClient.getInstance().getClient().create(FormApi::class.java)
                    .submitForm(partValues, partFiles)
                if (!response.isSuccessful) {
                    return@withContext FormRepository.Failure.SERVER_ERROR
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
                return@withContext FormRepository.Failure.SERVER_ERROR
            }
            return@withContext null
        }
    }
}