package ai.engagely.openbot.model.utils.helpers

import ai.engagely.openbot.model.constants.ViewConstants
import ai.engagely.openbot.model.pojos.internal.botsettings.IBotChatWindowConfig
import ai.engagely.openbot.model.utils.general.ViewUtils
import ai.engagely.openbot.view.activities.BotChatActivity
import ai.engagely.openbot.view.activities.ExoPlayerActivity
import ai.engagely.openbot.view.activities.PermissionCompatActivity
import ai.engagely.openbot.view.customviews.BotContainerView
import ai.engagely.openbot.view.customviews.WelcomeVideoView
import ai.engagely.openbot.viewmodel.OpenBotViewModel
import ai.engagely.openbot.viewmodel.VideoPlayerViewModel
import android.app.Activity
import android.content.Intent
import android.content.res.Configuration
import android.view.View
import androidx.activity.result.contract.ActivityResultContracts
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.ViewModelProvider

class BotViewHelper(
    activity: FragmentActivity,
    private val botViewContainerView: BotContainerView?
) : PermissionCompatActivity.ReadExternalStoragePermissionListener {

    private val openBotViewModel by lazy {
        ViewModelProvider(activity)[OpenBotViewModel::class.java]
    }

    private val playerViewModel by lazy {
        ViewModelProvider(activity)[VideoPlayerViewModel::class.java]
    }

    private val chatResult =
        activity.registerForActivityResult(ActivityResultContracts.StartActivityForResult()) {
            it?.let { activityResult ->
                openBotViewModel.resetWelcomeData()
                activityResult.data?.getParcelableExtra<IBotChatWindowConfig>(IBotChatWindowConfig.UNIQUE_NAME)
                    ?.let { botConfigData ->
                        openBotViewModel.updateBotConfigData(botConfigData)
                    }
                when (activityResult.resultCode) {
                    RESULT_SESSION_UPDATE_REQUESTED -> {
                        openBotViewModel.refreshSession()
                    }
                }
            }
        }

    private val exoPlayerResult =
        activity.registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { activityResult ->
            if (Activity.RESULT_OK == activityResult.resultCode) {
                activityResult.data?.let { resultData ->
                    resultData.getStringExtra(ExoPlayerActivity.VIDEO_URL_TAG)?.let { url ->
                        playerViewModel.updateInitialData(
                            resultData.getLongExtra(
                                ExoPlayerActivity.VIDEO_CURRENT_POSITION_TAG,
                                0
                            ),
                            resultData.getBooleanExtra(
                                ExoPlayerActivity.VIDEO_PLAY_WHEN_READY_TAG,
                                false
                            ),
                            resultData.getFloatExtra(
                                ExoPlayerActivity.VIDEO_PLAYBACK_SPEED_TAG,
                                1f
                            ),
                            url
                        )
                    }
                }
            }
        }

    init {
        activity.lifecycle.addObserver(playerViewModel)
        setObservers(activity)
        setClickListeners(activity)
    }

    fun init(botId: String, extras: Map<String, String>? = null) {
        openBotViewModel.init(botId, extras)
        openBotViewModel.fetchData()
    }

    private fun setClickListeners(activity: FragmentActivity) {
        botViewContainerView?.getBotView()?.setImageClickListener {
            val botChatConfig = openBotViewModel.botChatWindowConfig
            botChatConfig?.let {
                chatResult.launch(Intent(activity, BotChatActivity::class.java).apply {
                    putExtra(IBotChatWindowConfig.UNIQUE_NAME, botChatConfig)
                    putExtras(openBotViewModel.getExtras())
                })
            }
        }
        botViewContainerView?.getBotView()?.setImageLongClickListener {
            if (openBotViewModel.welcomeDataLiveData.value?.isNotEmpty() == true) {
                botViewContainerView.getWelcomeData()?.visibility = View.VISIBLE
                scrollWelcomeDataToEnd()
                true
            } else {
                false
            }
        }
        botViewContainerView?.getBotView()?.setErrorRetryClickListener {
            openBotViewModel.fetchData()
        }
    }

    private fun setObservers(activity: FragmentActivity) {
        openBotViewModel.isLoadingInitialApi.observe(activity) { value ->
            botViewContainerView?.getBotView()?.setLoading(value)
        }
        openBotViewModel.imageBitmapLiveData.observe(activity) { value ->
            value?.let { bitmap ->
                botViewContainerView?.getBotView()?.setImage(bitmap)
            }
        }
        openBotViewModel.errorStateLiveData.observe(activity) {
            botViewContainerView?.getBotView()?.setErrorView(it != null)
        }
        openBotViewModel.welcomeDataLiveData.observe(activity) {
            if (it?.isNotEmpty() == true) {
                val themeColor = ViewUtils.parseColorSafely(
                    openBotViewModel.botChatWindowConfig?.botThemColor,
                    ViewConstants.VIEW_DEFAULT_COLOR_STRING_DARK
                )
                botViewContainerView?.getWelcomeData()
                    ?.setData(it, themeColor, object : WelcomeVideoView.OnPlayClickListener {
                        override fun onPlayClicked(url: String?) {
                            playerViewModel.startPlaying(url)
                        }

                        override fun onFullScreenClicked() {
                            playerViewModel.getUrl()?.let { url ->
                                ExoPlayerActivity.getRequestIntent(
                                    activity,
                                    playerViewModel.getCurrentPosition(),
                                    playerViewModel.getPlayWhenReady(),
                                    playerViewModel.getPlaybackSpeed(),
                                    url
                                ).also { intent ->
                                    playerViewModel.stopPlaying()
                                    exoPlayerResult.launch(intent)
                                }
                            }
                        }

                        override fun onVolumeToggleClicked() {
                            playerViewModel.toggleVolume()
                        }

                        override fun onPlaybackSpeedChanged(speed: Float) {
                            playerViewModel.setPlaybackSpeed(speed)
                        }

                        override fun onDownloadMenuSelected() {
                            (activity as? PermissionCompatActivity)?.requestReadExternalStoragePermission(
                                REQUEST_READ_EXTERNAL_STORAGE_PERMISSION,
                                this@BotViewHelper
                            )
                        }
                    })
                botViewContainerView?.getWelcomeData()?.visibility = View.VISIBLE
                botViewContainerView?.getWelcomeData()?.setPLayer(playerViewModel.player)
                botViewContainerView?.getWelcomeData()
                    ?.setVolumeMuted(playerViewModel.volumeMutedLiveData.value)
                botViewContainerView?.getWelcomeData()
                    ?.setPlaybackSpeed(playerViewModel.playbackSpeedLiveData.value ?: 1f)
                scrollWelcomeDataToEnd()
            } else {
                botViewContainerView?.getWelcomeData()?.visibility = View.GONE
            }
        }
        openBotViewModel.welcomeVideoUrlLiveData.observe(activity) { url ->
            playerViewModel.startPlaying(url)
        }
        playerViewModel.playerLiveData.observe(activity) {
            botViewContainerView?.getWelcomeData()?.setPLayer(it)
        }
        playerViewModel.volumeMutedLiveData.observe(activity) {
            botViewContainerView?.getWelcomeData()?.setVolumeMuted(it)
        }
        playerViewModel.playbackSpeedLiveData.observe(activity) {
            botViewContainerView?.getWelcomeData()
                ?.setPlaybackSpeed(it ?: 1f)
        }
        openBotViewModel.botImageDimensionsLiveData.observe(activity) { botDimension ->
            botViewContainerView?.getBotView()?.setBotDimensions(botDimension)
        }
    }

    private fun scrollWelcomeDataToEnd() {
        botViewContainerView?.getWelcomeDataContainer()?.post {
            botViewContainerView.getWelcomeDataContainer()?.fullScroll(View.FOCUS_DOWN)
        }
    }

    fun onConfigurationChanged(newConfig: Configuration) {
        scrollWelcomeDataToEnd()
    }

    override fun onPermissionResponse(
        requestCode: Int,
        readExternalStoragePermission: PermissionCompatActivity.ReadExternalStoragePermission
    ) {
        if (readExternalStoragePermission == PermissionCompatActivity.ReadExternalStoragePermission.GRANTED) {
            if (requestCode == REQUEST_READ_EXTERNAL_STORAGE_PERMISSION) {
                playerViewModel.downloadVideo()
            }
        }
    }

    companion object {
        const val RESULT_SESSION_UPDATE_REQUESTED = 999
        const val RESULT_BOT_CONFIG_UPDATED = 1000
        const val REQUEST_READ_EXTERNAL_STORAGE_PERMISSION = 1001
    }
}