package ai.engagely.openbot.view.activities

import ai.engagely.openbot.R
import ai.engagely.openbot.databinding.ActivityExoPlayerBinding
import ai.engagely.openbot.view.customviews.PlaybackSpeedMenu
import ai.engagely.openbot.view.customviews.PlayerMoreMenuView
import ai.engagely.openbot.viewmodel.VideoPlayerViewModel
import android.app.Activity
import android.content.Intent
import android.content.res.Configuration
import android.os.Bundle
import android.view.View
import android.widget.ImageButton
import androidx.lifecycle.ViewModelProvider

class ExoPlayerActivity : BaseActivity() {

    private lateinit var binding: ActivityExoPlayerBinding
    private var isFullScreen = false
    private lateinit var fullScreenButton: ImageButton
    private var volumeHigh: ImageButton? = null
    private var volumeOff: ImageButton? = null
    private var playbackSpeedMenu: PlaybackSpeedMenu? = null
    private var playerMoreMenuView: PlayerMoreMenuView? = null

    private val playerViewModel by lazy {
        ViewModelProvider(this)[VideoPlayerViewModel::class.java]
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        binding = ActivityExoPlayerBinding.inflate(layoutInflater)
        setContentView(binding.root)

        lifecycle.addObserver(playerViewModel)

        setupActionBar()
        setupVolumeButton()
        setupVideo()
        setupFullScreenButton()
        setupPlaybackSpeedButton()
        setupDownloadOption()
    }

    private fun setupDownloadOption() {
        playerMoreMenuView = findViewById(R.id.playerMoreMenu)

        playerMoreMenuView?.setData(object : PlayerMoreMenuView.PlayerMoreMenuClickListener {
            override fun onDownloadMenuSelected() {
                playerViewModel.downloadVideo()
            }
        })
    }

    private fun setupPlaybackSpeedButton() {
        playbackSpeedMenu = findViewById(R.id.playbackSpeedMenu)

        val playbackSpeedListener = object : PlaybackSpeedMenu.PlaybackSpeedListener {
            override fun onPlaybackSpeedChanged(speed: Float) {
                playerViewModel.setPlaybackSpeed(speed)
            }
        }
        playerViewModel.playbackSpeedLiveData.observe(this) {
            playbackSpeedMenu?.setData(it ?: 1f, playbackSpeedListener)
        }
    }

    private fun setupVolumeButton() {
        volumeHigh = findViewById(R.id.ib_volume_high)
        volumeOff = findViewById(R.id.ib_volume_off)
        volumeHigh?.setOnClickListener {
            playerViewModel.toggleVolume()
        }
        volumeOff?.setOnClickListener {
            playerViewModel.toggleVolume()
        }

        playerViewModel.volumeMutedLiveData.observe(this) {
            val showMuted = it == true
            volumeOff?.visibility = if (showMuted) View.VISIBLE else View.GONE
            volumeHigh?.visibility = if (showMuted) View.GONE else View.VISIBLE
        }
    }

    private fun setupFullScreenButton() {
        fullScreenButton = findViewById(R.id.ib_full_screen)
        fullScreenButton.setOnClickListener {
            toggleFullScreen()
        }
    }

    private fun setupVideo() {
        intent.getStringExtra(VIDEO_URL_TAG)?.let { url ->
            playerViewModel.updateInitialData(
                intent.getLongExtra(VIDEO_CURRENT_POSITION_TAG, 0),
                intent.getBooleanExtra(VIDEO_PLAY_WHEN_READY_TAG, true),
                intent.getFloatExtra(VIDEO_PLAYBACK_SPEED_TAG, 1f),
                url
            )
            playerViewModel.startPlaying(url)
            playerViewModel.playerLiveData.observe(this) {
                binding.playerView.player = it
                binding.playerView.showController()
            }
        }
    }

    private fun setupActionBar() {
        setSupportActionBar(binding.toolbar)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.setHomeAsUpIndicator(R.drawable.ic_arrow_left_white)
        supportActionBar?.title = intent.getStringExtra(VIDEO_TITLE_TAG) ?: ""
        supportActionBar?.subtitle =
            intent.getStringExtra(VIDEO_SUBTITLE_TAG) ?: ""
    }

    private fun toggleFullScreen() {
        if (!isFullScreen) {
            hideControls()
        } else {
            showControls()
        }
    }

    private fun showControls() {
        unhideSystemBars()
        hideToolbar(false)
        fullScreenButton.setImageResource(R.drawable.ic_fullscreen)
        isFullScreen = false
    }

    private fun hideControls() {
        hideSystemBars()
        hideToolbar(true)
        fullScreenButton.setImageResource(R.drawable.ic_fullscreen_exit)
        isFullScreen = true
    }

    private fun hideToolbar(shouldHide: Boolean) {
        binding.toolbar.visibility = if (shouldHide) View.GONE else View.VISIBLE
    }

    override fun onSupportNavigateUp(): Boolean {
        onBackPressed()
        return super.onSupportNavigateUp()
    }

    override fun onBackPressed() {
        if (isLandscapeOrientation()) {
            setPortraitOrientation()
        } else {
            setResult(
                Activity.RESULT_OK,
                getResponseIntent(
                    playerViewModel.getCurrentPosition(),
                    playerViewModel.getPlayWhenReady(),
                    playerViewModel.getPlaybackSpeed(),
                    playerViewModel.getUrl()
                )
            )
            playerViewModel.stopPlaying()
            super.onBackPressed()
        }
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            hideControls()
        } else if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
            showControls()
        }
    }

    companion object {
        const val VIDEO_URL_TAG = "ExoPlayer_Video_Url"
        const val VIDEO_TITLE_TAG = "ExoPlayer_Video_Title"
        const val VIDEO_SUBTITLE_TAG = "ExoPlayer_Video_Subtitle"
        const val VIDEO_CURRENT_POSITION_TAG = "ExoPlayer_Current_Position_Subtitle"
        const val VIDEO_PLAY_WHEN_READY_TAG = "ExoPlayer_Play_When_Ready_Subtitle"
        const val VIDEO_PLAYBACK_SPEED_TAG = "ExoPlayer_Playback_Speed"

        fun getRequestIntent(
            activity: Activity,
            currentPosition: Long,
            playWhenReady: Boolean,
            speed: Float,
            url: String
        ): Intent {
            return Intent(activity, ExoPlayerActivity::class.java).apply {
                putExtra(
                    VIDEO_CURRENT_POSITION_TAG,
                    currentPosition
                )
                putExtra(
                    VIDEO_PLAY_WHEN_READY_TAG,
                    playWhenReady
                )
                putExtra(VIDEO_URL_TAG, url)
                putExtra(VIDEO_PLAYBACK_SPEED_TAG, speed)
            }
        }

        fun getResponseIntent(
            currentPosition: Long,
            playWhenReady: Boolean,
            speed: Float,
            url: String?
        ): Intent {
            return Intent().apply {
                putExtra(
                    VIDEO_CURRENT_POSITION_TAG,
                    currentPosition
                )
                putExtra(
                    VIDEO_PLAY_WHEN_READY_TAG,
                    playWhenReady
                )
                putExtra(VIDEO_URL_TAG, url)
                putExtra(VIDEO_PLAYBACK_SPEED_TAG, speed)
            }
        }
    }
}