package ai.engagely.openbot.view.activities

import ai.engagely.openbot.R
import ai.engagely.openbot.databinding.ActivityYoutubePlayerBinding
import ai.engagely.openbot.model.utils.exts.showToast
import android.content.res.Configuration
import android.os.Bundle
import android.view.View
import com.pierfrancescosoffritti.androidyoutubeplayer.core.player.YouTubePlayer
import com.pierfrancescosoffritti.androidyoutubeplayer.core.player.listeners.AbstractYouTubePlayerListener
import com.pierfrancescosoffritti.androidyoutubeplayer.core.player.listeners.YouTubePlayerListener
import com.pierfrancescosoffritti.androidyoutubeplayer.core.player.options.IFramePlayerOptions
import com.pierfrancescosoffritti.androidyoutubeplayer.core.ui.DefaultPlayerUiController

class YouTubePlayerActivity : BaseActivity() {

    private lateinit var binding: ActivityYoutubePlayerBinding
    private var isFullScreen = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        binding = ActivityYoutubePlayerBinding.inflate(layoutInflater)
        setContentView(binding.root)

        setupActionBar()
        setupPlayer()
    }

    private fun setupActionBar() {
        setSupportActionBar(binding.toolbar)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.setHomeAsUpIndicator(R.drawable.ic_arrow_left_white)
        supportActionBar?.title = intent.getStringExtra(VIDEO_TITLE_TAG) ?: ""
        supportActionBar?.subtitle =
            intent.getStringExtra(VIDEO_SUBTITLE_TAG) ?: ""
    }

    private fun setupPlayer() {
        intent.getStringExtra(VIDEO_ID_TAG)?.let { videoId ->
            lifecycle.addObserver(binding.ytPlayer)

            val listener: YouTubePlayerListener = object : AbstractYouTubePlayerListener() {
                override fun onReady(youTubePlayer: YouTubePlayer) {
                    // using pre-made custom ui
                    binding.ytPlayer.let {
                        val defaultPlayerUiController =
                            DefaultPlayerUiController(it, youTubePlayer)
                        defaultPlayerUiController.showMenuButton(false)
                        defaultPlayerUiController.showFullscreenButton(true)
                        defaultPlayerUiController.setFullScreenButtonClickListener {
                            toggleFullScreen()
                        }
                        it.setCustomPlayerUi(defaultPlayerUiController.rootView)

                    }
                    youTubePlayer.loadVideo(videoId, 0F)
                    hideCurtain()
                }
            }

            val options: IFramePlayerOptions = IFramePlayerOptions.Builder().controls(0).build()
            binding.ytPlayer.initialize(listener, options)
        } ?: run {
            showToast(getString(R.string.failed_to_open_link))
            finish()
        }
    }

    private fun hideCurtain() {
        binding.playerCurtain.visibility = View.GONE
    }

    private fun toggleFullScreen() {
        if (!isFullScreen) {
            hideControls()
        } else {
            showControls()
        }
    }

    private fun showControls() {
        unhideSystemBars()
        hideToolbar(false)
        isFullScreen = false
    }

    private fun hideControls() {
        hideSystemBars()
        hideToolbar(true)
        isFullScreen = true
    }

    private fun hideToolbar(shouldHide: Boolean) {
        binding.toolbar.visibility = if (shouldHide) View.GONE else View.VISIBLE
    }

    override fun onSupportNavigateUp(): Boolean {
        onBackPressed()
        return super.onSupportNavigateUp()
    }

    override fun onBackPressed() {
        if (isLandscapeOrientation()) {
            setPortraitOrientation()
        } else {
            super.onBackPressed()
        }
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            hideControls()
        } else if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
            showControls()
        }
    }

    companion object {
        const val VIDEO_ID_TAG = "YouTubePlayer_Video_Id"
        const val VIDEO_TITLE_TAG = "YouTubePlayer_Video_Title"
        const val VIDEO_SUBTITLE_TAG = "YouTubePlayer_Video_Subtitle"
    }
}