package ai.engagely.openbot.view.customviews

import ai.engagely.openbot.R
import ai.engagely.openbot.databinding.ViewFormTextFieldBinding
import ai.engagely.openbot.model.constants.ServerConstants
import ai.engagely.openbot.model.pojos.internal.history.IFormTextItem
import ai.engagely.openbot.model.utils.validators.FormItemValidator
import android.content.Context
import android.text.Editable
import android.text.InputType
import android.text.TextWatcher
import android.util.AttributeSet
import android.view.LayoutInflater
import android.widget.FrameLayout
import androidx.core.content.ContextCompat

class FormTextFieldView : FrameLayout {

    private var binding: ViewFormTextFieldBinding? = null
    private var textWatcher: TextWatcher? = null
    private var prefixTextWatcher: TextWatcher? = null

    constructor(context: Context) : super(context) {
        initUi()
    }

    constructor(context: Context, attrs: AttributeSet) : super(context, attrs) {
        initUi()
    }

    constructor(context: Context, attrs: AttributeSet, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        initUi()
    }

    private fun initUi() {
        binding = ViewFormTextFieldBinding.inflate(LayoutInflater.from(context), this, true)
    }

    fun setData(iFormTextItem: IFormTextItem?, isLightTheme: Boolean = false) {
        updateTheme(isLightTheme)
        binding?.let { rootView ->
            iFormTextItem?.let { textItem ->
                rootView.tvLabel.text = textItem.labelName
                rootView.etText.hint = textItem.placeHolder
                rootView.etText.setText(textItem.value)
                setError(textItem)

                when (textItem.textBoxTypeSelected) {
                    ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_TEXT -> {
                        rootView.etText.inputType = InputType.TYPE_CLASS_TEXT
                        rootView.etTextPrefix.visibility = GONE
                    }
                    ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_NUMBER -> {
                        rootView.etText.inputType = InputType.TYPE_CLASS_NUMBER
                        rootView.etTextPrefix.visibility = GONE
                    }
                    ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_EMAIL -> {
                        rootView.etText.inputType =
                            InputType.TYPE_CLASS_TEXT or InputType.TYPE_TEXT_VARIATION_EMAIL_ADDRESS
                        rootView.etTextPrefix.visibility = GONE
                    }
                    ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_PHONE_NUMBER -> {
                        rootView.etText.inputType =
                            InputType.TYPE_CLASS_PHONE or InputType.TYPE_TEXT_VARIATION_PHONETIC
                        rootView.etTextPrefix.inputType =
                            InputType.TYPE_CLASS_PHONE or InputType.TYPE_TEXT_VARIATION_PHONETIC
                        rootView.etTextPrefix.setText(textItem.prefixValue)
                        rootView.etTextPrefix.visibility = VISIBLE
                    }
                    ServerConstants.DIALOGUE_FORM_TEXT_FIELD_TYPE_PASSWORD -> {
                        rootView.etText.inputType =
                            InputType.TYPE_CLASS_TEXT or InputType.TYPE_TEXT_VARIATION_PASSWORD
                        rootView.etTextPrefix.visibility = GONE
                    }
                    else -> {
                        rootView.etText.inputType = InputType.TYPE_CLASS_TEXT
                        rootView.etTextPrefix.visibility = GONE
                    }
                }
            }

            textWatcher?.let {
                rootView.etText.removeTextChangedListener(it)
            }
            textWatcher = object : TextWatcher {
                override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {}

                override fun onTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {
                    iFormTextItem?.apply {
                        value = p0?.toString() ?: ""
                        FormItemValidator.validateFormItem(this, context)
                        setError(this)
                    }
                }

                override fun afterTextChanged(p0: Editable?) {}
            }
            rootView.etText.addTextChangedListener(textWatcher)

            prefixTextWatcher?.let {
                rootView.etTextPrefix.removeTextChangedListener(it)
            }
            prefixTextWatcher = object : TextWatcher {
                override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {}

                override fun onTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {
                    iFormTextItem?.apply {
                        prefixValue = p0?.toString() ?: ""
                        FormItemValidator.validateFormItem(this, context)
                        setError(this)
                    }
                }

                override fun afterTextChanged(p0: Editable?) {}
            }
            rootView.etTextPrefix.addTextChangedListener(prefixTextWatcher)
        }
    }

    private fun updateTheme(isLightTheme: Boolean) {
        val labelColor: Int
        val editTextDrawable: Int
        val errorColor: Int
        if (isLightTheme) {
            labelColor = R.color.white
            editTextDrawable = R.drawable.shape_form_edit_text_bg_border_light
            errorColor = R.color.white
        } else {
            labelColor = R.color.grey
            editTextDrawable = R.drawable.shape_form_edit_text_bg_border
            errorColor = R.color.red
        }
        binding?.tvLabel?.setTextColor(ContextCompat.getColor(context, labelColor))
        binding?.etTextPrefix?.background = ContextCompat.getDrawable(context, editTextDrawable)
        binding?.etText?.background = ContextCompat.getDrawable(context, editTextDrawable)
        binding?.tvError?.setTextColor(ContextCompat.getColor(context, errorColor))
    }

    private fun setError(textItem: IFormTextItem) {
        binding?.tvError?.visibility =
            if (!textItem.errorMessage.isNullOrBlank()) VISIBLE else GONE
        binding?.tvError?.text = textItem.errorMessage
    }
}