package ai.engagely.openbot.viewmodel

import ai.engagely.openbot.R
import ai.engagely.openbot.model.constants.AppConstants
import ai.engagely.openbot.model.constants.ServerConstants
import ai.engagely.openbot.model.localstorage.SimpleKeyValueStorage
import ai.engagely.openbot.model.localstorage.impl.PreferenceDataStore
import ai.engagely.openbot.model.network.ApiConstants
import ai.engagely.openbot.model.pojos.external.apiresponses.chat.ChatResponse
import ai.engagely.openbot.model.pojos.external.apiresponses.chat.LiveChatResponse
import ai.engagely.openbot.model.pojos.internal.botfeedback.IFeedbackJourney
import ai.engagely.openbot.model.pojos.internal.botsettings.*
import ai.engagely.openbot.model.pojos.internal.chat.*
import ai.engagely.openbot.model.pojos.internal.common.IBotFeedbackNegativeClick
import ai.engagely.openbot.model.pojos.internal.common.IBotFeedbackPositiveClick
import ai.engagely.openbot.model.pojos.internal.common.ISendJourney
import ai.engagely.openbot.model.pojos.internal.common.ISubmitTabularResponse
import ai.engagely.openbot.model.pojos.internal.faqautocomplete.IFaqAutoCompleteItem
import ai.engagely.openbot.model.pojos.internal.history.*
import ai.engagely.openbot.model.pojos.internal.livechat.*
import ai.engagely.openbot.model.pojos.internal.location.ILocation
import ai.engagely.openbot.model.repositories.*
import ai.engagely.openbot.model.repositories.impl.*
import ai.engagely.openbot.model.utils.converters.BotConfigConverter
import ai.engagely.openbot.model.utils.converters.ChatResponseConverter
import ai.engagely.openbot.model.utils.exts.isValidEmail
import ai.engagely.openbot.model.utils.exts.notifyObservers
import ai.engagely.openbot.model.utils.exts.obtainNonBlankOrNull
import ai.engagely.openbot.model.utils.exts.showToast
import ai.engagely.openbot.model.utils.general.*
import ai.engagely.openbot.model.utils.helpers.TextToSpeechHelper
import ai.engagely.openbot.view.adapters.ChatItemsRecyclerViewAdapter
import ai.engagely.openbot.view.customviews.FormFileUploadView
import android.app.Application
import android.content.ContentResolver
import android.graphics.Bitmap
import android.net.Uri
import android.os.Build
import android.os.Bundle
import androidx.lifecycle.*
import com.google.gson.Gson
import kotlinx.coroutines.*
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.channels.consumeEach
import java.io.File
import java.util.*

class BotChatViewModel(application: Application) : SocketViewModel(application),
    NetworkStateAwareViewModel.NetworkStateChangeListener,
    TextToSpeechHelper.OnSpeechCompletedListener {

    private var screenWidth: Int? = null
    private var screenHeight: Int? = null
    private var isLandscapeOrientation: Boolean? = null
    private var isLoadingMore: Boolean = false
    private lateinit var updatedBotConfigData: IBotChatWindowConfig
    private var extras: Bundle? = null

    private val simpleKeyValueStorage: SimpleKeyValueStorage by lazy {
        PreferenceDataStore(application)
    }

    private val sessionRepository: SessionRepository by lazy {
        AppSessionRepository(Dispatchers.Default, Dispatchers.IO, simpleKeyValueStorage)
    }
    private val historyRepository: ChatHistoryRepository by lazy {
        AppChatHistoryRepository(Dispatchers.IO, ChatResponseConverter(), liveChatRepository)
    }
    private val formRepository: FormRepository by lazy {
        AppFormRepository(Dispatchers.IO)
    }
    private val botSettingsRepository: BotSettingRepository by lazy {
        AppBotSettingsRepository(Dispatchers.IO)
    }

    private val languageRepository: LanguageRepository by lazy {
        AppLanguageRepository(Dispatchers.Default, simpleKeyValueStorage)
    }

    private val botFeedbackRepository: BotFeedbackRepository by lazy {
        AppBotFeedbackRepository(Dispatchers.Default)
    }

    private val chatResponseConverter: ChatResponseConverter by lazy {
        ChatResponseConverter()
    }

    private val botConfigConverter: BotConfigConverter by lazy {
        BotConfigConverter()
    }

    private val textToSpeechHelper: TextToSpeechHelper by lazy {
        TextToSpeechHelper(this)
    }

    private val chatItemsFromGreeting = ArrayList<IChatItem>()
    private val chatItemsFromConfig = ArrayList<IChatItem>()
    private val chatItemsFromHistory = ArrayList<IChatItem>(100)
    private val chatItemsFromUser = ArrayList<IChatItem>(100)
    private val _chatItemsLiveData = MutableLiveData<IChatItemsContainer>()
    val chatItemsLiveData: LiveData<IChatItemsContainer> = _chatItemsLiveData

    private val _greetingMessageLiveData = MutableLiveData<IGreetingMessageWithHeader>()
    val greetingMessageLiveData: LiveData<IGreetingMessageWithHeader> = _greetingMessageLiveData

    private val _historyErrorStateLiveData = MutableLiveData<ErrorState?>()
    val historyErrorStateLiveData: LiveData<ErrorState?> = _historyErrorStateLiveData

    private val _isHistoryLoadingLiveData = MutableLiveData<Boolean>()
    val isHistoryLoadingLiveData: LiveData<Boolean> = _isHistoryLoadingLiveData

    private val _botConfigLiveData = MutableLiveData<IBotChatWindowConfig>()
    val botConfigLiveData: LiveData<IBotChatWindowConfig> = _botConfigLiveData

    val actionBarTitleLiveData: LiveData<String> =
        Transformations.map(_botConfigLiveData, this::prepareActionbarTitle)

    val themeColorLiveData: LiveData<String> =
        Transformations.map(_botConfigLiveData, this::prepareThemeColor)

    val bgColorLiveData: LiveData<String> =
        Transformations.map(_botConfigLiveData, this::prepareBgColor)

    val headerTextColorLiveData: LiveData<String> =
        Transformations.map(_botConfigLiveData, this::prepareHeaderTextColor)

    val actionBarLogoLiveData: LiveData<Bitmap> =
        Transformations.map(_botConfigLiveData, this::prepareActionbarLogo)

    val quickMenuLiveData: LiveData<List<IQuickMenuItem>> =
        Transformations.map(_botConfigLiveData, this::prepareQuickMenu)

    private val _currentLanguageLiveData = MutableLiveData<ILanguage?>()
    val currentLanguageLiveData: LiveData<ILanguage?> = _currentLanguageLiveData

    private val _isLanguageLoadingLiveData = MutableLiveData<Boolean>()
    val isLanguageLoadingLiveData: LiveData<Boolean> = _isLanguageLoadingLiveData

    private val _liveChatStatusLiveData = MutableLiveData<ILiveChatDataContainer.ILiveChatStatus>()
    val liveChatStatusLiveData: LiveData<ILiveChatDataContainer.ILiveChatStatus> =
        _liveChatStatusLiveData

    private val _liveChatMediaLiveData = MutableLiveData<ILiveChatMediaData>()
    val liveChatMediaLiveData: LiveData<ILiveChatMediaData> = _liveChatMediaLiveData

    private val _userFeedbackLiveData = MutableLiveData<ILiveChatFeedbackRequest>()
    val userFeedbackLiveData: LiveData<ILiveChatFeedbackRequest> = _userFeedbackLiveData

    private val _enableBotFeedbackLiveData = MutableLiveData<IBotFeedbackRequest>()
    val enableBotFeedbackLiveData: LiveData<IBotFeedbackRequest> = _enableBotFeedbackLiveData

    private val _botFeedbackLiveData = MutableLiveData<IBotFeedbackRequest>()
    val botFeedbackLiveData: LiveData<IBotFeedbackRequest> = _botFeedbackLiveData

    private var previousLanguage: ILanguage? = null
    var allLanguagesData: List<ILanguage>? = null

    private var iBotChatWindowConfig: IBotChatWindowConfig? = null
    private var greetingJourneyFailed: Boolean? = null
    private var greetingJourney: String? = null

    private var liveChatData: ILiveChatData? = null
    private var liveChatStatus: ILiveChatDataContainer.ILiveChatStatus? = null

    private var showFeedbackAfterSecondsJob: Job? = null

    private val tabularInfoJobChannel = Channel<Job>(capacity = Channel.UNLIMITED).apply {
        viewModelScope.launch(Dispatchers.Default) {
            consumeEach {
                it.join()
            }
        }
    }

    init {
        setNetworkStateChangeListener(this)
    }

    fun setInitialData(
        iBotChatWindowConfig: IBotChatWindowConfig?,
        extras: Bundle?,
        lifecycle: Lifecycle
    ) {
        this.extras = extras
        lifecycle.addObserver(textToSpeechHelper)
        viewModelScope.launch(Dispatchers.Default) {
            initBotConfigData(iBotChatWindowConfig, true)
            initGreetingMessagesWithHeader(iBotChatWindowConfig)
            initLanguageData(iBotChatWindowConfig)
            fetchHistory()
        }
    }

    fun updateScreenConfigurationData(
        screenWidth: Int,
        screenHeight: Int,
        isLandscapeOrientation: Boolean
    ) {
        this.screenWidth = screenWidth
        this.screenHeight = screenHeight
        this.isLandscapeOrientation = isLandscapeOrientation
    }

    fun initChat() {
        initSocket()
    }

    private suspend fun initLanguageData(iBotChatWindowConfig: IBotChatWindowConfig?) {
        _currentLanguageLiveData.postValue(prepareCurrentLanguage(iBotChatWindowConfig))
    }

    private suspend fun initBotConfigData(
        iBotChatWindowConfig: IBotChatWindowConfig?,
        scrollToEnd: Boolean
    ) {
        greetingJourney = iBotChatWindowConfig?.greetingJourney
        allLanguagesData = iBotChatWindowConfig?.languageData

        this.iBotChatWindowConfig = iBotChatWindowConfig
        if (iBotChatWindowConfig?.enableSpeech == true) {
            textToSpeechHelper.init(getApplication())
        }
        iBotChatWindowConfig?.defaultILanguage?.languageCode?.let {
            languageRepository.setDefaultLanguage(languageCode = it)
        }
        _botConfigLiveData.postValue(iBotChatWindowConfig)

        chatItemsFromConfig.clear()
        botConfigConverter.convertToInitialGreetingItems(iBotChatWindowConfig).let {
            chatItemsFromConfig.addAll(it)
        }
        notifyChatItemsChange(scrollToEnd)
    }

    private fun initGreetingMessagesWithHeader(iBotChatWindowConfig: IBotChatWindowConfig?) {
        _greetingMessageLiveData.postValue(iBotChatWindowConfig?.greetingMessageWithHeader)
    }

    private fun initGreetingMessagesWithHeaderChatItems(iBotChatWindowConfig: IBotChatWindowConfig?) {
        botConfigConverter.convertToPostChatGreetingItems(iBotChatWindowConfig)?.let {
            chatItemsFromGreeting.clear()
            chatItemsFromGreeting.addAll(it)
        }
    }

    private fun prepareActionbarTitle(iBotChatWindowConfig: IBotChatWindowConfig?): String {
        return if (ILiveChatDataContainer.ILiveChatStatus.CONNECTED == liveChatStatus)
            liveChatData?.agentName ?: ""
        else iBotChatWindowConfig?.toolbarTitle ?: ""
    }

    private fun prepareThemeColor(iBotChatWindowConfig: IBotChatWindowConfig?): String {
        return iBotChatWindowConfig?.botThemColor ?: ""
    }

    private fun prepareBgColor(iBotChatWindowConfig: IBotChatWindowConfig?): String {
        return iBotChatWindowConfig?.botBgColor ?: ""
    }

    private fun prepareHeaderTextColor(iBotChatWindowConfig: IBotChatWindowConfig?): String {
        return iBotChatWindowConfig?.headerTextColor ?: ""
    }

    private fun prepareActionbarLogo(iBotChatWindowConfig: IBotChatWindowConfig?): Bitmap? {
        return iBotChatWindowConfig?.logo
    }

    private suspend fun prepareCurrentLanguage(iBotChatWindowConfig: IBotChatWindowConfig?): ILanguage? {
        languageRepository.getCurrentLanguage()?.let { currentLanguage ->
            return ILanguage(languageCode = currentLanguage, languageName = null)
        }
        if ((iBotChatWindowConfig?.languageData?.size ?: 0) > 1) {
            return iBotChatWindowConfig?.defaultILanguage
        }
        return null
    }

    private fun prepareQuickMenu(iBotChatWindowConfig: IBotChatWindowConfig?): List<IQuickMenuItem>? {
        return iBotChatWindowConfig?.quickMenuItems
    }

    fun hasQuickMenu(): Boolean = iBotChatWindowConfig?.quickMenuItems?.isNullOrEmpty() == false

    fun getQuickMenuPosition(): IQuickMenuPosition? = iBotChatWindowConfig?.quickMenuPosition

    fun isAutoCompleteEnabled(): Boolean = iBotChatWindowConfig?.autoCompleteEnabled ?: false

    fun isEnglishLanguageSelected(): Boolean {
        return "en" == _currentLanguageLiveData.value?.languageCode
    }

    fun isMicEnabled(): Boolean = iBotChatWindowConfig?.enableMic ?: false

    fun processLanguageSelection(position: Int, title: CharSequence?, languageCode: String?) {
        if (!isInternetAvailable()) {
            return
        }
        viewModelScope.launch(Dispatchers.Default) {
            updateLanguage(position, title, languageCode)
        }
    }

    private suspend fun updateLanguage(
        position: Int,
        languageName: CharSequence?,
        languageCode: String?
    ): Boolean {
        iBotChatWindowConfig?.languageData?.let { languageData ->
            if (position >= 0 && position < languageData.size) {
                return updateLanguage(languageData[position])
            } else if (!languageName.isNullOrBlank()) {
                languageData.find { it.languageName == languageName }?.let { selectedLanguage ->
                    return updateLanguage(selectedLanguage)
                }
            } else if (!languageCode.isNullOrBlank()) {
                languageData.find { it.languageCode == languageCode }?.let { selectedLanguage ->
                    return updateLanguage(selectedLanguage)
                }
            } else {
                return false
            }
        }
        return false
    }

    private suspend fun updateLanguage(selectedLanguage: ILanguage): Boolean {
        previousLanguage = _currentLanguageLiveData.value
        _currentLanguageLiveData.postValue(selectedLanguage)
        _isLanguageLoadingLiveData.postValue(true)
        return withContext(Dispatchers.Default) {
            try {
                iBotChatWindowConfig?.botId?.let { botId ->
                    selectedLanguage.languageCode?.let { languageCode ->
                        botSettingsRepository.getBotSettings(botId, languageCode)
                            ?.let {
                                initBotConfigData(it, false)
                                updateGreetingMessagesWithHeader()
                                languageRepository.setCurrentLanguage(languageCode)
                                _isLanguageLoadingLiveData.postValue(false)
                                updatedBotConfigData = it
                                return@withContext true
                            }
                    }
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            _currentLanguageLiveData.postValue(previousLanguage)
            _isLanguageLoadingLiveData.postValue(false)
            showToast(getApplication<Application>().resources.getString(R.string.failed_to_change_language))
            return@withContext false
        }
    }

    private fun updateGreetingMessagesWithHeader() {
        if (_greetingMessageLiveData.value == null) {
            initGreetingMessagesWithHeaderChatItems(iBotChatWindowConfig)
        } else {
            initGreetingMessagesWithHeader(iBotChatWindowConfig)
        }
    }

    private fun fetchHistory() {
        if (!isInternetAvailable()) {
            checkHistoryFailure()
            return
        }
        viewModelScope.launch(Dispatchers.IO) {
            try {
                setHistoryError(null)
                setHistoryLoading(true)
                if (isHistoryChatItemPresent()) {
                    return@launch
                }

                val getHistory = async {
                    val sessionId = sessionRepository.getSessionId()
                    val userId = sessionRepository.getUserId()
                    historyRepository.getChatHistory(
                        userId = userId,
                        sessionId = sessionId,
                        intId = iBotChatWindowConfig?.intId,
                        extras = extras
                    )
                }
                val connectToSocket = async { connectToSocket() }

                connectToSocket.await()
                val history = getHistory.await()
                syncBotFeedbackShownStatus(history?.isBotFeedbackShown)
                setHistoryLoading(false)
                when {
                    history == null -> {
                        checkHistoryFailure()
                        processGreetingJourney(iBotChatWindowConfig, true)
                    }
                    history.historyItems?.isNotEmpty() == true -> {
                        moveGreetingMessagesToChatItemsIfPresent()
                        addHistoryItemsToChat(history)
                    }
                    else -> {
                        processGreetingJourney(iBotChatWindowConfig, true)
                    }
                }
            } catch (e: Exception) {
                checkHistoryFailure()
                setHistoryLoading(false)
            }
        }
    }

    private fun checkHistoryFailure() {
        setHistoryError(
            if (NetworkUtils.isInternetAvailable(getApplication())) {
                ErrorState.SERVER_FAILURE
            } else {
                ErrorState.NETWORK_FAILURE
            }
        )
    }

    private fun setHistoryLoading(isLoading: Boolean) {
        _isHistoryLoadingLiveData.postValue(isLoading)
    }

    private fun isHistoryChatItemPresent() = chatItemsFromHistory.isNotEmpty()

    private fun setHistoryError(errorState: ErrorState?) {
        _historyErrorStateLiveData.postValue(errorState)
    }

    private fun processGreetingJourney(
        iBotChatWindowConfig: IBotChatWindowConfig?,
        startAfterDelay: Boolean = false
    ) {
        viewModelScope.launch(Dispatchers.Default) {
            greetingJourney?.obtainNonBlankOrNull()?.let {
                if (iBotChatWindowConfig?.enableGreetingJourney == true) {
                    if (!isInternetAvailable()) {
                        greetingJourneyFailed = true
                        return@let
                    }
                    if (!isSocketConnected()) {
                        greetingJourneyFailed = true
                        return@let
                    }
                    viewModelScope.launch(Dispatchers.IO) {
                        try {
                            setLoading(true)
                            if (startAfterDelay) {
                                delay(AppConstants.GREETING_JOURNEY_INITIAL_DELAY)
                            }
                            val sent = sendMessage(
                                channelName = ApiConstants.CHANNEL_QUICK_JOURNEY,
                                message = it,
                                fieldText = it,
                                isFromProcessTree = true,
                                queryType = if (iBotChatWindowConfig.enableSpeech)
                                    ApiConstants.CHAT_QUERY_TYPE_VOICE
                                else ApiConstants.CHAT_QUERY_TYPE_BUTTON
                            )
                            if (sent) {
                                greetingJourneyFailed = null
                                greetingJourney = null
                            }
                            setLoading(false)
                        } catch (e: Exception) {
                            setLoading(false)
                        }
                    }
                }
            }
        }
    }

    override fun storeSocketId(id: String?) {
        viewModelScope.launch(Dispatchers.Default) {
            sessionRepository.storeSocketId(id)
        }
    }

    override fun storeLiveChatSocketId(id: String?) {
        viewModelScope.launch(Dispatchers.Default) {
            sessionRepository.storeLiveChatSocketId(id)
        }
    }

    override fun onCleared() {
        super.onCleared()
        clearFormFilesIfPresent()
    }

    private fun clearFormFilesIfPresent() {
        getChatItems().forEach { chatItem ->
            if (chatItem is IChatFormItem) {
                chatItem.formItemContainers.forEach { formItemsContainer ->
                    formItemsContainer.formsItems.forEach { formItem ->
                        if (formItem is IFormFileItem) {
                            FileUtils.deleteParentAndSelfIfExists(formItem.file)
                        }
                    }
                }
            }
        }
    }

    fun sendMessageFromUserTextInput(message: String) {
        if (liveChatStatus == ILiveChatDataContainer.ILiveChatStatus.CONNECTED) {
            viewModelScope.launch(Dispatchers.IO) {
                sendLiveChatMessage(message, ApiConstants.CHAT_EXPECTED_RESPONSE_TYPE_TEXT)
            }
        } else {
            moveGreetingMessagesAndSendMessage(message)
        }
    }

    private fun moveGreetingMessagesAndSendMessage(message: String) {
        if (message.isBlank() || message.isBlank()) return

        if (!isInternetAvailable()) {
            return
        }

        moveGreetingMessagesToChatItemsIfPresent()

        addAndSendMessage(
            channelName = ApiConstants.CHANNEL_CHAT,
            messageToSend = message,
            messageToDisplay = message
        )
    }

    private suspend fun sendLiveChatMessage(message: Any, responseType: String?): Boolean {
        if (message is String && message.isBlank()) return false

        if (!isInternetAvailable()) {
            return false
        }
        return withContext(Dispatchers.Default) {
            try {
                val sessionId = sessionRepository.getSessionId()
                val userId = sessionRepository.getUserId()
                val sessionObj = sessionRepository.getSessionObj()
                val sent = liveChatRepository.sendMessage(
                    liveChatData = liveChatData,
                    channel = "bot",
                    intId = iBotChatWindowConfig?.intId,
                    intName = iBotChatWindowConfig?.intName,
                    prevLangCode = languageRepository.getPreviousLanguage(),
                    queryType = ApiConstants.CHAT_QUERY_TYPE_TEXT,
                    sessionId = sessionId,
                    sessionObj = sessionObj,
                    userId = userId,
                    userInput = message,
                    userLang = languageRepository.getCurrentLanguage(),
                    userLangCode = languageRepository.getCurrentLanguage(),
                    responseType = responseType
                )
                if (!sent) {
                    showToast(getApplication<Application>().resources.getString(R.string.failed_to_send))
                }
                return@withContext sent
            } catch (e: Exception) {
                LogUtils.logException(e)
                return@withContext false
            }
        }
    }

    private fun addAndSendMessage(
        channelName: String,
        messageToSend: String,
        messageToDisplay: String,
        fieldText: String? = null,
        isFromProcessTree: Boolean? = null,
        queryType: String? = ApiConstants.CHAT_QUERY_TYPE_TEXT,
    ) {
        if (messageToSend.isBlank() || messageToDisplay.isBlank()) return

        if (!isInternetAvailable()) {
            return
        }

        viewModelScope.launch(Dispatchers.Default) {
            val addedMessage = addMessageItemToChat(messageToDisplay, false)
            try {
                setLoading(true)
                val sent =
                    sendMessage(
                        channelName,
                        messageToSend,
                        fieldText,
                        isFromProcessTree,
                        queryType
                    )
                if (!sent) {
                    val pendingMessageItem = IPendingMessageItem(
                        channelName,
                        messageToSend,
                        fieldText,
                        isFromProcessTree,
                        queryType
                    )
                    setPendingData(addedMessage?.uuid, pendingMessageItem)
                }
                setLoading(false)
            } catch (e: Exception) {
                LogUtils.logException(e)
                setLoading(false)
            }
        }
    }

    private suspend fun sendMessage(
        channelName: String,
        message: String,
        fieldText: String?,
        isFromProcessTree: Boolean?,
        queryType: String?
    ): Boolean {
        val sessionId = sessionRepository.getSessionId()
        val userId = sessionRepository.getUserId()
        val sessionObj = sessionRepository.getSessionObj()
        var sent = false
        if (sessionObj != null) {
            sent = chatRepository.sendMessage(
                channelName = channelName,
                userInput = message,
                fieldText = fieldText,
                isFromProcessTree = isFromProcessTree,
                queryType = queryType,
                botId = iBotChatWindowConfig?.botId,
                intId = iBotChatWindowConfig?.intId,
                intName = iBotChatWindowConfig?.intName,
                sessionId = sessionId,
                sessionObj = sessionObj,
                userId = userId,
                prevLangCode = languageRepository.getPreviousLanguage(),
                userLang = languageRepository.getCurrentLanguage()
            )
            if (sent) {
                languageRepository.updatePreviousWithCurrent()
            }
        }
        showBotFeedbackBasedOnMessagesCount()
        showBotFeedbackBasedOnSeconds(skipIfNotAlreadyRun = false)
        moveGreetingMessagesToChatItemsIfPresent()
        return sent
    }

    private suspend fun sendFlow(
        dynamicSelectedField: String?,
        inputId: String?,
        intId: String?,
        journeyId: String?,
        messageType: String?,
        msgId: String?,
        nodeId: Int?,
        queryType: String?,
        selectedField: String?,
        userInput: String?,
        languageSelection: Boolean,
        languageCode: String?
    ): Boolean {
        val sessionId = sessionRepository.getSessionId()
        val sessionObj = sessionRepository.getSessionObj()
        var sent = false
        if (languageSelection) {
            sent = updateLanguage(-1, null, languageCode)
        }
        if (sessionObj != null) {
            sent = chatRepository.sendFlow(
                dynamicSelectedField = dynamicSelectedField,
                inputId = inputId,
                intId = intId,
                journeyId = journeyId,
                languageCode = languageRepository.getCurrentLanguage(),
                messageType = messageType,
                msgId = msgId,
                nodeId = nodeId,
                queryType = queryType,
                selectedField = selectedField,
                sessionId = sessionId,
                sessionObj = sessionObj,
                userInput = userInput
            )
        }
        showBotFeedbackBasedOnMessagesCount()
        showBotFeedbackBasedOnSeconds(skipIfNotAlreadyRun = false)
        moveGreetingMessagesToChatItemsIfPresent()
        return sent
    }

    private fun moveGreetingMessagesToChatItemsIfPresent() {
        if (_greetingMessageLiveData.value != null) {
            _greetingMessageLiveData.postValue(null)
            initGreetingMessagesWithHeaderChatItems(iBotChatWindowConfig)
        }
    }

    private fun setLoading(loading: Boolean) {
        this.isLoadingMore = loading
        notifyChatItemsChange(true)
    }

    override fun onNewSocketResponseReceived(channel: String, response: String) {
        super.onNewSocketResponseReceived(channel, response)
        when (channel) {
            ApiConstants.CHANNEL_CHAT -> {
                processNewChatResponse(response)
            }
            ApiConstants.CHANNEL_TAB_INFO -> {
                processNewTabularInfoResponse(response)
            }
            ApiConstants.CHANNEL_USER_REPLY,
            ApiConstants.CHANNEL_LIVECHAT_USER,
            ApiConstants.CHANNEL_LIVE_USER_DISCONNECT_STATUS -> {
                processUserReply(response, channel)
            }
        }
        showBotFeedbackBasedOnSeconds(skipIfNotAlreadyRun = false)
    }

    private fun processUserReply(response: String, channel: String) {
        viewModelScope.launch(Dispatchers.Default) {
            sessionRepository.getSessionId()?.let { sessionId ->
                chatResponseConverter.convertNewUserReplyToChatItems(response, sessionId)
                    .let { liveChatDataContainer ->
                        liveChatDataContainer.items?.let {
                            addNewItemsToChat(it)
                        }

                        if (liveChatDataContainer.status != ILiveChatDataContainer.ILiveChatStatus.IDLE) {
                            if (ApiConstants.CHANNEL_USER_REPLY == channel
                                || ApiConstants.CHANNEL_LIVE_USER_DISCONNECT_STATUS == channel
                            ) {
                                if (liveChatDataContainer.sendJoinMeHereRequest) {
                                    liveChatDataContainer.data?.let {
                                        liveChatRepository.sendJoinMeHereRequest(it)
                                    }
                                }

                                liveChatData = liveChatDataContainer.data
                                //dirty fix for chat transfer issue
                                if (liveChatData?.userIntName == null) {
                                    liveChatData?.userIntName = iBotChatWindowConfig?.intName
                                }
                                _botConfigLiveData.notifyObservers()

                                liveChatStatus = liveChatDataContainer.status
                                _liveChatStatusLiveData.postValue(liveChatStatus)
                                saveLiveChatDetails()
                            }
                        }


                        if (liveChatStatus == ILiveChatDataContainer.ILiveChatStatus.DISCONNECTED) {
                            disconnectLiveChatSocket()
                            removeLiveChatTeamOptionObject(true)

                            liveChatDataContainer.feedbackRequest?.let { iLiveChatFeedbackRequest ->
                                _userFeedbackLiveData.postValue(iLiveChatFeedbackRequest)
                            }
                        }
                    }
            }

        }
    }

    private fun joinLiveChatIfConnected() {
        viewModelScope.launch(Dispatchers.IO) {
            if (liveChatStatus == ILiveChatDataContainer.ILiveChatStatus.CONNECTED) {
                liveChatData?.let {
                    liveChatRepository.sendJoinMeHereRequest(it)
                }
            }
        }
    }

    private fun processNewChatResponse(chatResponseString: String?) {
        viewModelScope.launch(Dispatchers.Default) {
            try {
                val chatResponse = convertToChatResponse(chatResponseString)
                chatResponseConverter.convertChatResponseToChatItems(
                    chatResponse = chatResponse,
                    chatResponseString = chatResponseString,
                    sessionId = sessionRepository.getSessionId(),
                    languageRepository.getCurrentLanguage() ?: LocaleUtils.getLanguageCode()
                )?.let { chatItems ->
                    if (chatResponseConverter.isTabularInfoItems(chatResponse)) {
                        addNewItemsToChat(chatItems)
                    } else {
                        chatItems.forEachIndexed { index, iChatItem ->
                            addNewItemToChat(iChatItem)
                            delay(AppConstants.MULTIPLE_MESSAGE_DELAY_PART_1)
                            speakMessage(iChatItem, index == 0)
                            delay(AppConstants.MULTIPLE_MESSAGE_DELAY_PART_2)
                        }
                    }
                }
                chatResponseConverter.convertChatResponseToJourney(chatResponse = chatResponse)
                    .forEach { journeyName ->
                        sendQuickJourney(
                            channelName = ApiConstants.CHANNEL_QUICK_JOURNEY,
                            message = journeyName,
                            fieldText = journeyName,
                            isFromProcessTree = true,
                            queryType = ApiConstants.CHAT_QUERY_TYPE_BUTTON
                        )
                        delay(AppConstants.MULTIPLE_MESSAGE_DELAY)
                    }


                chatResponseConverter.parseLiveChatDecisionResponseIfPresent(chatResponse = chatResponse)
                    ?.let { liveChatDecisionResponse ->
                        addLiveChatDecisionItems(
                            liveChatResponse = liveChatDecisionResponse,
                            queryType = chatResponse?.queryType
                        )
                    } ?: kotlin.run {
                    chatResponseConverter.parseDialogueLiveChatDecisionItem(chatResponse = chatResponse)
                        ?.let { dialogueLiveChatResponse ->
                            addLiveChatDecisionItems(
                                liveChatResponse = dialogueLiveChatResponse.livechatResponse,
                                queryType = chatResponse?.queryType,
                                userTeam = dialogueLiveChatResponse.userTeam
                            )
                        }
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
        }
    }

    private suspend fun addLiveChatDecisionItems(
        liveChatResponse: LiveChatResponse?,
        queryType: String?,
        userTeam: String? = null
    ) {
        setLoading(true)
        liveChatRepository.getLiveAgentConnectionDetails(
            iBotChatWindowConfig?.intId,
            sessionRepository.getSessionId()
        )?.let { liveChatForm ->
            liveChatForm.userTeam = userTeam
            chatResponseConverter.convertChatResponseToLiveChatItems(
                liveChatResponse = liveChatResponse,
                liveChatForm = liveChatForm,
                queryType = queryType,
                hideDecisionItems = userTeam?.isNotBlank() == true
            )?.let { chatItems ->
                chatItems.forEachIndexed { index, iChatItem ->
                    addNewItemToChat(iChatItem)
                    delay(AppConstants.MULTIPLE_MESSAGE_DELAY_PART_1)
                    speakMessage(iChatItem, index == 0)
                    delay(AppConstants.MULTIPLE_MESSAGE_DELAY_PART_2)
                }
            }

            if (userTeam?.isNotBlank() == true) {
                processLiveChatTrueClick(
                    liveChatForm = liveChatForm,
                    screenWidth = screenWidth ?: 0,
                    screenHeight = screenHeight ?: 0,
                    isLandscapeOrientation = isLandscapeOrientation == true,
                    location = null,
                )
            }
        }
        setLoading(false)
    }

    private suspend fun speakMessage(chatItem: IChatItem, flushExisting: Boolean) {
        if (iBotChatWindowConfig?.enableSpeech == true) {
            if (chatItem is IMessageItem && chatItem.isVoiceMessage) {
                startSpeech(chatItem, flushExisting)
            }
            if (chatItem is IMessagesItem && chatItem.isVoiceMessage) {
                startSpeech(chatItem, flushExisting)
            }
        }
    }

    private fun processNewTabularInfoResponse(chatResponseString: String?) {
        tabularInfoJobChannel.trySend(
            viewModelScope.launch(context = Dispatchers.Default, start = CoroutineStart.LAZY) {
                try {
                    val chatResponse = convertToChatResponse(chatResponseString)
                    (chatResponse?.responseId as? String)?.let { responseId ->
                        var tabularValueItem: ITabularValueItem? = null
                        var tabularInfoItem: ITabularInfoItem? = null

                        getChatItems().forEach { chatItem ->
                            if (chatItem is ITabularInfoItem && responseId == chatItem.responseId) {
                                val targetItem = chatItem.makeACopy() as ITabularInfoItem
                                targetItem.childItems?.forEach { childItem ->
                                    childItem.enHeading?.let { heading ->
                                        chatResponseConverter.convertToTabularItemChildDropDownItems(
                                            chatResponse,
                                            heading
                                        )?.let { newDropDownItems ->
                                            if (newDropDownItems.size > 1) {
                                                childItem.items = newDropDownItems
                                                childItem.selectedPosition = 0
                                                childItem.isLoading = false
                                                updateChatItem(targetItem, false)
                                            }
                                        }
                                    }
                                }
                                tabularInfoItem = targetItem
                            }

                            if (chatItem is ITabularValueItem && responseId == chatItem.responseId) {
                                tabularValueItem = chatItem
                            }
                        }

                        chatResponseConverter.convertAndGetTabularValueItem(
                            chatResponse.response,
                            Date(),
                            responseId
                        )?.let { newTabularValueItem ->
                            if (tabularInfoItem?.date != null) {
                                tabularInfoItem?.date = null
                                updateChatItem(tabularInfoItem, false)
                            }
                            tabularValueItem?.let { tvi ->
                                tvi.values = newTabularValueItem.values
                                updateChatItem(tvi, false)
                            } ?: kotlin.run {
                                tabularInfoItem?.let { tii ->
                                    addNewItemToChatAfter(newTabularValueItem, tii)
                                }
                            }
                        }
                    }
                } catch (e: Exception) {
                    LogUtils.logException(e)
                }
            }
        )
    }

    private fun convertToChatResponse(chatResponseString: String?): ChatResponse? {
        val chatResponse = Gson().fromJson(
            chatResponseString,
            ChatResponse::class.java
        )
        chatResponse.sessionObj?.let {
            viewModelScope.launch(Dispatchers.Default) {
                sessionRepository.updateSessionObj(it)
            }
        }
        return chatResponse
    }

    private suspend fun sendQuickJourney(
        channelName: String,
        message: String,
        fieldText: String,
        isFromProcessTree: Boolean,
        queryType: String
    ) {
        try {
            if (!isInternetAvailable()) {
                return
            }

            setLoading(true)
            val sent =
                sendMessage(channelName, message, fieldText, isFromProcessTree, queryType)
            if (!sent) {
                showToast(getApplication<Application>().getString(R.string.failed_to_send))
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
            showToast(getApplication<Application>().getString(R.string.failed_to_send))
        }
        setLoading(false)
    }

    private fun addMessageItemToChat(message: String?, isBotItem: Boolean): IMessageItem? {
        message?.obtainNonBlankOrNull()?.let {
            val iChatItem = IMessageItem(
                message = it,
                date = Date(),
                isLoading = false,
                isBotItem = isBotItem,
                isHtml = true
            )

            viewModelScope.launch(Dispatchers.Default) {
                addNewItemToChat(iChatItem)
            }
            return iChatItem
        }
        return null
    }

    override fun onNetworkStateChanged(networkState: NetworkState) {
        if (networkState == NetworkState.CONNECTED) {
            if (greetingJourneyFailed == true) {
                processGreetingJourney(iBotChatWindowConfig)
            }
            if (ErrorState.NETWORK_FAILURE == _historyErrorStateLiveData.value) {
                fetchHistory()
            }
            resetSocketConnection()
            resetLiveChatSocketIfConnected()
        }
    }

    private fun resetLiveChatSocketIfConnected() {
        viewModelScope.launch(Dispatchers.IO) {
            if (liveChatStatus == ILiveChatDataContainer.ILiveChatStatus.CONNECTED) {
                val connected = resetLiveChatSocketConnection()
                if (connected) {
                    joinLiveChatIfConnected()
                    refreshHistory()
                }
            }
        }
    }

    private suspend fun refreshHistory() {
        try {
            setHistoryLoading(true)
            val sessionId = sessionRepository.getSessionId()
            val userId = sessionRepository.getUserId()
            val history = historyRepository.getChatHistory(
                userId = userId,
                sessionId = sessionId,
                intId = iBotChatWindowConfig?.intId,
                extras = extras
            )
            history?.historyItems?.let {
                chatItemsFromUser.clear()
                chatItemsFromHistory.clear()
                chatItemsFromHistory.addAll(it)
                notifyChatItemsChange(scrollToEnd = true, avoidSmoothScroll = true)
            }
            setHistoryLoading(false)
        } catch (e: Exception) {
            setHistoryLoading(false)
            LogUtils.logException(e)
        }
    }

    fun sendJourneyMessage(
        message: String,
        fieldText: String? = null,
        isFromProcessTree: Boolean? = null
    ) {
        if (true == isFromProcessTree) {
            viewModelScope.launch(Dispatchers.IO) {
                sendQuickJourney(
                    channelName = ApiConstants.CHANNEL_QUICK_JOURNEY,
                    message = message,
                    fieldText = fieldText ?: message,
                    isFromProcessTree = true,
                    queryType = ApiConstants.CHAT_QUERY_TYPE_BUTTON
                )
            }
            return
        }

        addAndSendMessage(
            channelName = ApiConstants.CHANNEL_QUICK_JOURNEY,
            messageToSend = message,
            messageToDisplay = message,
            fieldText = message,
            isFromProcessTree = null,
            queryType = ApiConstants.CHAT_QUERY_TYPE_BUTTON
        )
    }

    fun sendTopJourneyMessage(topJr: ITopJr) {
        topJr.newTopic.obtainNonBlankOrNull()?.let { newTopic ->
            topJr.topic.obtainNonBlankOrNull()?.let { topic ->
                addAndSendMessage(
                    channelName = ApiConstants.CHANNEL_QUICK_JOURNEY,
                    messageToSend = topic,
                    messageToDisplay = newTopic,
                    fieldText = newTopic,
                    isFromProcessTree = null,
                    queryType = ApiConstants.CHAT_QUERY_TYPE_BUTTON
                )
            }
        }
    }

    fun sendQuickMenuMessage(quickMenuItem: IQuickMenuItem) {
        quickMenuItem.journey.obtainNonBlankOrNull()?.let { journey ->
            quickMenuItem.topic.obtainNonBlankOrNull()?.let { topic ->
                addAndSendMessage(
                    channelName = ApiConstants.CHANNEL_QUICK_JOURNEY,
                    messageToSend = journey,
                    messageToDisplay = topic,
                    fieldText = topic,
                    isFromProcessTree = null,
                    queryType = ApiConstants.CHAT_QUERY_TYPE_BUTTON
                )
            }
        }
    }

    fun sendButtonClick(iField: IField) {
        if (!isInternetAvailable()) {
            return
        }

        (iField.translatedText ?: iField.text)?.obtainNonBlankOrNull()?.let {
            viewModelScope.launch(Dispatchers.Default) {
                val addedMessage = addMessageItemToChat(it, false)
                setLoading(true)
                try {
                    val sent = sendFlow(
                        dynamicSelectedField = iField.text,
                        inputId = iField.inputId,
                        intId = iBotChatWindowConfig?.intId,
                        journeyId = iField.journeyId,
                        messageType = ServerConstants.CHAT_ITEM_TYPE_PROCESS_TREE,
                        msgId = iField.messageId,
                        nodeId = iField.nodeId,
                        queryType = ServerConstants.CHAT_ITEM_QUERY_TYPE_BUTTON,
                        selectedField = iField.text,
                        userInput = iField.journeyName,
                        languageSelection = iField.languageSelection,
                        languageCode = iField.languageCode
                    )
                    if (!sent) {
                        val pendingMessageItem = IPendingFlow(
                            dynamicSelectedField = iField.text,
                            inputId = iField.inputId,
                            intId = iBotChatWindowConfig?.intId,
                            journeyId = iField.journeyId,
                            messageType = ServerConstants.CHAT_ITEM_TYPE_PROCESS_TREE,
                            msgId = iField.messageId,
                            nodeId = iField.nodeId,
                            queryType = ServerConstants.CHAT_ITEM_QUERY_TYPE_BUTTON,
                            selectedField = iField.text,
                            userInput = iField.journeyName,
                            languageSelection = iField.languageSelection,
                            languageCode = iField.languageCode
                        )
                        setPendingData(addedMessage?.uuid, pendingMessageItem)
                    }

                } catch (e: Exception) {
                    LogUtils.logException(e)
                }
                setLoading(false)
            }
        }
    }

    fun processPendingMessage(pendingItem: Any) {
        if (!isInternetAvailable()) {
            return
        }

        viewModelScope.launch(Dispatchers.Default) {
            try {
                setLoading(true)
                if (pendingItem is IMessageItem) {
                    var sent = false
                    if (pendingItem.pendingData is IPendingMessageItem) {
                        val pendingMessageItem = pendingItem.pendingData as IPendingMessageItem
                        sent = sendMessage(
                            channelName = pendingMessageItem.channelName,
                            message = pendingMessageItem.message,
                            fieldText = pendingMessageItem.fieldText,
                            isFromProcessTree = pendingMessageItem.isFromProcessTree,
                            queryType = pendingMessageItem.queryType
                        )
                    } else if (pendingItem.pendingData is IPendingFlow) {
                        val pendingFlowItem = pendingItem.pendingData as IPendingFlow
                        sent = sendFlow(
                            dynamicSelectedField = pendingFlowItem.dynamicSelectedField,
                            inputId = pendingFlowItem.inputId,
                            intId = pendingFlowItem.intId,
                            journeyId = pendingFlowItem.journeyId,
                            messageType = pendingFlowItem.messageType,
                            msgId = pendingFlowItem.msgId,
                            nodeId = pendingFlowItem.nodeId,
                            queryType = pendingFlowItem.queryType,
                            selectedField = pendingFlowItem.selectedField,
                            userInput = pendingFlowItem.userInput,
                            languageSelection = pendingFlowItem.languageSelection,
                            languageCode = pendingFlowItem.languageCode
                        )
                    } else if (pendingItem.pendingData is IChatConflictChildItem) {
                        val iChatConflictChildItem =
                            pendingItem.pendingData as IChatConflictChildItem
                        sent = sendFaq(
                            faqId = iChatConflictChildItem.faqId,
                            journeyId = iChatConflictChildItem.journeyId,
                            question = iChatConflictChildItem.question
                        )
                    } else if (pendingItem.pendingData is IFaqAutoCompleteItem) {
                        val iFaqAutoCompleteItem = pendingItem.pendingData as IFaqAutoCompleteItem
                        sent = sendFaq(
                            faqId = iFaqAutoCompleteItem.faqId,
                            journeyId = iFaqAutoCompleteItem.journeyId,
                            question = iFaqAutoCompleteItem.question
                        )
                    }
                    if (sent) {
                        setPendingData(pendingItem.uuid, null)
                    }
                }
                setLoading(false)
            } catch (e: Exception) {
                LogUtils.logException(e)
                setLoading(false)
            }
        }
    }

    private fun setPendingData(uuid: String?, pendingData: Any?) {
        if (uuid == null) return
        applyPendingData(uuid, pendingData, chatItemsFromUser)
        notifyChatItemsChange(false)
    }

    private fun applyPendingData(
        uuid: String,
        newPendingData: Any?,
        chatItems: ArrayList<IChatItem>
    ) {
        chatItems.forEachIndexed { index, iChatItem ->
            if (iChatItem.uuid == uuid) {
                chatItems[index] = iChatItem.makeACopy().apply {
                    pendingData = newPendingData
                }
                return
            }
        }
    }

    fun processCarousalButtonClick(buttonData: ICarousalButtonData) {
        if (ServerConstants.DIALOGUE_BUTTON_TYPE_JOURNEY == buttonData.buttonType) {
            buttonData.data?.let {
                if (!isInternetAvailable()) {
                    return
                }

                viewModelScope.launch(Dispatchers.Default) {
                    try {
                        setLoading(true)
                        val sent = sendMessage(
                            channelName = ApiConstants.CHANNEL_QUICK_JOURNEY,
                            message = it,
                            fieldText = it,
                            isFromProcessTree = true,
                            queryType = ApiConstants.CHAT_QUERY_TYPE_BUTTON
                        )
                        if (!sent) {
                            showToast(getApplication<Application>().resources.getString(R.string.failed_to_send))
                        }
                        setLoading(false)
                    } catch (e: Exception) {
                        LogUtils.logException(e)
                        setLoading(false)
                    }
                }
            }
        }

    }

    fun setSelectedFile(
        fileUri: Uri,
        contentResolver: ContentResolver,
        formFileItem: IFormFileItem?,
        formFileUploadView: FormFileUploadView?
    ) {
        viewModelScope.launch(Dispatchers.Main) {
            FileUtils.fetchFileMetadataFromUri(fileUri, contentResolver)?.let { fileMetadata ->
                if (fileMetadata.fileSize > AppConstants.MAX_FILE_SIZE_FOR_UPLOAD) {
                    val errorMessage =
                        getApplication<Application>().getString(R.string.file_size_2mb_exceeded)
                    showToast(errorMessage)
                    applySelectedFileData(
                        null,
                        null,
                        null,
                        null,
                        errorMessage,
                        formFileItem,
                        formFileUploadView
                    )
                } else {
                    try {
                        FileUtils.createTempFileForUri(
                            getApplication(),
                            contentResolver,
                            fileUri,
                            fileMetadata
                        )?.let { file ->
                            applySelectedFileData(
                                file,
                                fileMetadata.mimeType,
                                file.name,
                                fileUri,
                                null,
                                formFileItem,
                                formFileUploadView
                            )
                        } ?: run {
                            val errorMessage =
                                getApplication<Application>().getString(R.string.failed_to_select_file)
                            applySelectedFileData(
                                null,
                                null,
                                null,
                                null,
                                errorMessage,
                                formFileItem,
                                formFileUploadView
                            )
                        }
                    } catch (e: Exception) {
                        LogUtils.logException(e)
                        val errorMessage =
                            getApplication<Application>().getString(R.string.failed_to_select_file)
                        showToast(errorMessage)
                        applySelectedFileData(
                            null,
                            null,
                            null,
                            null,
                            errorMessage,
                            formFileItem,
                            formFileUploadView
                        )
                    }
                }
            }
        }
    }

    private fun applySelectedFileData(
        tempFile: File?,
        mimeType: String?,
        fileOriginalName: String?,
        fileUri: Uri?,
        errorMessage: String?,
        formFileItem: IFormFileItem?,
        formFileUploadView: FormFileUploadView?
    ) {
        formFileItem?.apply {
            FileUtils.deleteParentAndSelfIfExists(this.file)
            this.file = tempFile
            this.mimeType = mimeType
            this.fileOriginalName = fileOriginalName
            this.fileUri = fileUri
            this.errorMessage = errorMessage
        }?.also {
            formFileUploadView?.updateFileData(it.fileOriginalName)
            formFileUploadView?.setError(it)
        }
    }

    fun submitForm(
        chatFormItem: IChatFormItem,
        buttonItem: IFormButtonItem,
        adapter: ChatItemsRecyclerViewAdapter,
    ) {
        if (!isInternetAvailable()) {
            buttonItem.isLoading = false
            adapter.notifyDataSetChanged()
            return
        }
        viewModelScope.launch(Dispatchers.Default) {
            if (chatFormItem.submitType == IChatFormItem.SubmitType.API) {
                val sessionObj = sessionRepository.getSessionObj()
                val socketId = sessionRepository.getSocketId()
                val result = formRepository.submitForm(
                    iChatFormItem = chatFormItem,
                    botId = iBotChatWindowConfig?.botId,
                    inputId = chatFormItem.inputId,
                    journeyId = chatFormItem.journeyId,
                    languageCode = chatFormItem.languageCode,
                    messageType = chatFormItem.messageType,
                    msgId = chatFormItem.messageId,
                    nodeId = chatFormItem.nodeId,
                    onSubmit = chatFormItem.onSubmit,
                    pageUrl = "",
                    respondName = chatFormItem.respondName,
                    respondType = chatFormItem.respondType,
                    sessionId = chatFormItem.sessionId,
                    sessionObj = sessionObj,
                    socketId = socketId,
                    userInput = chatFormItem.userInput,
                    webSocketId = socketId,
                )

                if (result == FormRepository.Failure.SERVER_ERROR) {
                    showToast(getApplication<Application>().getString(R.string.failed_to_submit_form))
                }

                buttonItem.isSubmitted = result == null
            }
            withContext(Dispatchers.Main) {
                buttonItem.isLoading = false
                adapter.notifyDataSetChanged()
            }
        }
    }

    fun submitLiveChatForm(
        chatFormItem: IChatFormItem,
        buttonItem: IFormButtonItem,
        adapter: ChatItemsRecyclerViewAdapter,
        screenWidth: Int,
        screenHeight: Int,
        isLandscapeOrientation: Boolean,
        location: ILocation?
    ) {
        if (!isInternetAvailable()) {
            buttonItem.isLoading = false
            adapter.notifyDataSetChanged()
            return
        }
        viewModelScope.launch(Dispatchers.Default) {
            if (chatFormItem.submitType == IChatFormItem.SubmitType.SOCKET && buttonItem.extras is ILiveChatForm) {
                val connected = ensureLiveChatConnection()
                if (connected) {
                    val sent = liveChatRepository.sendAgentConnectRequest(
                        botId = iBotChatWindowConfig?.botId,
                        channel = "bot",
                        displayOutput = buttonItem.extras.displayOutput,
                        fetchApi = false,
                        formData = chatFormItem.toFormData(),
                        prevLangCode = languageRepository.getPreviousLanguage(),
                        sessionId = sessionRepository.getSessionId(),
                        settingId = buttonItem.extras.settingId,
                        userId = sessionRepository.getUserId(),
                        userIntId = iBotChatWindowConfig?.intId,
                        userIntName = iBotChatWindowConfig?.intName,
                        osName = AppConstants.OS_NAME,
                        osVersion = Build.VERSION.SDK_INT.toString(),
                        screenWidth = screenWidth,
                        screenHeight = screenHeight,
                        screenOrientation = if (isLandscapeOrientation) AppConstants.LANDSCAPE_ORIENTATION
                        else AppConstants.PORTRAIT_ORIENTATION,
                        userName = null,
                        currentSessionId = null,
                        socketId = null,
                        userTeam = buttonItem.extras.userTeam,
                        teamList = null,
                        location = location,
                        userProperty = null,
                    )
                    if (sent) {
                        languageRepository.updatePreviousWithCurrent()
                        removeChatItemsToBeRemoved(buttonItem.extras)
                        notifyChatItemsChange(false)
                    }
                }
            }

            withContext(Dispatchers.Main) {
                buttonItem.isLoading = false
                adapter.notifyDataSetChanged()
            }
        }
    }

    private suspend fun ensureLiveChatConnection(): Boolean {
        val connected: Boolean = if (!isLiveChatSocketConnected()) {
            connectToLiveChatSocket()
        } else {
            true
        }
        return connected
    }

    fun sendFaqConflictChildClick(iChatConflictChildItem: IChatConflictChildItem) {
        if (!isInternetAvailable()) {
            return
        }

        viewModelScope.launch(Dispatchers.IO) {
            val addedMessage = addMessageItemToChat(iChatConflictChildItem.question, false)
            try {
                setLoading(true)
                val sent = sendFaq(
                    iChatConflictChildItem.faqId,
                    iChatConflictChildItem.journeyId,
                    iChatConflictChildItem.question
                )
                if (!sent) {
                    setPendingData(addedMessage?.uuid, iChatConflictChildItem)
                }
                setLoading(false)
            } catch (e: Exception) {
                LogUtils.logException(e)
                setLoading(false)
            }
        }
    }

    private suspend fun sendFaq(faqId: String?, journeyId: String?, question: String?): Boolean {
        val sessionId = sessionRepository.getSessionId()
        val userId = sessionRepository.getUserId()
        val sessionObj = sessionRepository.getSessionObj()
        var sent = false
        if (sessionObj != null) {
            sent = chatRepository.sendFaq(
                faqId = faqId,
                journeyID = journeyId,
                botId = iBotChatWindowConfig?.botId,
                channel = ApiConstants.REQUEST_CHANNEL,
                inputDateNTime = DateUtils.getCurrentTimeInISO8601(),
                intId = iBotChatWindowConfig?.intId,
                intName = iBotChatWindowConfig?.intName,
                pageUrl = "",
                question = question,
                resBy = "user",
                resTime = DateUtils.getCurrentTime(),
                sessionId = sessionId,
                sessionObj = sessionObj,
                userId = userId,
                prevLangCode = languageRepository.getPreviousLanguage(),
                userLang = languageRepository.getCurrentLanguage()
            )
            if (sent) {
                languageRepository.updatePreviousWithCurrent()
            }
        }
        return sent
    }

    fun sendFaqAutoCompleteItem(iFaqAutoCompleteItem: IFaqAutoCompleteItem): Boolean {
        if (!isInternetAvailable()) return false

        viewModelScope.launch(Dispatchers.IO) {
            val addedMessage = addMessageItemToChat(iFaqAutoCompleteItem.question, false)
            try {
                setLoading(true)
                val sent = sendFaq(
                    iFaqAutoCompleteItem.faqId,
                    iFaqAutoCompleteItem.journeyId,
                    iFaqAutoCompleteItem.question
                )
                if (!sent) {
                    setPendingData(addedMessage?.uuid, iFaqAutoCompleteItem)
                }
                setLoading(false)
            } catch (e: Exception) {
                LogUtils.logException(e)
                setLoading(false)
            }
        }
        return true
    }

    private fun isInternetAvailable(): Boolean {
        if (NetworkUtils.isInternetAvailable(getApplication())) {
            return true
        }
        showToast(getApplication<Application>().resources.getString(R.string.no_internet))
        return false
    }

    fun processTabularItemChildDropDownSelection(
        tabularInfoItem: ITabularInfoItem?,
        tabularInfoItemChildItem: ITabularInfoChildItem?
    ) {
        viewModelScope.launch(Dispatchers.Default) {

            if (tabularInfoItem == null) return@launch

            try {
                var childItemToRefresh: ITabularInfoChildItem? = null
                val childItemsIterator = tabularInfoItem.childItems?.iterator()
                while (childItemsIterator?.hasNext() == true) {
                    if (childItemsIterator.next() == tabularInfoItemChildItem && childItemsIterator.hasNext()) {
                        childItemToRefresh = childItemsIterator.next()
                        break
                    }
                }
                if (childItemToRefresh != null) {
                    clearSelections(childItemToRefresh)
                    while (childItemsIterator?.hasNext() == true) {
                        clearSelections(childItemsIterator.next())
                    }

                    if (tabularInfoItemChildItem?.selectedPosition == 0) {
                        updateChatItem(tabularInfoItem, false)
                        return@launch
                    }

                    childItemToRefresh.isLoading = true
                    updateChatItem(tabularInfoItem, false)

                    sendTabularInfo(
                        tabularInfoItem = tabularInfoItem,
                        nextDimension = childItemToRefresh.enHeading,
                        uniqueId = null
                    )

                    childItemToRefresh.isLoading = false
                    updateChatItem(tabularInfoItem, false)
                } else {
                    updateChatItem(tabularInfoItem, false)
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
        }
    }

    private suspend fun sendTabularInfo(
        tabularInfoItem: ITabularInfoItem,
        nextDimension: String?,
        uniqueId: String?
    ) {
        val selectedEnVal = HashMap<String, String>()
        val selectedTransVal = HashMap<String, String>()
        tabularInfoItem.childItems?.forEach { tabularInfoChildItem ->
            if (tabularInfoChildItem.selectedPosition > 0) {
                tabularInfoChildItem.enHeading?.let { heading ->
                    tabularInfoChildItem.items?.get(tabularInfoChildItem.selectedPosition)?.enVal?.let { enVal ->
                        selectedEnVal.put(heading, enVal)
                    }
                }
                tabularInfoChildItem.heading?.let { heading ->
                    tabularInfoChildItem.items?.get(tabularInfoChildItem.selectedPosition)?.transVal?.let { transVal ->
                        selectedTransVal.put(heading, transVal)
                    }
                }
            }
        }

        chatRepository.sendTabularInfo(
            headings = tabularInfoItem.childItems?.map { tabularInfoChildItem ->
                tabularInfoChildItem.enHeading ?: ""
            },
            inputId = tabularInfoItem.inputId,
            intId = iBotChatWindowConfig?.intId,
            isMapEnabled = tabularInfoItem.isMapEnabled,
            journeyId = tabularInfoItem.journeyId,
            languageCode = languageRepository.getCurrentLanguage(),
            messageType = tabularInfoItem.messageType,
            msgId = tabularInfoItem.msgId,
            nextDimension = nextDimension,
            responseId = tabularInfoItem.responseId,
            responseParams = tabularInfoItem.responseParams,
            selectedEnVal = selectedEnVal,
            selectedTransVal = selectedTransVal,
            sessionId = sessionRepository.getSessionId(),
            sessionObj = sessionRepository.getSessionObj(),
            userInput = tabularInfoItem.journeyName,
            uniqueId = uniqueId
        )
    }

    private fun updateChatItem(
        updatedItem: IChatItem?,
        scrollToEnd: Boolean
    ) {
        if (updatedItem == null) return

        val itemCopy = updatedItem.makeACopy()
        updateChatItem(itemCopy, chatItemsFromHistory)
        updateChatItem(itemCopy, chatItemsFromUser)
        notifyChatItemsChange(scrollToEnd)
    }

    private fun updateChatItem(
        updatedItem: IChatItem,
        chatItems: ArrayList<IChatItem>
    ) {
        chatItems.forEachIndexed { index, iChatItem ->
            if (iChatItem.uuid == updatedItem.uuid) {
                chatItems[index] = updatedItem
                return
            }
        }
    }

    private fun clearSelections(childItemToRefresh: ITabularInfoChildItem) {
        childItemToRefresh.items = ArrayList<ITabularDropDownItem>().apply {
            add(ITabularDropDownItem(null, null))
        }
        childItemToRefresh.selectedPosition = 0
    }

    fun setSpeechText(recognizedText: String?) {
        if (recognizedText?.isNotBlank() == true) {
            addAndSendMessage(
                channelName = ApiConstants.CHANNEL_CHAT,
                messageToSend = recognizedText,
                messageToDisplay = recognizedText,
                queryType = ApiConstants.CHAT_QUERY_TYPE_VOICE
            )
        }
    }

    private fun addHistoryItemsToChat(historyContainer: IHistoryContainer?) {
        historyContainer?.historyItems?.let {
            chatItemsFromHistory.clear()
            chatItemsFromHistory.addAll(it)
            notifyChatItemsChange(true)
        }

        resumeLiveChatIfSaved(historyContainer?.liveChatItem)

        historyContainer?.feedbackRequest?.let { iLiveChatFeedbackRequest ->
            _userFeedbackLiveData.postValue(iLiveChatFeedbackRequest)
        }
    }

    private fun syncBotFeedbackShownStatus(isBotFeedbackShown: Boolean?) {
        if (isBotFeedbackShown == true) {
            iBotChatWindowConfig?.iBotFeedbackRequest?.showAfterSeconds = 0
            iBotChatWindowConfig?.iBotFeedbackRequest?.showAfterMessages = 0
        }
    }

    private fun showBotFeedbackBasedOnMessagesCount() {
        var showAfterMessages =
            iBotChatWindowConfig?.iBotFeedbackRequest?.showAfterMessages ?: 0
        if (--showAfterMessages > 0) {
            iBotChatWindowConfig?.iBotFeedbackRequest?.showAfterMessages = showAfterMessages
        } else if (showAfterMessages == 0) {
            iBotChatWindowConfig?.iBotFeedbackRequest?.showAfterMessages = showAfterMessages
            _enableBotFeedbackLiveData.postValue(iBotChatWindowConfig?.iBotFeedbackRequest)
        }
    }

    private fun showBotFeedbackBasedOnSeconds(skipIfNotAlreadyRun: Boolean) {
        if (skipIfNotAlreadyRun && showFeedbackAfterSecondsJob == null) {
            return
        }
        try {
            showFeedbackAfterSecondsJob?.cancel()
        } catch (ignore: Exception) {
        }
        var showBotFeedbackAfterSeconds =
            iBotChatWindowConfig?.iBotFeedbackRequest?.showAfterSeconds ?: 0
        if (showBotFeedbackAfterSeconds > 0) {
            showFeedbackAfterSecondsJob = viewModelScope.launch(Dispatchers.Default) {
                while (showBotFeedbackAfterSeconds-- > 0) {
                    delay(1000)
                }
                iBotChatWindowConfig?.iBotFeedbackRequest?.showAfterSeconds = 0
                _enableBotFeedbackLiveData.postValue(iBotChatWindowConfig?.iBotFeedbackRequest)
            }
        }
    }

    private fun addNewItemToChat(chatItem: IChatItem) {
        chatItemsFromUser.add(chatItem)
        notifyChatItemsChange(true)
    }

    private fun addNewItemsToChat(chatItems: List<IChatItem>) {
        chatItemsFromUser.addAll(chatItems)
        notifyChatItemsChange(true)
    }

    private fun addNewItemToChatAfter(newItem: IChatItem, referenceItem: IChatItem) {
        addNewItemToChatAfter(newItem, referenceItem, chatItemsFromHistory)
        addNewItemToChatAfter(newItem, referenceItem, chatItemsFromUser)
        notifyChatItemsChange(false)
    }

    private fun addNewItemToChatAfter(
        newItem: IChatItem,
        referenceItem: IChatItem,
        chatItems: ArrayList<IChatItem>
    ) {
        findNextIndexOfItem(referenceItem, chatItems)?.let { foundIndex ->
            if (foundIndex >= chatItems.size) {
                chatItems.add(newItem)
            } else {
                chatItems.add(foundIndex, newItem)
            }
        }
    }

    private fun findNextIndexOfItem(referenceItem: IChatItem, chatItems: List<IChatItem>): Int? {
        chatItems.forEachIndexed { index, iChatItem ->
            if (iChatItem.uuid == referenceItem.uuid) {
                return index + 1
            }
        }
        return null
    }

    private fun notifyChatItemsChange(
        scrollToEnd: Boolean,
        avoidSmoothScroll: Boolean = false,
        callback: (() -> Unit)? = null
    ) {
        synchronized(this) {
            val newItems = ArrayList<IChatItem>()
            newItems.addAll(chatItemsFromGreeting)
            newItems.addAll(chatItemsFromConfig)
            newItems.addAll(chatItemsFromHistory)
            newItems.addAll(chatItemsFromUser)
            if (isLoadingMore) {
                newItems.add(IBotLoadingItem())
            }
            _chatItemsLiveData.postValue(
                IChatItemsContainer(
                    newItems,
                    scrollToEnd,
                    avoidSmoothScroll
                )
            )
            callback?.invoke()
            showBotFeedbackBasedOnSeconds(skipIfNotAlreadyRun = true)
        }
    }

    private fun getChatItems(): List<IChatItem> {
        return chatItemsFromGreeting + chatItemsFromConfig + chatItemsFromHistory + chatItemsFromUser
    }

    enum class ErrorState { NETWORK_FAILURE, SERVER_FAILURE }

    override fun onSpeechCompleted(utteranceId: String?) {
        setPlaying(utteranceId, false)
    }

    override fun onSpeechStarted(utteranceId: String?) {
        setPlaying(utteranceId, true)
    }

    private fun setPlaying(utteranceId: String?, isPlaying: Boolean) {
        viewModelScope.launch(Dispatchers.Default) {
            setPlaying(utteranceId, isPlaying, chatItemsFromConfig)
            setPlaying(utteranceId, isPlaying, chatItemsFromHistory)
            setPlaying(utteranceId, isPlaying, chatItemsFromUser)
            notifyChatItemsChange(false)
        }
    }

    private fun setPlaying(
        utteranceId: String?, isPlaying: Boolean, targetItems: ArrayList<IChatItem>
    ) {
        targetItems.forEachIndexed { index, iChatItem ->
            if (iChatItem.uuid == utteranceId) {
                if (iChatItem is IMessageItem) {
                    targetItems[index] = iChatItem.makeACopy().apply {
                        (this as IMessageItem).isVoicePlaying = isPlaying
                    }
                } else if (iChatItem is IMessagesItem) {
                    targetItems[index] = iChatItem.makeACopy().apply {
                        (this as IMessagesItem).isVoicePlaying = isPlaying
                    }
                }
            }
        }
    }

    private suspend fun setAllAsNotPlaying(callback: (() -> Unit)? = null) {
        withContext(Dispatchers.Default) {
            val anyValueUpdated = setAllNotPlaying(chatItemsFromConfig)
                    || setAllNotPlaying(chatItemsFromHistory)
                    || setAllNotPlaying(chatItemsFromUser)
            if (anyValueUpdated) {
                notifyChatItemsChange(scrollToEnd = false, callback = callback)
            } else {
                callback?.invoke()
            }
        }
    }

    private fun setAllNotPlaying(targetItems: ArrayList<IChatItem>): Boolean {
        var anyValueUpdated = false
        targetItems.forEachIndexed { index, iChatItem ->
            if (iChatItem is IMessageItem && iChatItem.isVoicePlaying) {
                targetItems[index] = iChatItem.makeACopy().apply {
                    (this as IMessageItem).isVoicePlaying = false
                }
                anyValueUpdated = true
            } else if (iChatItem is IMessagesItem && iChatItem.isVoicePlaying) {
                targetItems[index] = iChatItem.makeACopy().apply {
                    (this as IMessagesItem).isVoicePlaying = false
                }
                anyValueUpdated = true
            }
        }
        return anyValueUpdated
    }

    fun requestSpeechStart(iChatItem: IChatItem, flushExisting: Boolean) {
        viewModelScope.launch(Dispatchers.Default) {
            startSpeech(iChatItem, flushExisting)
        }
    }

    private suspend fun startSpeech(iChatItem: IChatItem, flushExisting: Boolean) {
        val stopPlayingOnly = isInVoicePlayingState(iChatItem)
        if (flushExisting) {
            stopExistingSpeech()
        }
        if (stopPlayingOnly) {
            return
        }
        if (iChatItem is IMessageItem) {
            startSpeech(
                iChatItem.message,
                iChatItem.isHtml,
                flushExisting,
                iChatItem.uuid
            )
        }
        if (iChatItem is IMessagesItem) {
            iChatItem.messages.fold("", operation = { initialVal, currentVal ->
                "$initialVal, $currentVal"
            }).let { message ->
                startSpeech(
                    message,
                    iChatItem.isHtml,
                    flushExisting,
                    iChatItem.uuid
                )
            }
        }
    }

    private fun isInVoicePlayingState(iChatItem: IChatItem): Boolean {
        if (iChatItem is IMessageItem && iChatItem.isVoicePlaying) {
            return true
        } else if (iChatItem is IMessagesItem && iChatItem.isVoicePlaying) {
            return true
        }
        return false
    }

    fun requestSpeechStop(callback: (() -> Unit)? = null) {
        viewModelScope.launch {
            stopSpeech()
            setAllAsNotPlaying(callback)
        }
    }

    private suspend fun stopExistingSpeech() {
        stopSpeech()
        setAllAsNotPlaying()
    }

    private suspend fun stopSpeech() {
        withContext(Dispatchers.Main) {
            textToSpeechHelper.stopSpeaking()
        }
    }

    private fun startSpeech(
        message: String,
        isHtml: Boolean,
        flushExisting: Boolean,
        utteranceId: String
    ) {
        textToSpeechHelper.startSpeech(message, isHtml, flushExisting, utteranceId)
    }

    fun getUpdatedBotConfigData(): IBotChatWindowConfig? {
        if (!::updatedBotConfigData.isInitialized) return null

        return updatedBotConfigData
    }

    fun processLiveChatFalseClick(uuid: String) {
        removeItemByUuid(uuid, chatItemsFromUser)
        notifyChatItemsChange(true)
    }

    private fun removeItemByUuid(uuid: String?, chatItems: ArrayList<IChatItem>) {
        if (uuid == null) return

        val chatItemsIterator = chatItems.iterator()
        while (chatItemsIterator.hasNext()) {
            if (chatItemsIterator.next().uuid == uuid) {
                chatItemsIterator.remove()
                break
            }
        }
    }

    fun processLiveChatTrueClick(
        liveChatForm: ILiveChatForm,
        screenWidth: Int,
        screenHeight: Int,
        isLandscapeOrientation: Boolean,
        location: ILocation?
    ) {
        viewModelScope.launch(Dispatchers.Default) {
            try {
                val liveChatItems = ArrayList<IChatItem>()
                removeChatItemsToBeRemoved(liveChatForm)

                when (liveChatForm.inputType) {
                    ServerConstants.LIVE_CHAT_INPUT_TYPE_NO_FORM -> {
                        setLoading(true)
                        val connected = ensureLiveChatConnection()
                        if (connected) {
                            liveChatRepository.sendAgentConnectRequest(
                                botId = iBotChatWindowConfig?.botId,
                                channel = "bot",
                                displayOutput = liveChatForm.displayOutput,
                                fetchApi = false,
                                formData = liveChatForm.value,
                                prevLangCode = languageRepository.getPreviousLanguage(),
                                sessionId = sessionRepository.getSessionId(),
                                settingId = liveChatForm.settingId,
                                userId = sessionRepository.getUserId(),
                                userIntId = iBotChatWindowConfig?.intId,
                                userIntName = iBotChatWindowConfig?.intName,
                                osName = AppConstants.OS_NAME,
                                osVersion = Build.VERSION.SDK_INT.toString(),
                                screenWidth = screenWidth,
                                screenHeight = screenHeight,
                                screenOrientation = if (isLandscapeOrientation) AppConstants.LANDSCAPE_ORIENTATION
                                else AppConstants.PORTRAIT_ORIENTATION,
                                userName = null,
                                currentSessionId = null,
                                socketId = null,
                                userTeam = liveChatForm.userTeam,
                                teamList = null,
                                location = location,
                                userProperty = null
                            )
                            languageRepository.updatePreviousWithCurrent()
                        }
                        setLoading(false)
                    }
                    ServerConstants.LIVE_CHAT_INPUT_TYPE_ID,
                    ServerConstants.LIVE_CHAT_INPUT_TYPE_PHONE_NUMBER,
                    ServerConstants.LIVE_CHAT_INPUT_TYPE_EMAIL_ID,
                    ServerConstants.LIVE_CHAT_INPUT_TYPE_DEFAULT_FORM -> {
                        chatResponseConverter.convertFieldMessageToChatItem(liveChatForm)?.let {
                            liveChatItems.add(it)
                        }
                        chatResponseConverter.convertCustomerInfoInputToChatItem(liveChatForm)
                            ?.let {
                                liveChatItems.add(it)
                            }
                    }
                    ServerConstants.LIVE_CHAT_INPUT_TYPE_FORM -> {
                        chatResponseConverter.convertFieldMessageToChatItem(liveChatForm)?.let {
                            liveChatItems.add(it)
                        }
                        chatResponseConverter.convertLiveChatFormToChatItems(liveChatForm)?.let {
                            liveChatItems.addAll(it)
                        }
                    }
                }
                addNewItemsToChat(liveChatItems)
            } catch (e: Exception) {
                LogUtils.logException(e)
                setLoading(false)
            }
        }
    }

    private fun removeChatItemsToBeRemoved(liveChatForm: ILiveChatForm) {
        liveChatForm.chatItemIdsToRemove?.forEach { uuid ->
            removeItemByUuid(uuid, chatItemsFromUser)
        }
        liveChatForm.clearChatItemsToRemove()
    }

    fun processMessagesActionClick(
        iMessagesItem: IMessagesItem,
        screenWidth: Int,
        screenHeight: Int,
        isLandscapeOrientation: Boolean,
        location: ILocation?
    ) {
        if (isLoadingMore) return

        viewModelScope.launch(Dispatchers.Default) {
            try {
                if (IMessagesItem.IMessagesStyle.FORM == iMessagesItem.messagesStyle) {
                    (iMessagesItem.extras as? ILiveChatForm)?.let { iLiveChatForm ->
                        if (iLiveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_ID
                            || iLiveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_PHONE_NUMBER
                            || iLiveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_EMAIL_ID
                            || iLiveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_DEFAULT_FORM
                        ) {
                            if (iMessagesItem.messages.isNotEmpty()) {
                                iMessagesItem.messages.forEach { item ->
                                    (item as? IFormTextItem)?.value.let { value ->
                                        if (value.isNullOrBlank()) {
                                            showToast(getApplication<Application>().getString(R.string.value_required))
                                            return@launch
                                        }
                                        if (iLiveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_EMAIL_ID && !value.isValidEmail()) {
                                            return@launch
                                        }
                                    }
                                }

                                if (!isInternetAvailable()) return@launch

                                setLoading(true)
                                val connected = ensureLiveChatConnection()
                                if (connected) {
                                    liveChatRepository.sendAgentConnectRequest(
                                        botId = iBotChatWindowConfig?.botId,
                                        channel = "bot",
                                        displayOutput = iLiveChatForm.displayOutput,
                                        fetchApi = false,
                                        formData = HashMap<String?, String?>()
                                            .apply {
                                                if (iLiveChatForm.inputType == ServerConstants.LIVE_CHAT_INPUT_TYPE_DEFAULT_FORM) {
                                                    iMessagesItem.messages.forEach { item ->
                                                        (item as? IFormTextItem)?.let { iFormTextItem ->
                                                            put(
                                                                iFormTextItem.labelName,
                                                                iFormTextItem.value
                                                            )
                                                        }
                                                    }
                                                } else {
                                                    put(
                                                        "user_name",
                                                        (iMessagesItem.messages.first() as? IFormTextItem)?.value
                                                    )
                                                }
                                            },
                                        prevLangCode = languageRepository.getPreviousLanguage(),
                                        sessionId = sessionRepository.getSessionId(),
                                        settingId = iLiveChatForm.settingId,
                                        userId = sessionRepository.getUserId(),
                                        userIntId = iBotChatWindowConfig?.intId,
                                        userIntName = iBotChatWindowConfig?.intName,
                                        osName = AppConstants.OS_NAME,
                                        osVersion = Build.VERSION.SDK_INT.toString(),
                                        screenWidth = screenWidth,
                                        screenHeight = screenHeight,
                                        screenOrientation = if (isLandscapeOrientation) AppConstants.LANDSCAPE_ORIENTATION
                                        else AppConstants.PORTRAIT_ORIENTATION,
                                        userName = (iMessagesItem.messages.first() as? IFormTextItem)?.value,
                                        currentSessionId = null,
                                        socketId = null,
                                        userTeam = iLiveChatForm.userTeam,
                                        teamList = null,
                                        location = location,
                                        userProperty = null
                                    )
                                    languageRepository.updatePreviousWithCurrent()
                                    removeChatItemsToBeRemoved(iLiveChatForm)
                                }
                                setLoading(false)
                            }
                        }
                    }
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
                setLoading(false)
            }
        }
    }

    fun processLiveChatTeamClick(
        iLiveChatTeam: ILiveChatTeam,
        screenWidth: Int,
        screenHeight: Int,
        landscapeOrientation: Boolean
    ) {
        if (!isInternetAvailable()) return

        viewModelScope.launch(Dispatchers.Default) {
            setLoading(true)
            val connected = ensureLiveChatConnection()
            if (connected) {
                liveChatRepository.sendAgentConnectRequest(
                    botId = iBotChatWindowConfig?.botId,
                    channel = "bot",
                    displayOutput = iLiveChatTeam.session?.displayOutput,
                    fetchApi = false,
                    formData = iLiveChatTeam.session?.formData,
                    prevLangCode = languageRepository.getPreviousLanguage(),
                    sessionId = sessionRepository.getSessionId(),
                    settingId = iLiveChatTeam.session?.settingId,
                    userId = sessionRepository.getUserId(),
                    userIntId = iBotChatWindowConfig?.intId,
                    userIntName = iBotChatWindowConfig?.intName,
                    osName = AppConstants.OS_NAME,
                    osVersion = Build.VERSION.SDK_INT.toString(),
                    screenWidth = screenWidth,
                    screenHeight = screenHeight,
                    screenOrientation = if (landscapeOrientation) AppConstants.LANDSCAPE_ORIENTATION
                    else AppConstants.PORTRAIT_ORIENTATION,
                    userName = iLiveChatTeam.session?.userName,
                    currentSessionId = iLiveChatTeam.session?.currentSessionId,
                    socketId = iLiveChatTeam.session?.socketId,
                    userTeam = iLiveChatTeam.team,
                    teamList = iLiveChatTeam.teamList,
                    location = null,
                    userProperty = iLiveChatTeam.userProperty
                )
                languageRepository.updatePreviousWithCurrent()
                removeLiveChatTeamOptionObject(false)
            }
            setLoading(false)
        }
    }

    fun endLiveChat() {
        if (!isInternetAvailable()) return

        viewModelScope.launch(Dispatchers.Default) {
            val connected = ensureLiveChatConnection()
            if (connected) {
                if (ILiveChatDataContainer.ILiveChatStatus.READY == liveChatStatus
                    || ILiveChatDataContainer.ILiveChatStatus.QUEUED == liveChatStatus
                ) {
                    liveChatRepository.terminateSession(
                        sessionId = liveChatData?.sessionId,
                        intId = iBotChatWindowConfig?.intId,
                        userId = sessionRepository.getUserId(),
                        endedBy = "Customer",
                        prevLangCode = languageRepository.getPreviousLanguage()
                    )
                } else if (ILiveChatDataContainer.ILiveChatStatus.CONNECTED == liveChatStatus) {
                    liveChatRepository.endLiveChat(
                        agentLangCode = liveChatData?.agentLangCode,
                        botId = liveChatData?.botId,
                        endedBy = "Customer",
                        sessionId = sessionRepository.getSessionId(),
                        userId = sessionRepository.getUserId(),
                        userLangCode = languageRepository.getCurrentLanguage()
                    )
                }
                liveChatRepository.saveLiveChatDetails(null)
            }
        }
    }

    private fun removeLiveChatTeamOptionObject(scrollToEnd: Boolean) {
        var optionObjectRemoved = false
        chatItemsFromUser.forEachIndexed { index, iChatItem ->
            if (iChatItem is IMultipleOptionsItem) {
                chatItemsFromUser[index] = iChatItem.makeACopy().apply {
                    if (this is IMultipleOptionsItem) {
                        options.forEach { option ->
                            if (option.optionObject is ILiveChatTeam) {
                                option.optionObject = null
                                optionObjectRemoved = true
                            }
                        }
                    }
                }
                return@forEachIndexed
            }
        }
        notifyChatItemsChange(optionObjectRemoved || scrollToEnd)
    }

    fun setLiveChatMedia(photoData: Uri?, contentResolver: ContentResolver) {
        viewModelScope.launch(Dispatchers.IO) {
            try {
                if (!isInternetAvailable()) {
                    setLiveChatMediaUploadError(
                        photoData,
                        ILiveChatMediaUploadFailure.NETWORK_FAILURE
                    )
                    return@launch
                }
                _liveChatMediaLiveData.postValue(
                    ILiveChatMediaData(
                        showPreview = true,
                        isUploading = true,
                        isSendingMessage = false
                    )
                )
                photoData?.let {
                    FileUtils.fetchFileMetadataFromUri(it, contentResolver)
                        ?.let { fileMetadata ->
                            FileUtils.createTempFileForUri(
                                getApplication(),
                                contentResolver,
                                photoData,
                                fileMetadata
                            )?.let { fileToUpload ->
                                iBotChatWindowConfig?.botId?.let { botId ->
                                    liveChatRepository.handleMedia(
                                        botId,
                                        fileToUpload,
                                        fileMetadata.mimeType
                                    )?.let { uploadedData ->
                                        _liveChatMediaLiveData.postValue(uploadedData)
                                        return@launch
                                    } ?: kotlin.run {
                                        if (NetworkUtils.isInternetAvailable(getApplication())) {
                                            setLiveChatMediaUploadError(
                                                photoData,
                                                ILiveChatMediaUploadFailure.SERVER_FAILURE
                                            )
                                        } else {
                                            setLiveChatMediaUploadError(
                                                photoData,
                                                ILiveChatMediaUploadFailure.NETWORK_FAILURE
                                            )
                                        }
                                    }
                                }
                            }
                        }
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            _liveChatMediaLiveData.postValue(
                ILiveChatMediaData(
                    showPreview = false,
                    isUploading = false,
                    isSendingMessage = false
                )
            )
        }
    }

    private fun setLiveChatMediaUploadError(
        photoData: Uri?,
        failureReason: ILiveChatMediaUploadFailure?
    ) {
        _liveChatMediaLiveData.postValue(
            ILiveChatMediaData(
                showPreview = true,
                isUploading = false,
                isSendingMessage = false,
                retryData = ILiveChatMediaRetryData(
                    failedMediaData = photoData,
                    failureReason = failureReason
                )
            )
        )
    }

    fun sendLiveChatPhotoAndMessage(liveChatPhotoData: ILiveChatMediaData, message: String?) {
        if (message?.isBlank() == true) return

        liveChatPhotoData.mediaText = message

        viewModelScope.launch(Dispatchers.IO) {
            _liveChatMediaLiveData.postValue(_liveChatMediaLiveData.value?.apply {
                isSendingMessage = true
            })
            val sent = sendLiveChatMessage(
                liveChatPhotoData,
                ApiConstants.CHAT_EXPECTED_RESPONSE_TYPE_MEDIA
            )
            if (sent) {
                _liveChatMediaLiveData.postValue(
                    ILiveChatMediaData(
                        showPreview = false,
                        isUploading = false,
                        isSendingMessage = false,
                        clearMessage = true
                    )
                )
            } else {
                _liveChatMediaLiveData.postValue(_liveChatMediaLiveData.value?.apply {
                    isSendingMessage = false
                })
            }
        }
    }

    private fun saveLiveChatDetails() {
        viewModelScope.launch {
            if (liveChatStatus == ILiveChatDataContainer.ILiveChatStatus.CONNECTED) {
                liveChatRepository.saveLiveChatDetails(liveChatData)
            } else {
                liveChatRepository.saveLiveChatDetails(null)
            }
        }
    }

    private fun resumeLiveChatIfSaved(liveChatData: ILiveChatData?) {
        liveChatData?.let {
            viewModelScope.launch {
                this@BotChatViewModel.liveChatData = it
                liveChatStatus = ILiveChatDataContainer.ILiveChatStatus.CONNECTED
                _liveChatStatusLiveData.postValue(liveChatStatus)
                ensureLiveChatConnection()
                joinLiveChatIfConnected()
            }
        }
    }

    fun handleLiveChatFeedbackSubmitClick(iLiveChatFeedbackRequest: ILiveChatFeedbackRequest?) {
        iLiveChatFeedbackRequest?.liveChatData?.let { lcd ->

            if (!isInternetAvailable()) return

            if (iLiveChatFeedbackRequest.additionalQuestion == null && iLiveChatFeedbackRequest.selectedIndex == 0) {
                showToast(getApplication<Application>().resources.getString(R.string.please_select_at_least_one_star))
                return
            }

            viewModelScope.launch(Dispatchers.IO) {
                setFeedbackLoading(true)
                val feedbackResponse =
                    if (iLiveChatFeedbackRequest.additionalQuestion == null) {
                        liveChatRepository.sendUserFeedback(
                            liveChatData = lcd,
                            iLiveChatFeedbackRequest.selectedIndex
                        )
                    } else {
                        liveChatRepository.sendUserAdditionalFeedback(
                            liveChatData = lcd,
                            additionalQuestion = iLiveChatFeedbackRequest.additionalQuestion
                        )
                    }
                when (feedbackResponse) {
                    null -> {
                        showToast(
                            getApplication<Application>().resources
                                .getString(R.string.failed_to_submit_feedback)
                        )
                        setFeedbackLoading(false)
                    }
                    is ILiveChatFeedbackAdditionalQuestion -> {
                        _userFeedbackLiveData.postValue(_userFeedbackLiveData.value?.apply {
                            this.additionalQuestion = feedbackResponse
                        })
                        setFeedbackLoading(false)
                    }
                    is ILiveChatDataContainer -> {
                        if (!feedbackResponse.items.isNullOrEmpty()) {
                            addNewItemsToChat(feedbackResponse.items)
                        }
                        _userFeedbackLiveData.postValue(null)
                    }
                }
            }
        }
    }

    private fun setFeedbackLoading(loading: Boolean) {
        _userFeedbackLiveData.postValue(_userFeedbackLiveData.value?.apply {
            isLoading = loading
        })
    }

    fun handleBotFeedbackPositiveClick(iBotFeedbackPositiveClick: IBotFeedbackPositiveClick) {
        iBotFeedbackPositiveClick.iBotFeedbackRequest.iBotFeedbackClickedStatus =
            IBotFeedbackRequest.IBotFeedbackClickedStatus.CLICKED_POSITIVE
        _botFeedbackLiveData.postValue(iBotFeedbackPositiveClick.iBotFeedbackRequest)
    }

    fun handleBotFeedbackNegativeClick(iBotFeedbackNegativeClick: IBotFeedbackNegativeClick) {
        iBotFeedbackNegativeClick.iBotFeedbackRequest.iBotFeedbackClickedStatus =
            IBotFeedbackRequest.IBotFeedbackClickedStatus.CLICKED_NEGATIVE
        _botFeedbackLiveData.postValue(iBotFeedbackNegativeClick.iBotFeedbackRequest)
    }

    fun handleBotFeedbackSubmitClick(iBotFeedbackRequest: IBotFeedbackRequest) {
        if (!isInternetAvailable()) {
            return
        }

        if (iBotFeedbackRequest.selectedIndex == 0) {
            showToast(getApplication<Application>().resources.getString(R.string.please_select_at_least_one_star))
            return
        }

        viewModelScope.launch(Dispatchers.IO) {
            setBotFeedbackSubmitting(true)
            try {
                val feedbackSet = botFeedbackRepository.setFeedback(
                    iBotChatWindowConfig?.intId,
                    sessionRepository.getSessionId()
                )
                if (feedbackSet) {
                    val csatGiven = botFeedbackRepository.giveCsat(
                        iBotChatWindowConfig?.intId,
                        sessionRepository.getSessionId(),
                        sessionRepository.getSocketId(),
                        iBotFeedbackRequest.selectedIndex
                    )
                    if (csatGiven) {
                        _botFeedbackLiveData.postValue(null)
                        _enableBotFeedbackLiveData.postValue(_enableBotFeedbackLiveData.value?.apply {
                            this.isSubmitting = false
                            if (iBotFeedbackRequest.iBotFeedbackClickedStatus == IBotFeedbackRequest.IBotFeedbackClickedStatus.CLICKED_POSITIVE) {
                                this.iBotFeedbackSubmitStatus =
                                    IBotFeedbackRequest.IBotFeedbackSubmitStatus.SUBMITTED_POSITIVE
                            } else if (iBotFeedbackRequest.iBotFeedbackClickedStatus == IBotFeedbackRequest.IBotFeedbackClickedStatus.CLICKED_NEGATIVE) {
                                this.iBotFeedbackSubmitStatus =
                                    IBotFeedbackRequest.IBotFeedbackSubmitStatus.SUBMITTED_NEGATIVE
                            }
                        })
                        return@launch
                    }
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            showToast(getApplication<Application>().resources.getString(R.string.failed_to_submit_feedback))
            setBotFeedbackSubmitting(false)
        }
    }

    private fun setBotFeedbackSubmitting(isSubmitting: Boolean) {
        _botFeedbackLiveData.postValue(_botFeedbackLiveData.value?.apply {
            this.isSubmitting = isSubmitting
        })
    }

    fun handleBotFeedbackJourney(iFeedbackJourney: IFeedbackJourney) {
        iFeedbackJourney.journeyName?.let { journeyName ->
            sendJourneyMessage(
                message = journeyName,
                fieldText = iFeedbackJourney.fieldText,
                isFromProcessTree = true
            )
        }
    }

    fun handleBotFeedbackCancelClick() {
        _botFeedbackLiveData.postValue(null)
    }

    fun handleUserFeedbackCancelClick() {
        _userFeedbackLiveData.postValue(null)
    }

    fun sendJourney(iSendJourney: ISendJourney) {
        sendJourneyMessage(
            message = iSendJourney.journeyName,
            fieldText = iSendJourney.fieldText,
            isFromProcessTree = iSendJourney.isFromProcessTree
        )
    }

    fun submitTabularInfo(iSubmitTabularResponse: ISubmitTabularResponse) {
        iSubmitTabularResponse.iTabularInfoItem?.let { tabularInfoItem ->
            viewModelScope.launch(Dispatchers.Default) {
                try {
                    if (!isInternetAvailable()) {
                        return@launch
                    }

                    tabularInfoItem.isLoading = true
                    updateChatItem(tabularInfoItem, false)

                    delay(300)
                    sendTabularInfo(
                        tabularInfoItem = tabularInfoItem,
                        nextDimension = null,
                        uniqueId = "${Date().time}"
                    )

                    tabularInfoItem.isLoading = false
                    updateChatItem(tabularInfoItem, false)
                } catch (e: Exception) {
                    LogUtils.logException(e)
                }
            }
        }
    }
}