package ai.engagely.openbot.model.repositories.impl

import ai.engagely.openbot.model.localstorage.SimpleKeyValueStorage
import ai.engagely.openbot.model.network.ApiClient
import ai.engagely.openbot.model.network.ApiConstants
import ai.engagely.openbot.model.network.interfaces.LiveChatApi
import ai.engagely.openbot.model.pojos.external.apirequests.livechat.*
import ai.engagely.openbot.model.pojos.internal.livechat.*
import ai.engagely.openbot.model.pojos.internal.location.ILocation
import ai.engagely.openbot.model.repositories.LiveChatRepository
import ai.engagely.openbot.model.utils.converters.ChatResponseConverter
import ai.engagely.openbot.model.utils.general.DateUtils
import ai.engagely.openbot.model.utils.general.FileUtils
import ai.engagely.openbot.model.utils.general.LogUtils
import ai.engagely.openbot.model.utils.general.RetrofitUtils
import com.google.gson.Gson
import io.socket.client.Ack
import io.socket.client.Socket
import kotlinx.coroutines.*
import okhttp3.MultipartBody
import okhttp3.RequestBody
import java.io.File
import java.net.HttpURLConnection
import kotlin.coroutines.resume

class AppLiveChatRepository(
    private val dispatcher: CoroutineDispatcher,
    private val socket: Socket,
    private val simpleKeyValueStorage: SimpleKeyValueStorage,
    private val chatResponseConverter: ChatResponseConverter
) : LiveChatRepository {

    override suspend fun getLiveAgentConnectionDetails(
        intId: String?,
        sessionId: String?
    ): ILiveChatForm? {
        return withContext(dispatcher) {
            try {
                val agentDetailsApi =
                    ApiClient.getInstance().getClient().create(LiveChatApi::class.java)
                val liveAgentConnectionDetailsRequest =
                    LiveAgentConnectionDetailsRequest(intId = intId, sessionId = sessionId)
                val response =
                    agentDetailsApi.getLiveAgentConnectionDetails(liveAgentConnectionDetailsRequest)
                if (response.isSuccessful && response.code() == HttpURLConnection.HTTP_OK
                    && response.body()?.success == true
                ) {
                    response.body()?.response?.let {
                        return@withContext ILiveChatForm(
                            displayField = it.displayField,
                            displayOutput = it.displayOutput,
                            field = it.field,
                            formData = it.formData,
                            inputMethod = it.inputMethod,
                            inputType = it.inputType,
                            settingId = it.settingId,
                            value = it.value,
                            userTeam = null,
                            chatItemIdsToRemove = null
                        )
                    }

                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            return@withContext null
        }
    }

    override suspend fun sendAgentConnectRequest(
        botId: String?,
        channel: String,
        displayOutput: String?,
        fetchApi: Boolean,
        formData: Any?,
        prevLangCode: String?,
        sessionId: String?,
        settingId: String?,
        userId: String?,
        userIntId: String?,
        userIntName: String?,
        osName: String?,
        osVersion: String?,
        screenWidth: Int?,
        screenHeight: Int?,
        screenOrientation: String?,
        userName: String?,
        currentSessionId: String?,
        socketId: String?,
        userTeam: String?,
        teamList: List<ILiveChatTeam>?,
        location: ILocation?,
        userProperty: Any?
    ): Boolean {

        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val agentConnectRequest = Gson().toJson(
                        AgentConnectRequest(
                            botId = botId,
                            channel = channel,
                            displayOutput = displayOutput,
                            fetchApi = fetchApi,
                            formData = formData,
                            prevLangCode = prevLangCode,
                            sessionId = sessionId,
                            settingId = settingId,
                            userId = userId,
                            userIntId = userIntId,
                            userIntName = userIntName,
                            userProperty = userProperty
                                ?: UserProperty(
                                    deviceInfo = DeviceInfo(
                                        browser = null,
                                        os = Os(
                                            name = osName,
                                            version = osVersion,
                                        ),
                                        screen = Screen(
                                            height = screenHeight,
                                            width = screenWidth,
                                            orientation = screenOrientation,
                                            pixelDepth = null
                                        )
                                    ),
                                    engagementInfo = null,
                                    entity = null,
                                    intent = null,
                                    locationInfo = if (location != null) LocationInfo(
                                        latitude = location.latitude,
                                        longitude = location.longitude
                                    ) else null
                                ),
                            userName = userName,
                            currentSessionId = currentSessionId,
                            socketId = socketId,
                            userTeam = userTeam,
                            teamList = teamList?.map { iTeam ->
                                Team(iTeam.team, iTeam.translatedTeam)
                            }
                        )
                    )
                    emitMessage(
                        ApiConstants.CHANNEL_AGENT_CONNECT_REQUEST,
                        agentConnectRequest,
                        cnt
                    )
                }
            } ?: false
        }
    }

    override suspend fun sendJoinMeHereRequest(liveChatData: ILiveChatData): Boolean {
        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val joinMeHereRequest = Gson().toJson(
                        JoinMeHereRequest(
                            agentEmailId = liveChatData.agentEmailId,
                            agentIntId = liveChatData.agentIntId,
                            agentIntName = liveChatData.agentIntName,
                            agentLangCode = liveChatData.agentLangCode,
                            agentName = liveChatData.agentName,
                            agentSessionId = liveChatData.agentSessionId,
                            agentTeam = liveChatData.agentTeam,
                            availableChats = liveChatData.availableChats,
                            botId = liveChatData.botId,
                            chattingWith = liveChatData.chattingWith,
                            currentChatCount = liveChatData.currentChatCount,
                            imageName = liveChatData.imageName,
                            isAvailable = liveChatData.isAvailable,
                            maxChatCount = liveChatData.maxChatCount,
                            onBreak = liveChatData.onBreak,
                            role = liveChatData.role,
                            roomId = liveChatData.roomId,
                            shortMessage = liveChatData.shortMessage,
                            unreadEmailCount = liveChatData.unreadEmailCount,
                            totalEmailCount = liveChatData.totalEmailCount,
                            channel = liveChatData.channel,
                            userIntId = liveChatData.userIntId,
                            userId = liveChatData.userId,
                            userLangCode = liveChatData.userLangCode
                        )
                    )
                    emitMessage(ApiConstants.CHANNEL_JOIN_ME_HERE, joinMeHereRequest, cnt)
                }
            } ?: false
        }
    }

    override suspend fun sendMessage(
        liveChatData: ILiveChatData?,
        channel: String?,
        intId: String?,
        intName: String?,
        prevLangCode: String?,
        queryType: String?,
        sessionId: String?,
        sessionObj: Any?,
        userId: String?,
        userInput: Any?,
        userLang: String?,
        userLangCode: String?,
        responseType: String?
    ): Boolean {
        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val liveChatMessageRequest = Gson().toJson(
                        LiveChatMessageRequest(
                            agentDetails = AgentDetails(
                                agentEmailId = liveChatData?.agentEmailId,
                                agentIntId = liveChatData?.agentIntId,
                                agentIntName = liveChatData?.agentIntName,
                                agentLangCode = liveChatData?.agentLangCode,
                                agentName = liveChatData?.agentName,
                                agentSessionId = liveChatData?.agentSessionId,
                                agentTeam = liveChatData?.agentTeam,
                                availableChats = liveChatData?.availableChats,
                                botId = liveChatData?.botId,
                                chattingWith = liveChatData?.chattingWith,
                                currentChatCount = liveChatData?.currentChatCount,
                                imageName = "",
                                isAvailable = liveChatData?.isAvailable,
                                maxChatCount = liveChatData?.maxChatCount,
                                onBreak = liveChatData?.onBreak,
                                role = liveChatData?.role,
                                roomId = liveChatData?.roomId,
                                shortMessage = ""
                            ),
                            agentLangCode = liveChatData?.agentLangCode,
                            botId = liveChatData?.botId,
                            channel = "bot",
                            currentSessionId = liveChatData?.currentSessionId,
                            inputDateNtime = DateUtils.getCurrentTimeInISO8601(),
                            intId = intId,
                            intName = intName,
                            pageUrl = "",
                            prevLangCode = prevLangCode,
                            queryType = when (userInput) {
                                is String -> queryType
                                is ILiveChatMediaData -> {
                                    null
                                }
                                else -> null
                            },
                            resBy = "user",
                            resTime = DateUtils.getCurrentTime(),
                            responseType = responseType,
                            sessionId = sessionId,
                            sessionObj = sessionObj,
                            userId = userId,
                            userInput = when (userInput) {
                                is String -> userInput
                                is ILiveChatMediaData -> {
                                    LiveChatPhotoInput(
                                        mediaName = userInput.mediaName,
                                        mediaSize = userInput.mediaSize,
                                        mediaText = userInput.mediaText,
                                        mediaThumbnail = userInput.mediaThumbnail,
                                        mediaType = userInput.mediaType,
                                        mediaUrl = userInput.uploadedUrl
                                    )
                                }
                                else -> null
                            },
                            userIntId = liveChatData?.userIntId,
                            userIntName = liveChatData?.userIntName,
                            userLang = userLang,
                            userLangCode = userLangCode,
                            userTeam = liveChatData?.userTeam
                        )
                    )
                    emitMessage(ApiConstants.CHANNEL_LIVECHAT_USER, liveChatMessageRequest, cnt)
                }
            } ?: false
        }
    }

    override suspend fun endLiveChat(
        agentLangCode: String?,
        botId: String?,
        endedBy: String?,
        sessionId: String?,
        userId: String?,
        userLangCode: String?
    ): Boolean {
        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val endChatRequest = Gson().toJson(
                        EndChatRequest(
                            agentLangCode = agentLangCode,
                            botId = botId,
                            endedBy = endedBy,
                            sessionId = sessionId,
                            userId = userId,
                            userLangCode = userLangCode
                        )
                    )
                    emitMessage(ApiConstants.CHANNEL_USER_END_CHAT, endChatRequest, cnt)
                }
            } ?: false
        }
    }

    override suspend fun terminateSession(
        sessionId: String?,
        intId: String?,
        userId: String?,
        endedBy: String?,
        prevLangCode: String?
    ): Boolean {
        return withContext(dispatcher) {
            return@withContext withTimeoutOrNull(ApiConstants.SOCKET_TIMEOUT) {
                suspendCancellableCoroutine { cnt ->
                    val terminateSessionRequest = Gson().toJson(
                        TerminateSessionRequest(
                            sessionId = sessionId,
                            intId = intId,
                            userId = userId,
                            endedBy = endedBy,
                            prevLangCode = prevLangCode
                        )
                    )
                    emitMessage(
                        ApiConstants.CHANNEL_LIVE_TERMINATE_SESSION,
                        terminateSessionRequest,
                        cnt
                    )
                }
            } ?: false
        }
    }

    override suspend fun handleMedia(
        botId: String,
        file: File,
        mimeType: String
    ): ILiveChatMediaData? {
        return withContext(dispatcher) {
            try {
                val handleMediaApi =
                    ApiClient.getInstance().getClient().create(LiveChatApi::class.java)

                val partValues: HashMap<String, RequestBody> = HashMap()
                val partFiles: ArrayList<MultipartBody.Part> = ArrayList()

                partValues["bot_id"] = RetrofitUtils.createPartFromString(botId)
                partFiles.add(RetrofitUtils.prepareFilePart("file", file, mimeType))
                val response = handleMediaApi.handleMedia(partValues, partFiles)
                FileUtils.deleteParentAndSelfIfExists(file)
                if (response.isSuccessful && response.code() == HttpURLConnection.HTTP_OK
                    && response.body()?.success == true
                ) {
                    response.body()?.data?.response?.let { response ->
                        return@withContext ILiveChatMediaData(
                            showPreview = true,
                            isUploading = false,
                            isSendingMessage = false,
                            uploadedUrl = response.mediaUrl,
                            mediaType = response.mediaType,
                            mediaThumbnail = response.mediaThumbnail,
                            mediaText = null,
                            mediaSize = response.mediaSize,
                            mediaName = response.mediaName
                        )
                    }
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            return@withContext null
        }
    }

    override suspend fun saveLiveChatDetails(liveChatData: ILiveChatData?) {
        withContext(dispatcher) {
            liveChatData?.let {
                try {
                    val liveChatDataString = Gson().toJson(it)
                    simpleKeyValueStorage.storeStringDataForKey(
                        KEY_LIVE_CHAT_DATA,
                        liveChatDataString
                    )
                } catch (e: Exception) {
                    LogUtils.logException(e)
                }
            } ?: kotlin.run {
                simpleKeyValueStorage.storeStringDataForKey(
                    KEY_LIVE_CHAT_DATA,
                    null
                )
            }
        }

    }

    override suspend fun readLiveChatDetails(): ILiveChatData? {
        return withContext(dispatcher) {
            try {
                val liveChatDataString =
                    simpleKeyValueStorage.readStringDataForKey(KEY_LIVE_CHAT_DATA)
                if (liveChatDataString?.isNotBlank() == true) {
                    return@withContext Gson().fromJson(
                        liveChatDataString,
                        ILiveChatData::class.java
                    )
                }
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            return@withContext null
        }
    }

    override suspend fun sendUserFeedback(
        liveChatData: ILiveChatData,
        selectedFeedbackIndex: Int
    ): Any? {
        return withContext(dispatcher) {
            return@withContext forwardUserFeedback(
                liveChatData = liveChatData,
                selectedFeedbackIndex = selectedFeedbackIndex,
                additionalQuestion = null
            )
        }
    }

    private suspend fun forwardUserFeedback(
        liveChatData: ILiveChatData,
        selectedFeedbackIndex: Int?,
        additionalQuestion: ILiveChatFeedbackAdditionalQuestion?
    ): Any? {
        try {
            val userFeedbackRequest =
                UserFeedbackRequest(
                    agentEmailId = liveChatData.agentEmailId,
                    agentIntId = liveChatData.agentIntId,
                    agentLangCode = liveChatData.agentLangCode,
                    agentName = liveChatData.agentName,
                    botId = liveChatData.botId,
                    channel = liveChatData.channel,
                    currentSessionId = liveChatData.currentSessionId,
                    resolved = liveChatData.resolved,
                    roomId = liveChatData.roomId,
                    sessionId = liveChatData.sessionId,
                    userFeedback = selectedFeedbackIndex,
                    userId = liveChatData.userId,
                    userIntId = liveChatData.userIntId,
                    userIntName = liveChatData.userIntName,
                    userLangCode = liveChatData.userLangCode,
                    userName = liveChatData.userName,
                    userTeam = liveChatData.userSelectedTeam,
                    responseMessage = additionalQuestion?.messages,
                    message = additionalQuestion?.messages,
                    isPositive = additionalQuestion?.isPositive,
                    type = additionalQuestion?.type,
                    surveyAnswer = additionalQuestion?.additionalAnswer
                )
            val liveChatApi =
                ApiClient.getInstance().getClient().create(LiveChatApi::class.java)
            val response =
                liveChatApi.submitUserFeedback(userFeedbackRequest)
            if (response.isSuccessful && response.code() == HttpURLConnection.HTTP_OK
                && response.body()?.success == true
            ) {
                response.body()?.let { additionalFeedbackDetails ->
                    return chatResponseConverter.convertFeedbackSubmitResponseToChatItems(
                        liveChatData,
                        additionalFeedbackDetails
                    )
                }
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
        }
        return null
    }

    override suspend fun sendUserAdditionalFeedback(
        liveChatData: ILiveChatData,
        additionalQuestion: ILiveChatFeedbackAdditionalQuestion?
    ): Any? {
        return withContext(dispatcher) {
            try {
                return@withContext forwardUserFeedback(
                    liveChatData = liveChatData,
                    selectedFeedbackIndex = null,
                    additionalQuestion = additionalQuestion
                )
            } catch (e: Exception) {
                LogUtils.logException(e)
            }
            return@withContext null
        }
    }

    private fun emitMessage(
        channelToSendTo: String,
        flowRequest: String?,
        cnt: CancellableContinuation<Boolean?>
    ) {
        if (socket.connected()) {
            LogUtils.log("Emitting on channel = ${channelToSendTo}, message = $flowRequest")
            socket.emit(channelToSendTo, flowRequest, object : Ack {
                override fun call(vararg args: Any?) {
                    LogUtils.log("Callback from emitting $channelToSendTo $args")
                    cnt.resume(true)
                }
            })
        } else {
            cnt.resume(false)
        }
    }

    companion object {
        const val KEY_LIVE_CHAT_DATA = "live_chat_data"
    }
}