package ai.engagely.openbot.view.activities

import ai.engagely.openbot.R
import ai.engagely.openbot.databinding.ActivityExoPlayerBinding
import ai.engagely.openbot.model.utils.exts.showToast
import ai.engagely.openbot.model.utils.general.LogUtils
import ai.engagely.openbot.view.customviews.PlaybackSpeedMenu
import ai.engagely.openbot.view.customviews.PlayerMoreMenuView
import ai.engagely.openbot.viewmodel.VideoPlayerViewModel
import android.app.Activity
import android.app.PendingIntent
import android.app.PictureInPictureParams
import android.app.RemoteAction
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.graphics.Rect
import android.graphics.drawable.Icon
import android.os.Build
import android.os.Bundle
import android.util.Rational
import android.view.View
import android.widget.ImageButton
import androidx.annotation.RequiresApi
import androidx.lifecycle.ViewModelProvider
import androidx.localbroadcastmanager.content.LocalBroadcastManager

class ExoPlayerActivity : BaseActivity() {

    private lateinit var binding: ActivityExoPlayerBinding
    private var isFullScreen = false
    private lateinit var fullScreenButton: ImageButton
    private var volumeHigh: ImageButton? = null
    private var volumeOff: ImageButton? = null
    private var playbackSpeedMenu: PlaybackSpeedMenu? = null
    private var playerMoreMenuView: PlayerMoreMenuView? = null

    private val playerViewModel by lazy {
        ViewModelProvider(this)[VideoPlayerViewModel::class.java]
    }

    private var isInPIPMode: Boolean = false
    private var pipWidth: Int = 0
    private var pipHeight: Int = 0
    private var pipBroadcastNeeded = false

    private val playbackPauseFromPIPReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context?, intent: Intent?) {
            playerViewModel.pausePlayer()
            syncPIPParams()
        }
    }

    private val playbackPlayFromPIPReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context?, intent: Intent?) {
            playerViewModel.playPlayer()
            syncPIPParams()
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        binding = ActivityExoPlayerBinding.inflate(layoutInflater)
        setContentView(binding.root)

        lifecycle.addObserver(playerViewModel)

        setupVideo(intent)
        setupActionBar()
        setupVolumeButton()
        setupFullScreenButton()
        setupPlaybackSpeedButton()
        setupDownloadOption()
        setupBroadcastReceivers()
    }

    override fun onNewIntent(intent: Intent?) {
        super.onNewIntent(intent)
        sendStopPipMessageIfStarted()
        processPIPModeChange(false)
        setupVideo(intent)
    }

    private fun setupBroadcastReceivers() {
        registerReceiver(
            playbackPauseFromPIPReceiver,
            IntentFilter(VIDEO_PLAYER_PAUSE_UPDATE_FROM_PIP_BROADCAST_EVENT)
        )
        registerReceiver(
            playbackPlayFromPIPReceiver,
            IntentFilter(VIDEO_PLAYER_PLAY_UPDATE_FROM_PIP_BROADCAST_EVENT)
        )
    }

    private fun setupDownloadOption() {
        playerMoreMenuView = findViewById(R.id.playerMoreMenu)

        playerMoreMenuView?.setData(object : PlayerMoreMenuView.PlayerMoreMenuClickListener {
            override fun onDownloadMenuSelected() {
                playerViewModel.downloadVideo()
            }

            override fun onPictureInPictureSelected() {
                openPIPMode(null)
            }
        })
    }

    private fun openPIPMode(isPlaying: Boolean?) {
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N && packageManager.hasSystemFeature(
                    PackageManager.FEATURE_PICTURE_IN_PICTURE
                )
            ) {
                isInPIPMode = true
                hideToolbar(true)
                binding.playerView.hideController()
                binding.playerView.useController = false
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    enterPictureInPictureMode(createPictureInPictureParams(isPlaying))
                } else {
                    enterPictureInPictureMode()
                }
            } else {
                showToast(getString(R.string.service_unavailable))
            }
        } catch (e: Exception) {
            LogUtils.logException(e)
            showToast(getString(R.string.service_unavailable))
        }
    }

    @RequiresApi(Build.VERSION_CODES.O)
    private fun createPictureInPictureParams(
        isPlaying: Boolean?
    ): PictureInPictureParams {
        var width = pipWidth
        var height = pipHeight
        if (width == 0 || height == 0) {
            width = 240
            height = 160
        }
        val aspectRatio = Rational(width, height)
        val visibleRect = Rect()
        binding.playerView.getGlobalVisibleRect(visibleRect)

        val actions: ArrayList<RemoteAction> = ArrayList()

        if (playerViewModel.player?.isPlaying == true || isPlaying == true) {
            val pauseAction = RemoteAction(
                Icon.createWithResource(
                    this,
                    R.drawable.ic_pause
                ),
                "Pause", "To pause the playback",
                PendingIntent.getBroadcast(
                    this, VIDEO_PLAYER_PAUSE_UPDATE_FROM_PIP_REQUEST_CODE,
                    Intent(VIDEO_PLAYER_PAUSE_UPDATE_FROM_PIP_BROADCAST_EVENT),
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) PendingIntent.FLAG_IMMUTABLE
                    else PendingIntent.FLAG_UPDATE_CURRENT
                )
            )
            actions.add(pauseAction)
        } else {
            val playAction = RemoteAction(
                Icon.createWithResource(
                    this,
                    R.drawable.ic_play
                ),
                "Play", "To play the playback",
                PendingIntent.getBroadcast(
                    this, VIDEO_PLAYER_PLAY_UPDATE_FROM_PIP_REQUEST_CODE,
                    Intent(VIDEO_PLAYER_PLAY_UPDATE_FROM_PIP_BROADCAST_EVENT),
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) PendingIntent.FLAG_IMMUTABLE
                    else PendingIntent.FLAG_UPDATE_CURRENT
                )
            )
            actions.add(playAction)
        }

        val params = PictureInPictureParams.Builder()
            .setAspectRatio(aspectRatio)
            // Specify the portion of the screen that turns into the picture-in-picture mode.
            // This makes the transition animation smoother.
            .setSourceRectHint(visibleRect)
            .setActions(actions)
            .build()
        setPictureInPictureParams(params)
        return params
    }

    override fun onPictureInPictureModeChanged(
        isInPictureInPictureMode: Boolean,
        newConfig: Configuration?
    ) {
        super.onPictureInPictureModeChanged(isInPictureInPictureMode, newConfig)
        processPIPModeChange(isInPictureInPictureMode)
    }

    private fun processPIPModeChange(isInPictureInPictureMode: Boolean) {
        if (pipBroadcastNeeded) {
            LocalBroadcastManager.getInstance(this@ExoPlayerActivity)
                .sendBroadcast(Intent(VIDEO_PLAYER_PIP_BROADCAST_EVENT).apply {
                    putExtra(VIDEO_IS_IN_PICTURE_IN_PICTURE_MODE, isInPictureInPictureMode)
                })
        }

        if (isInPictureInPictureMode) {
            hideControls()
            binding.playerView.hideController()
            binding.playerView.useController = false
        } else {
            showControls()
            binding.playerView.useController = true
        }
        isInPIPMode = isInPictureInPictureMode
    }

    private fun syncPIPParams() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            createPictureInPictureParams(null)
        }
    }

    private fun setupPlaybackSpeedButton() {
        playbackSpeedMenu = findViewById(R.id.playbackSpeedMenu)

        val playbackSpeedListener = object : PlaybackSpeedMenu.PlaybackSpeedListener {
            override fun onPlaybackSpeedChanged(speed: Float) {
                playerViewModel.setPlaybackSpeed(speed)
            }
        }
        playerViewModel.playbackSpeedLiveData.observe(this) {
            playbackSpeedMenu?.setData(it ?: 1f, playbackSpeedListener)
        }
    }

    private fun setupVolumeButton() {
        volumeHigh = findViewById(R.id.ib_volume_high)
        volumeOff = findViewById(R.id.ib_volume_off)
        volumeHigh?.setOnClickListener {
            playerViewModel.toggleVolume()
        }
        volumeOff?.setOnClickListener {
            playerViewModel.toggleVolume()
        }

        playerViewModel.volumeMutedLiveData.observe(this) {
            val showMuted = it == true
            volumeOff?.visibility = if (showMuted) View.VISIBLE else View.GONE
            volumeHigh?.visibility = if (showMuted) View.GONE else View.VISIBLE
        }
    }

    private fun setupFullScreenButton() {
        fullScreenButton = findViewById(R.id.ib_full_screen)
        fullScreenButton.setOnClickListener {
            toggleFullScreen()
        }
    }

    private fun setupVideo(intent: Intent?) {
        intent?.getStringExtra(VIDEO_URL_TAG)?.let { url ->
            playerViewModel.updateInitialData(
                intent.getLongExtra(VIDEO_CURRENT_POSITION_TAG, 0),
                intent.getBooleanExtra(VIDEO_PLAY_WHEN_READY_TAG, true),
                intent.getFloatExtra(VIDEO_PLAYBACK_SPEED_TAG, 1f),
                url
            )
            playerViewModel.startPlaying(url)
            playerViewModel.playerLiveData.observe(this) {
                binding.playerView.player = it
            }

            pipWidth = intent.getIntExtra(VIDEO_PICTURE_IN_PICTURE_WIDTH, 0)
            pipHeight = intent.getIntExtra(VIDEO_PICTURE_IN_PICTURE_HEIGHT, 0)
            pipBroadcastNeeded =
                intent.getBooleanExtra(VIDEO_PICTURE_IN_PICTURE_BROADCAST_NEEDED, false)
            if (intent.getBooleanExtra(VIDEO_PLAY_IN_PICTURE_IN_PICTURE_MODE, false)) {
                openPIPMode(
                    intent.getBooleanExtra(VIDEO_PICTURE_IN_PICTURE_IS_PLAYING, false)
                )
            } else {
                hideToolbar(false)
            }

            playerViewModel.isPlayingLiveData.observe(this) {
                if (it != null) {
                    syncPIPParams()
                }
            }
        }
    }

    private fun setupActionBar() {
        setSupportActionBar(binding.toolbar)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.setHomeAsUpIndicator(R.drawable.ic_arrow_left_white)
        supportActionBar?.title = intent.getStringExtra(VIDEO_TITLE_TAG) ?: ""
        supportActionBar?.subtitle =
            intent.getStringExtra(VIDEO_SUBTITLE_TAG) ?: ""
    }

    private fun toggleFullScreen() {
        if (!isFullScreen) {
            hideControls()
        } else {
            showControls()
        }
    }

    private fun showControls() {
        unhideSystemBars()
        hideToolbar(false)
        setFullScreenButtonResource(R.drawable.ic_fullscreen)
        isFullScreen = false
    }

    private fun hideControls() {
        hideSystemBars()
        hideToolbar(true)
        setFullScreenButtonResource(R.drawable.ic_fullscreen_exit)
        isFullScreen = true
    }

    private fun hideToolbar(shouldHide: Boolean) {
        if (!shouldHide && isInPIPMode) {
            binding.toolbar.visibility = View.GONE
            return
        }
        binding.toolbar.visibility = if (shouldHide) View.GONE else View.VISIBLE
    }

    private fun setFullScreenButtonResource(imageResId: Int) {
        if (this::fullScreenButton.isInitialized) {
            fullScreenButton.setImageResource(imageResId)
        }
    }

    override fun onSupportNavigateUp(): Boolean {
        onBackPressed()
        return super.onSupportNavigateUp()
    }

    override fun onBackPressed() {
        if (isLandscapeOrientation()) {
            setPortraitOrientation()
        } else {
            setResult(
                Activity.RESULT_OK,
                getResponseIntent(
                    playerViewModel.getCurrentPosition(),
                    playerViewModel.getPlayWhenReady(),
                    playerViewModel.getPlaybackSpeed(),
                    playerViewModel.getUrl()
                )
            )
            playerViewModel.stopPlaying()
            super.onBackPressed()
        }
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            hideControls()
        } else if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
            showControls()
        }
    }

    override fun onStop() {
        super.onStop()
        sendStopPipMessageIfStarted()
        if (isInPIPMode) {
            finish()
        }
    }

    private fun sendStopPipMessageIfStarted() {
        if (pipBroadcastNeeded) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                if (isInPIPMode) {
                    LocalBroadcastManager.getInstance(this@ExoPlayerActivity)
                        .sendBroadcast(Intent(VIDEO_PLAYER_POSITION_BROADCAST_EVENT).apply {
                            putExtra(
                                VIDEO_CURRENT_POSITION_TAG,
                                playerViewModel.getSavedCurrentPosition()
                            )
                            putExtra(VIDEO_PLAYBACK_SPEED_TAG, playerViewModel.getPlaybackSpeed())
                        })
                }
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        clearBroadcastReceivers()
    }

    private fun clearBroadcastReceivers() {
        unregisterReceiver(playbackPauseFromPIPReceiver)
        unregisterReceiver(playbackPlayFromPIPReceiver)
    }

    companion object {
        const val VIDEO_URL_TAG = "ExoPlayer_Video_Url"
        const val VIDEO_TITLE_TAG = "ExoPlayer_Video_Title"
        const val VIDEO_SUBTITLE_TAG = "ExoPlayer_Video_Subtitle"
        const val VIDEO_CURRENT_POSITION_TAG = "ExoPlayer_Current_Position_Subtitle"
        const val VIDEO_PLAY_WHEN_READY_TAG = "ExoPlayer_Play_When_Ready_Subtitle"
        const val VIDEO_PLAYBACK_SPEED_TAG = "ExoPlayer_Playback_Speed"
        const val VIDEO_PLAYER_PIP_BROADCAST_EVENT = "ExoPlayer_PIP_Broadcast_Event"
        const val VIDEO_PLAYER_POSITION_BROADCAST_EVENT = "ExoPlayer_Position_Broadcast_Event"
        const val VIDEO_IS_IN_PICTURE_IN_PICTURE_MODE = "ExoPlayer_Is_In_Picture_In_Picture_Mode"
        const val VIDEO_PLAY_IN_PICTURE_IN_PICTURE_MODE =
            "ExoPlayer_Play_In_Picture_In_Picture_Mode"
        const val VIDEO_PICTURE_IN_PICTURE_WIDTH = "ExoPlayer_Picture_In_Picture_Width"
        const val VIDEO_PICTURE_IN_PICTURE_HEIGHT = "ExoPlayer_Picture_In_Picture_Height"
        const val VIDEO_PICTURE_IN_PICTURE_IS_PLAYING = "ExoPlayer_Picture_In_Picture_Is_Playing"
        const val VIDEO_PICTURE_IN_PICTURE_BROADCAST_NEEDED =
            "ExoPlayer_Picture_In_Picture_Broadcast_Needed"

        const val VIDEO_PLAYER_PAUSE_UPDATE_FROM_PIP_BROADCAST_EVENT =
            "ExoPlayer_Pause_Update_From_Pip_Broadcast_Event"
        const val VIDEO_PLAYER_PLAY_UPDATE_FROM_PIP_BROADCAST_EVENT =
            "ExoPlayer_Play_Update_From_Pip_Broadcast_Event"
        const val VIDEO_PLAYER_PAUSE_UPDATE_FROM_PIP_REQUEST_CODE = 2001
        const val VIDEO_PLAYER_PLAY_UPDATE_FROM_PIP_REQUEST_CODE = 2002

        fun getRequestIntent(
            activity: Activity,
            currentPosition: Long,
            playWhenReady: Boolean,
            speed: Float,
            url: String,
            playInPIP: Boolean,
            pipWidth: Int,
            pipHeight: Int,
            isPlaying: Boolean?,
            pipBroadcastNeeded: Boolean
        ): Intent {
            return Intent(activity, ExoPlayerActivity::class.java).apply {
                putExtra(
                    VIDEO_CURRENT_POSITION_TAG,
                    currentPosition
                )
                putExtra(
                    VIDEO_PLAY_WHEN_READY_TAG,
                    playWhenReady
                )
                putExtra(VIDEO_URL_TAG, url)
                putExtra(VIDEO_PLAYBACK_SPEED_TAG, speed)
                putExtra(VIDEO_PLAY_IN_PICTURE_IN_PICTURE_MODE, playInPIP)
                putExtra(VIDEO_PICTURE_IN_PICTURE_WIDTH, pipWidth)
                putExtra(VIDEO_PICTURE_IN_PICTURE_HEIGHT, pipHeight)
                putExtra(VIDEO_PICTURE_IN_PICTURE_IS_PLAYING, isPlaying)
                putExtra(VIDEO_PICTURE_IN_PICTURE_BROADCAST_NEEDED, pipBroadcastNeeded)
            }
        }

        fun getResponseIntent(
            currentPosition: Long,
            playWhenReady: Boolean,
            speed: Float,
            url: String?
        ): Intent {
            return Intent().apply {
                putExtra(
                    VIDEO_CURRENT_POSITION_TAG,
                    currentPosition
                )
                putExtra(
                    VIDEO_PLAY_WHEN_READY_TAG,
                    playWhenReady
                )
                putExtra(VIDEO_URL_TAG, url)
                putExtra(VIDEO_PLAYBACK_SPEED_TAG, speed)
            }
        }
    }
}