package ai.engagely.openbot.view.activities

import ai.engagely.openbot.R
import ai.engagely.openbot.model.pojos.internal.location.ILocationFetchStatus
import ai.engagely.openbot.model.utils.exts.showToast
import ai.engagely.openbot.model.utils.general.LogUtils
import android.Manifest
import android.annotation.SuppressLint
import android.content.Intent
import android.content.IntentSender
import android.content.pm.PackageManager
import android.os.Looper
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.lifecycle.lifecycleScope
import com.google.android.gms.common.api.ResolvableApiException
import com.google.android.gms.location.*
import com.google.android.gms.tasks.Task
import kotlinx.coroutines.*

abstract class PermissionCompatActivity : AppCompatActivity() {

    private var fusedLocationClient: FusedLocationProviderClient? = null
    private var locationCallback: LocationCallback? = null
    private var locationDependantObject: Any? = null
    private var locationRequestFallbackJob: Job? = null
    private var readExternalStorageRequestCode: Int = 0
    private var accessCameraRequestCode: Int = 0
    private var accessVideoCameraRequestCode: Int = 0
    private val readExternalStoragePermissionListeners: HashMap<Int, ReadExternalStoragePermissionListener> by lazy {
        HashMap()
    }

    private fun checkForLocationServices() {
        val locationRequest = createLocationRequest()

        val builder = LocationSettingsRequest.Builder()
            .addLocationRequest(locationRequest)

        val client: SettingsClient = LocationServices.getSettingsClient(this)
        val task: Task<LocationSettingsResponse> = client.checkLocationSettings(builder.build())
        task.addOnSuccessListener {
            // All location settings are satisfied.
            checkForLocationPermission()
        }

        task.addOnFailureListener { exception ->
            if (exception is ResolvableApiException) {
                // Location settings are not satisfied
                try {
                    // Show the dialog by calling startResolutionForResult(),
                    // and check the result in onActivityResult().
                    exception.startResolutionForResult(
                        this,
                        REQUEST_CHECK_SETTINGS
                    )
                } catch (sendEx: IntentSender.SendIntentException) {
                    onLocationServicesResult(ILocationFetchStatus.SERVICE_NOT_ENABLED)
                }
            } else {
                onLocationServicesResult(ILocationFetchStatus.SERVICE_NOT_ENABLED)
            }
        }
    }

    private fun createLocationRequest(): LocationRequest {
        val locationRequest = LocationRequest.create().apply {
            interval = 3000
            fastestInterval = 2000
            priority = LocationRequest.PRIORITY_HIGH_ACCURACY
        }
        return locationRequest
    }

    private fun checkForLocationPermission() {
        if ((ContextCompat.checkSelfPermission(
                this, Manifest.permission.ACCESS_COARSE_LOCATION
            )
                    != PackageManager.PERMISSION_GRANTED) ||
            ContextCompat.checkSelfPermission(
                this, Manifest.permission.ACCESS_FINE_LOCATION
            )
            != PackageManager.PERMISSION_GRANTED
        ) {
            ActivityCompat.requestPermissions(
                this, arrayOf(
                    Manifest.permission.ACCESS_COARSE_LOCATION,
                    Manifest.permission.ACCESS_FINE_LOCATION
                ),
                PERMISSIONS_ACCESS_COARSE_LOCATION
            )
        } else {
            onLocationServicesResult(ILocationFetchStatus.PERMISSION_GRANTED)
        }
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        when (requestCode) {
            REQUEST_CHECK_SETTINGS -> if (resultCode == RESULT_OK) {
                checkForLocationPermission()
            } else {
                onLocationServicesResult(ILocationFetchStatus.SERVICE_NOT_ENABLED)
            }
        }
    }

    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        when (requestCode) {
            PERMISSIONS_ACCESS_COARSE_LOCATION -> if (grantResults.isNotEmpty()) {
                if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    onLocationServicesResult(ILocationFetchStatus.PERMISSION_GRANTED)
                } else {
                    onLocationServicesResult(ILocationFetchStatus.PERMISSION_DENIED)
                }
            }
            PERMISSIONS_ACCESS_READ_EXTERNAL_STORAGE -> if (grantResults.isNotEmpty()) {
                if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    onReadExternalStoragePermissionResponse(
                        readExternalStorageRequestCode, ReadExternalStoragePermission.GRANTED
                    )
                } else {
                    showToast(getString(R.string.storage_permission_not_granted))
                    onReadExternalStoragePermissionResponse(
                        readExternalStorageRequestCode, ReadExternalStoragePermission.DENIED
                    )
                }
            }
            PERMISSIONS_ACCESS_CAMERA -> if (grantResults.isNotEmpty()) {
                if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    onCameraPermissionResponse(accessCameraRequestCode, CameraPermission.GRANTED)
                } else {
                    showToast(getString(R.string.camera_permission_not_granted))
                    onCameraPermissionResponse(accessCameraRequestCode, CameraPermission.DENIED)
                }
            }
            PERMISSIONS_ACCESS_VIDEO_CAMERA -> if (grantResults.isNotEmpty()) {
                if (grantResults.size == 2 && grantResults[0] == PackageManager.PERMISSION_GRANTED
                    && grantResults[1] == PackageManager.PERMISSION_GRANTED
                ) {
                    onVideoCameraPermissionResponse(
                        accessVideoCameraRequestCode, CameraPermission.GRANTED
                    )
                } else {
                    showToast(getString(R.string.video_camera_permission_not_granted))
                    onVideoCameraPermissionResponse(
                        accessVideoCameraRequestCode, CameraPermission.DENIED
                    )
                }
            }
        }
    }

    private fun onLocationServicesResult(iLocationFetchStatus: ILocationFetchStatus) {
        when (iLocationFetchStatus) {
            ILocationFetchStatus.PERMISSION_GRANTED -> getLastKnownLocation()
            ILocationFetchStatus.PERMISSION_DENIED -> showToast(getString(R.string.location_permission_denied))
            ILocationFetchStatus.SERVICE_NOT_ENABLED -> showToast(getString(R.string.location_service_not_enabled))
            ILocationFetchStatus.FETCHING_FAILED -> showToast(getString(R.string.current_location_unavailable))
        }
        if (iLocationFetchStatus != ILocationFetchStatus.PERMISSION_GRANTED) {
            onLocationNotFetched(locationDependantObject)
        }
    }

    @SuppressLint("MissingPermission")
    private fun getLastKnownLocation() {
        fusedLocationClient = LocationServices.getFusedLocationProviderClient(this)
        fusedLocationClient?.lastLocation?.addOnCompleteListener {
            if (it.result == null) {
                //Uh oh! :( No last known location
                fetchCurrentLocation()
            } else {
                onLocationFetched(it.result.latitude, it.result.longitude, locationDependantObject)
            }
        }
    }

    /**
     * This will be called when location fetch is successfully completed. Subclasses can override
     * this method to get the location fetch callback.
     */
    protected open fun onLocationFetched(
        latitude: Double,
        longitude: Double,
        locationDependantObject: Any?
    ) {
    }

    /**
     * This will be called when location fetch is failed. Subclasses can override this method
     * to get the location fetch callback.
     */
    protected open fun onLocationNotFetched(locationDependantObject: Any?) {}

    @SuppressLint("MissingPermission")
    private fun fetchCurrentLocation() {
        locationCallback = object : LocationCallback() {
            override fun onLocationResult(locationResult: LocationResult) {
                try {
                    locationRequestFallbackJob?.cancel()
                } catch (e: Exception) {
                    LogUtils.logException(e)
                }
                processLocationResult(locationResult)
            }
        }.also {
            fusedLocationClient?.requestLocationUpdates(
                createLocationRequest(),
                it,
                Looper.getMainLooper()
            )

            locationRequestFallbackJob = lifecycleScope.launch(Dispatchers.Default) {
                delay(4000)
                withContext(Dispatchers.Main) {
                    processLocationResult(null)
                }
            }
        }
    }

    private fun processLocationResult(locationResult: LocationResult?) {
        stopLocationUpdates()
        if (locationResult?.locations?.isNotEmpty() == true) {
            locationResult.locations.first()?.let { location ->
                onLocationFetched(
                    location.latitude,
                    location.longitude,
                    locationDependantObject
                )
            } ?: onLocationServicesResult(ILocationFetchStatus.FETCHING_FAILED)
        } else {
            onLocationServicesResult(ILocationFetchStatus.FETCHING_FAILED)
        }
    }

    private fun stopLocationUpdates() {
        fusedLocationClient?.let { fLocationClient ->
            locationCallback?.let { lCallback ->
                fLocationClient.removeLocationUpdates(lCallback)
            }
        }
        fusedLocationClient = null
        locationCallback = null
    }

    protected fun requestForLocation(dependantObject: Any?) {
        this.locationDependantObject = dependantObject
        checkForLocationServices()
    }

    protected fun checkForReadExternalStoragePermission(requestCode: Int) {
        if ((ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.READ_EXTERNAL_STORAGE
            ) != PackageManager.PERMISSION_GRANTED)
            || (ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.WRITE_EXTERNAL_STORAGE
            ) != PackageManager.PERMISSION_GRANTED)
        ) {
            readExternalStorageRequestCode = requestCode
            ActivityCompat.requestPermissions(
                this, arrayOf(
                    Manifest.permission.READ_EXTERNAL_STORAGE,
                    Manifest.permission.WRITE_EXTERNAL_STORAGE
                ),
                PERMISSIONS_ACCESS_READ_EXTERNAL_STORAGE
            )
        } else {
            onReadExternalStoragePermissionResponse(
                requestCode,
                ReadExternalStoragePermission.GRANTED
            )
        }
    }

    /**
     * This will be called when the storage permission request is completed. Subclasses can override
     * this method to get the storage permission response.
     */
    protected open fun onReadExternalStoragePermissionResponse(
        requestCode: Int,
        readExternalStoragePermission: ReadExternalStoragePermission
    ) {
        readExternalStoragePermissionListeners[requestCode]?.also {
            it.onPermissionResponse(requestCode, readExternalStoragePermission)
            readExternalStoragePermissionListeners.remove(requestCode)
        }
    }

    protected fun checkForCameraPermission(requestCode: Int) {

        if ((ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.CAMERA
            ) != PackageManager.PERMISSION_GRANTED)
        ) {
            accessCameraRequestCode = requestCode
            ActivityCompat.requestPermissions(
                this, arrayOf(Manifest.permission.CAMERA),
                PERMISSIONS_ACCESS_CAMERA
            )
        } else {
            onCameraPermissionResponse(requestCode, CameraPermission.GRANTED)
        }
    }

    /**
     * This function will be called when camera permission is processed. Subclasses can override
     * this for camera permission response
     */
    protected open fun onCameraPermissionResponse(
        requestCode: Int,
        cameraPermission: CameraPermission
    ) {

    }

    protected fun checkForVideoCameraPermission(requestCode: Int) {
        if ((ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.CAMERA
            ) != PackageManager.PERMISSION_GRANTED)
            || (ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.RECORD_AUDIO
            ) != PackageManager.PERMISSION_GRANTED)
        ) {
            accessVideoCameraRequestCode = requestCode
            ActivityCompat.requestPermissions(
                this, arrayOf(
                    Manifest.permission.CAMERA,
                    Manifest.permission.RECORD_AUDIO
                ),
                PERMISSIONS_ACCESS_VIDEO_CAMERA
            )
        } else {
            onVideoCameraPermissionResponse(requestCode, CameraPermission.GRANTED)
        }
    }

    /**
     * This function will be called when camera permission is processed. Subclasses can override
     * this for camera permission response
     */
    protected open fun onVideoCameraPermissionResponse(
        requestCode: Int,
        cameraPermission: CameraPermission
    ) {

    }

    fun requestReadExternalStoragePermission(
        requestCode: Int,
        readExternalStoragePermissionListener: ReadExternalStoragePermissionListener
    ) {
        readExternalStoragePermissionListeners[requestCode] = readExternalStoragePermissionListener
        checkForReadExternalStoragePermission(requestCode)
    }

    companion object {
        private const val REQUEST_CHECK_SETTINGS = 10
        private const val PERMISSIONS_ACCESS_COARSE_LOCATION = 11
        private const val PERMISSIONS_ACCESS_READ_EXTERNAL_STORAGE = 12
        private const val PERMISSIONS_ACCESS_CAMERA = 13
        private const val PERMISSIONS_ACCESS_VIDEO_CAMERA = 14
    }

    enum class ReadExternalStoragePermission { GRANTED, DENIED }

    enum class CameraPermission { GRANTED, DENIED }

    interface ReadExternalStoragePermissionListener {
        fun onPermissionResponse(
            requestCode: Int,
            readExternalStoragePermission: ReadExternalStoragePermission
        )
    }
}