package ai.engagely.openbot.viewmodel

import ai.engagely.openbot.model.localstorage.SimpleKeyValueStorage
import ai.engagely.openbot.model.localstorage.impl.PreferenceDataStore
import ai.engagely.openbot.model.pojos.internal.botsettings.IBotChatWindowConfig
import ai.engagely.openbot.model.pojos.internal.botsettings.IBotDimensions
import ai.engagely.openbot.model.pojos.internal.engagement.IWelcomeDataItem
import ai.engagely.openbot.model.repositories.BotSettingRepository
import ai.engagely.openbot.model.repositories.EngagementRepository
import ai.engagely.openbot.model.repositories.LanguageRepository
import ai.engagely.openbot.model.repositories.SessionRepository
import ai.engagely.openbot.model.repositories.impl.AppBotSettingsRepository
import ai.engagely.openbot.model.repositories.impl.AppEngagementRepository
import ai.engagely.openbot.model.repositories.impl.AppLanguageRepository
import ai.engagely.openbot.model.repositories.impl.AppSessionRepository
import ai.engagely.openbot.model.utils.exts.notifyObservers
import ai.engagely.openbot.model.utils.general.LogUtils
import ai.engagely.openbot.model.utils.general.NetworkUtils
import android.app.Application
import android.graphics.Bitmap
import android.os.Bundle
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Transformations
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.*
import java.util.concurrent.TimeUnit

class OpenBotViewModel(application: Application) : NetworkStateAwareViewModel(application),
    NetworkStateAwareViewModel.NetworkStateChangeListener {

    private lateinit var botId: String
    var extras: Map<String, String>? = null
        private set
    private val openBotSettingRepository: BotSettingRepository by lazy {
        AppBotSettingsRepository(Dispatchers.IO)
    }

    private val simpleKeyValueStorage: SimpleKeyValueStorage by lazy {
        PreferenceDataStore(application)
    }

    private val sessionRepository: SessionRepository by lazy {
        AppSessionRepository(Dispatchers.Default, Dispatchers.IO, simpleKeyValueStorage)
    }

    private val engagementRepository: EngagementRepository by lazy {
        AppEngagementRepository(Dispatchers.IO)
    }

    private val languageRepository: LanguageRepository by lazy {
        AppLanguageRepository(Dispatchers.Default, simpleKeyValueStorage)
    }

    private val _imageBitmapLiveData = MutableLiveData<Bitmap>()
    val imageBitmapLiveData: LiveData<Bitmap> = _imageBitmapLiveData

    private val _isLoadingInitialApi = MutableLiveData<Boolean>()
    val isLoadingInitialApi = _isLoadingInitialApi

    private val _errorStateLiveData = MutableLiveData<ErrorState?>()
    val errorStateLiveData: LiveData<ErrorState?> = _errorStateLiveData

    private val _welcomeDataLiveData = MutableLiveData<List<IWelcomeDataItem>>()
    val welcomeDataLiveData: LiveData<List<IWelcomeDataItem>> = _welcomeDataLiveData

    val welcomeVideoUrlLiveData: LiveData<String> =
        Transformations.map(_welcomeDataLiveData, this::prepareWelcomeVideoUrl)

    private val _botImageDimensionsLiveData = MutableLiveData<IBotDimensions>()
    val botImageDimensionsLiveData: LiveData<IBotDimensions> = _botImageDimensionsLiveData

    var botChatWindowConfig: IBotChatWindowConfig? = null
        private set

    private var fetchingLocked: Boolean = false

    init {
        setNetworkStateChangeListener(this)
    }

    fun fetchData() {
        fetchingLocked = true
        setError(null)
        _isLoadingInitialApi.postValue(true)
        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                try {
                    val getBotSettings =
                        async {
                            openBotSettingRepository.getBotSettings(
                                botId,
                                languageRepository.getCurrentLanguage()
                            )
                        }
                    val ensureSession = async { sessionRepository.ensureSession(botId) }
                    val getWelcomeData =
                        async(start = CoroutineStart.LAZY) {
                            engagementRepository.updateEngagement(
                                botId,
                                sessionRepository.getUserId(),
                                sessionRepository.getSessionId()
                            )
                        }
                    botChatWindowConfig = getBotSettings.await()
                    ensureSession.await()
                    getWelcomeData.start()
                    val welcomeData: List<IWelcomeDataItem>? =
                        if (botChatWindowConfig?.showWelcomeMessage == true) {
                            getWelcomeData.await()
                        } else {
                            getWelcomeData.cancel()
                            null
                        }
                    _welcomeDataLiveData.postValue(welcomeData)
                    if (botChatWindowConfig != null) {
                        _botImageDimensionsLiveData.postValue(
                            IBotDimensions(
                                botChatWindowConfig?.botImageShape,
                                botChatWindowConfig?.botImageSize
                            )
                        )
                        _imageBitmapLiveData.postValue(botChatWindowConfig?.logo)
                        _isLoadingInitialApi.postValue(false)
                    } else {
                        delay(TimeUnit.SECONDS.toMillis(1))
                        checkFailure()
                    }
                    fetchingLocked = false
                } catch (e: Exception) {
                    LogUtils.logException(e)
                    delay(TimeUnit.SECONDS.toMillis(1))
                    checkFailure()
                    fetchingLocked = false
                }
            }
        }
    }

    private fun checkFailure() {
        setError(
            if (NetworkUtils.isInternetAvailable(getApplication()))
                ErrorState.SERVER_FAILURE
            else
                ErrorState.NETWORK_FAILURE
        )
    }

    private fun setError(errorState: ErrorState?) {
        _errorStateLiveData.postValue(errorState)
    }

    fun refreshSession() {
        viewModelScope.launch {
            sessionRepository.clearSession()
            sessionRepository.ensureSession(botId)
        }
    }

    fun resetWelcomeData() {
        _welcomeDataLiveData.notifyObservers()
    }

    override fun onNetworkStateChanged(networkState: NetworkState) {
        if (networkState == NetworkState.CONNECTED
            && _errorStateLiveData.value == ErrorState.NETWORK_FAILURE
            && !fetchingLocked
        ) {
            fetchData()
        }
    }

    fun init(botId: String, extras: Map<String, String>?) {
        this.botId = botId
        this.extras = extras
    }

    fun getExtras(): Bundle {
        return Bundle().apply {
            extras?.let { extras ->
                extras.keys.forEach { key ->
                    putString(key, extras[key])
                }
            }
        }
    }

    fun updateBotConfigData(botConfigData: IBotChatWindowConfig) {
        botChatWindowConfig = botConfigData
    }

    private fun prepareWelcomeVideoUrl(welcomeDataItems: List<IWelcomeDataItem>?): String? {
        welcomeDataItems?.forEach {
            if (it.type == IWelcomeDataItem.IWelcomeDataType.VIDEO) {
                return it.data
            }
        }
        return null
    }

    enum class ErrorState { NETWORK_FAILURE, SERVER_FAILURE }
}