/*
 * Decompiled with CFR 0.152.
 */
package ai.grakn.engine.session;

import ai.grakn.GraknGraph;
import ai.grakn.concept.ResourceType;
import ai.grakn.concept.Type;
import ai.grakn.concept.TypeName;
import ai.grakn.exception.ConceptException;
import ai.grakn.exception.GraknValidationException;
import ai.grakn.graql.Printer;
import ai.grakn.graql.Query;
import ai.grakn.graql.internal.printer.Printers;
import com.google.common.base.Splitter;
import java.io.IOException;
import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import mjson.Json;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.WebSocketException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GraqlSession {
    private final Session session;
    private final boolean showImplicitTypes;
    private final boolean infer;
    private final boolean materialise;
    private GraknGraph graph;
    private final Supplier<GraknGraph> getGraph;
    private final String outputFormat;
    private Printer printer;
    private StringBuilder queryStringBuilder = new StringBuilder();
    private final Logger LOG = LoggerFactory.getLogger(GraqlSession.class);
    private static final int QUERY_CHUNK_SIZE = 1000;
    private static final int PING_INTERVAL = 60000;
    private boolean queryCancelled = false;
    private final ExecutorService queryExecutor = Executors.newSingleThreadExecutor();

    GraqlSession(Session session, Supplier<GraknGraph> getGraph, String outputFormat, boolean showImplicitTypes, boolean infer, boolean materialise) {
        this.showImplicitTypes = showImplicitTypes;
        this.infer = infer;
        this.materialise = materialise;
        this.session = session;
        this.getGraph = getGraph;
        this.outputFormat = outputFormat;
        this.printer = this.getPrinter(new ResourceType[0]);
        this.queryExecutor.submit(() -> {
            try {
                this.refreshGraph();
                this.sendTypes();
                this.sendEnd();
            }
            catch (Throwable e) {
                this.LOG.error(ExceptionUtils.getFullStackTrace((Throwable)e));
                throw e;
            }
        });
        Thread thread = new Thread(this::ping, "graql-session-ping");
        thread.setDaemon(true);
        thread.start();
    }

    private void refreshGraph() {
        this.graph = this.getGraph.get();
        this.graph.showImplicitConcepts(this.showImplicitTypes);
    }

    void handleMessage(Json json) {
        switch (json.at("action").asString()) {
            case "query": {
                this.receiveQuery(json);
                break;
            }
            case "end": {
                this.executeQuery();
                break;
            }
            case "queryAbort": {
                this.abortQuery();
                break;
            }
            case "commit": {
                this.commit();
                break;
            }
            case "rollback": {
                this.rollback();
                break;
            }
            case "display": {
                this.setDisplayOptions(json);
                break;
            }
            case "ping": {
                break;
            }
            default: {
                throw new RuntimeException("Unrecognized message: " + json);
            }
        }
    }

    private void ping() {
        while (this.session.isOpen()) {
            try {
                this.sendJson(Json.object((Object[])new Object[]{"action", "ping"}));
            }
            catch (WebSocketException e) {
                if (!this.session.isOpen()) continue;
                this.LOG.error(e.getMessage());
            }
            finally {
                try {
                    Thread.sleep(60000L);
                }
                catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    void close() {
        this.queryExecutor.submit(() -> {
            try {
                this.graph.close();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    void receiveQuery(Json json) {
        this.queryExecutor.submit(() -> {
            String queryString = json.at("query").asString();
            this.queryStringBuilder.append(queryString);
        });
    }

    void executeQuery() {
        this.queryExecutor.submit(() -> {
            String errorMessage = null;
            Collection queries = null;
            try {
                String queryString = this.queryStringBuilder.toString();
                this.queryStringBuilder = new StringBuilder();
                queries = this.graph.graql().infer(this.infer).materialise(this.materialise).parseList(queryString);
                queries.stream().flatMap(query -> query.resultsString(this.printer)).forEach(result -> {
                    if (this.queryCancelled) {
                        return;
                    }
                    this.sendQueryResult((String)result);
                });
                this.queryCancelled = false;
            }
            catch (ConceptException | IllegalArgumentException | IllegalStateException e) {
                errorMessage = e.getMessage();
                this.LOG.error(errorMessage, e);
            }
            catch (Throwable e) {
                errorMessage = "An unexpected error occurred";
                this.LOG.error(errorMessage, e);
            }
            finally {
                this.attemptRefresh();
                if (errorMessage != null) {
                    if (queries != null && !queries.stream().allMatch(Query::isReadOnly)) {
                        this.attemptRollback();
                    }
                    this.sendQueryError(errorMessage);
                }
                this.sendEnd();
            }
        });
    }

    void abortQuery() {
        this.queryCancelled = true;
    }

    void commit() {
        this.queryExecutor.submit(() -> {
            try {
                this.graph.commit();
            }
            catch (GraknValidationException e) {
                this.sendCommitError(e.getMessage());
            }
            this.sendEnd();
        });
    }

    void rollback() {
        this.queryExecutor.submit(() -> ((GraknGraph)this.graph).rollback());
    }

    private void attemptRefresh() {
        try {
            this.refreshGraph();
        }
        catch (Throwable e) {
            this.LOG.error("Error during refresh", e);
        }
    }

    void setDisplayOptions(Json json) {
        this.queryExecutor.submit(() -> {
            ResourceType[] displayOptions = (ResourceType[])json.at("display").asJsonList().stream().map(Json::asString).map(arg_0 -> ((GraknGraph)this.graph).getResourceType(arg_0)).filter(Objects::nonNull).toArray(ResourceType[]::new);
            this.printer = this.getPrinter(displayOptions);
        });
    }

    private boolean attemptRollback() {
        try {
            this.graph.rollback();
            return true;
        }
        catch (UnsupportedOperationException ignored) {
            return false;
        }
        catch (Throwable e) {
            this.LOG.error("Error during rollback", e);
            return false;
        }
    }

    private void sendQueryResult(String result) {
        Iterable splitResult = Splitter.fixedLength((int)1000).split((CharSequence)(result + "\n"));
        for (String resultChunk : splitResult) {
            this.sendJson(Json.object((Object[])new Object[]{"action", "query", "result", resultChunk}));
        }
    }

    private void sendEnd() {
        this.sendJson(Json.object((Object[])new Object[]{"action", "end"}));
    }

    private void sendQueryError(String errorMessage) {
        Iterable splitError = Splitter.fixedLength((int)1000).split((CharSequence)(errorMessage + "\n"));
        for (String errorChunk : splitError) {
            this.sendJson(Json.object((Object[])new Object[]{"action", "error", "error", errorChunk}));
        }
    }

    private void sendCommitError(String errorMessage) {
        this.sendJson(Json.object((Object[])new Object[]{"action", "error", "error", errorMessage}));
    }

    private void sendTypes() {
        this.sendJson(Json.object((Object[])new Object[]{"action", "types", "types", GraqlSession.getTypes(this.graph).map(TypeName::getValue).collect(Collectors.toList())}));
    }

    private void sendJson(Json json) {
        this.queryExecutor.submit(() -> {
            this.LOG.debug("Sending message: " + json);
            try {
                this.session.getRemote().sendString(json.toString());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private static Stream<TypeName> getTypes(GraknGraph graph) {
        return graph.admin().getMetaConcept().subTypes().stream().map(Type::getName);
    }

    private Printer getPrinter(ResourceType ... resources) {
        switch (this.outputFormat) {
            default: {
                return Printers.graql((ResourceType[])resources);
            }
            case "json": {
                return Printers.json();
            }
            case "hal": 
        }
        return Printers.hal();
    }
}

