/*
 * Decompiled with CFR 0.152.
 */
package ai.grakn.engine;

import ai.grakn.engine.GraknEngineConfig;
import ai.grakn.engine.controller.AuthController;
import ai.grakn.engine.controller.CommitLogController;
import ai.grakn.engine.controller.ConceptController;
import ai.grakn.engine.controller.DashboardController;
import ai.grakn.engine.controller.GraqlController;
import ai.grakn.engine.controller.SystemController;
import ai.grakn.engine.controller.TasksController;
import ai.grakn.engine.controller.UserController;
import ai.grakn.engine.factory.EngineGraknGraphFactory;
import ai.grakn.engine.lock.LockProvider;
import ai.grakn.engine.lock.ProcessWideLockProvider;
import ai.grakn.engine.lock.RedissonLockProvider;
import ai.grakn.engine.session.RemoteSession;
import ai.grakn.engine.tasks.connection.RedisCountStorage;
import ai.grakn.engine.tasks.manager.StandaloneTaskManager;
import ai.grakn.engine.tasks.manager.TaskManager;
import ai.grakn.engine.tasks.manager.redisqueue.RedisTaskManager;
import ai.grakn.engine.user.UsersHandler;
import ai.grakn.engine.util.EngineID;
import ai.grakn.engine.util.JWTHandler;
import ai.grakn.exception.GraknBackendException;
import ai.grakn.exception.GraknServerException;
import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.ImmutableSet;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import javax.annotation.Nullable;
import mjson.Json;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.apache.http.entity.ContentType;
import org.redisson.Redisson;
import org.redisson.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.JedisSentinelPool;
import redis.clients.util.Pool;
import spark.HaltException;
import spark.Request;
import spark.Response;
import spark.Service;

public class GraknEngineServer
implements AutoCloseable {
    private static final String LOAD_SYSTEM_ONTOLOGY_LOCK_NAME = "load-system-ontology";
    private static final Logger LOG = LoggerFactory.getLogger(GraknEngineServer.class);
    private static final Set<String> unauthenticatedEndPoints = new HashSet<String>(Arrays.asList("/auth/session/", "/shell/remote", "/configuration", "/auth/enabled/"));
    private final GraknEngineConfig prop;
    private final EngineID engineId = EngineID.me();
    private final Service spark = Service.ignite();
    private final TaskManager taskManager;
    private final EngineGraknGraphFactory factory;
    private final RedisCountStorage redisCountStorage;
    private final MetricRegistry metricRegistry;
    private final Pool<Jedis> jedisPool;
    private final boolean inMemoryQueue;
    private final LockProvider lockProvider;

    public GraknEngineServer(GraknEngineConfig prop) {
        this.prop = prop;
        int redisPort = Integer.parseInt(prop.tryProperty("redis.port").orElse("6379"));
        String redisUrl = prop.tryProperty("redis.host").orElse("localhost");
        this.jedisPool = this.instantiateRedis(prop, redisUrl, redisPort);
        this.redisCountStorage = RedisCountStorage.create(this.jedisPool);
        String taskManagerClassName = prop.getProperty("taskmanager.implementation");
        this.inMemoryQueue = !taskManagerClassName.contains("RedisTaskManager");
        this.lockProvider = this.inMemoryQueue ? new ProcessWideLockProvider() : this.instantiateRedissonLockProvider(redisPort, redisUrl);
        this.factory = EngineGraknGraphFactory.create(prop.getProperties());
        this.metricRegistry = new MetricRegistry();
        this.taskManager = this.startTaskManager(this.inMemoryQueue, this.redisCountStorage, this.jedisPool, this.lockProvider);
    }

    public static void main(String[] args) {
        GraknEngineConfig prop = GraknEngineConfig.create();
        GraknEngineServer graknEngineServer = new GraknEngineServer(prop);
        graknEngineServer.start();
        Thread closeThread = new Thread(graknEngineServer::close, "GraknEngineServer-shutdown");
        Runtime.getRuntime().addShutdownHook(closeThread);
    }

    public void start() {
        this.lockAndInitializeSystemOntology();
        this.startHTTP();
        GraknEngineServer.printStartMessage(this.prop.getProperty("server.host"), this.prop.getProperty("server.port"));
    }

    @Override
    public void close() {
        this.stopHTTP();
        this.stopTaskManager();
    }

    private void lockAndInitializeSystemOntology() {
        try {
            Lock lock = this.lockProvider.getLock(LOAD_SYSTEM_ONTOLOGY_LOCK_NAME);
            if (lock.tryLock(60L, TimeUnit.SECONDS)) {
                this.loadAndUnlock(lock);
            } else {
                LOG.info("{} found system ontology lock already acquired by other engine", (Object)this.engineId);
            }
        }
        catch (InterruptedException e) {
            LOG.warn("{} was interrupted while initializing system ontology", (Object)this.engineId);
        }
    }

    private void loadAndUnlock(Lock lock) {
        try {
            LOG.info("{} is initializing the system ontology", (Object)this.engineId);
            this.factory.systemKeyspace().loadSystemOntology();
        }
        finally {
            lock.unlock();
        }
    }

    private TaskManager startTaskManager(boolean inMemoryQueue, RedisCountStorage redisCountStorage, Pool<Jedis> jedisPool, LockProvider lockProvider) {
        TaskManager taskManager = !inMemoryQueue ? new RedisTaskManager(this.engineId, this.prop, jedisPool, this.factory, lockProvider, this.metricRegistry) : new StandaloneTaskManager(this.engineId, this.prop, redisCountStorage, this.factory, lockProvider, this.metricRegistry);
        taskManager.start();
        return taskManager;
    }

    public void startHTTP() {
        boolean passwordProtected = this.prop.getPropertyAsBool("password.protected", false);
        Optional<String> secret = this.prop.tryProperty("JWT.secret");
        JWTHandler jwtHandler = secret.map(JWTHandler::create).orElse(null);
        UsersHandler usersHandler = UsersHandler.create(this.prop.getProperty("admin.password"), this.factory);
        GraknEngineServer.configureSpark(this.spark, this.prop, jwtHandler);
        RemoteSession graqlWebSocket = passwordProtected ? RemoteSession.passwordProtected(usersHandler) : RemoteSession.create();
        this.spark.webSocket("/shell/remote", (Object)graqlWebSocket);
        String defaultKeyspace = this.prop.getProperty("graph.default-keyspace");
        int postProcessingDelay = this.prop.getPropertyAsInt("tasks.postprocessing.delay");
        new GraqlController(this.factory, this.spark, this.metricRegistry);
        new ConceptController(this.factory, this.spark, this.metricRegistry);
        new DashboardController(this.factory, this.spark);
        new SystemController(this.factory, this.spark, this.metricRegistry);
        new AuthController(this.spark, passwordProtected, jwtHandler, usersHandler);
        new UserController(this.spark, usersHandler);
        new CommitLogController(this.spark, defaultKeyspace, postProcessingDelay, this.taskManager);
        new TasksController(this.spark, this.taskManager, this.metricRegistry);
        this.spark.awaitInitialization();
    }

    public static void configureSpark(Service spark, GraknEngineConfig prop, @Nullable JWTHandler jwtHandler) {
        GraknEngineServer.configureSpark(spark, prop.getProperty("server.host"), Integer.parseInt(prop.getProperty("server.port")), prop.getPath("server.static-file-dir"), prop.getPropertyAsBool("password.protected", false), jwtHandler);
    }

    public static void configureSpark(Service spark, String hostName, int port, String staticFolder, boolean passwordProtected, @Nullable JWTHandler jwtHandler) {
        spark.ipAddress(hostName);
        spark.port(port);
        spark.staticFiles.externalLocation(staticFolder);
        spark.webSocketIdleTimeoutMillis(3600000);
        boolean isPasswordProtected = passwordProtected;
        if (isPasswordProtected) {
            spark.before((req, res) -> GraknEngineServer.checkAuthorization(spark, req, jwtHandler));
        }
        spark.exception(GraknBackendException.class, (e, req, res) -> GraknEngineServer.handleGraknServerError(e, res));
        spark.exception(Exception.class, (e, req, res) -> GraknEngineServer.handleInternalError(e, res));
    }

    public void stopHTTP() {
        this.spark.stop();
        boolean running = true;
        while (running) {
            try {
                this.spark.port();
            }
            catch (IllegalStateException e) {
                LOG.debug("Spark server has been stopped");
                running = false;
            }
        }
    }

    private void stopTaskManager() {
        try {
            this.taskManager.close();
        }
        catch (Exception e) {
            LOG.error(ExceptionUtils.getFullStackTrace((Throwable)e));
        }
    }

    public TaskManager getTaskManager() {
        return this.taskManager;
    }

    public EngineGraknGraphFactory factory() {
        return this.factory;
    }

    private static void checkAuthorization(Service spark, Request request, JWTHandler jwtHandler) throws HaltException {
        if (!unauthenticatedEndPoints.contains(request.pathInfo())) {
            boolean authenticated;
            try {
                if (request.headers("Authorization") == null || !request.headers("Authorization").startsWith("Bearer ")) {
                    throw GraknServerException.authenticationFailure();
                }
                String token = request.headers("Authorization").substring(7);
                authenticated = jwtHandler.verifyJWT(token);
                request.attribute("user", (Object)jwtHandler.extractUserFromJWT(token));
            }
            catch (GraknBackendException e) {
                throw e;
            }
            catch (Exception e) {
                throw GraknServerException.serverException((int)400, (Exception)e);
            }
            if (!authenticated) {
                spark.halt(401, "User not authenticated.");
            }
        }
    }

    private static void handleGraknServerError(Exception exception, Response response) {
        LOG.error("REST error", (Throwable)exception);
        response.status(((GraknServerException)exception).getStatus());
        response.body(Json.object((Object[])new Object[]{"exception", exception.getMessage()}).toString());
        response.type(ContentType.APPLICATION_JSON.getMimeType());
    }

    private static void handleInternalError(Exception exception, Response response) {
        LOG.error("REST error", (Throwable)exception);
        response.status(500);
        response.body(Json.object((Object[])new Object[]{"exception", exception.getMessage()}).toString());
        response.type(ContentType.APPLICATION_JSON.getMimeType());
    }

    private Pool<Jedis> instantiateRedis(GraknEngineConfig prop, String redisUrl, int redisPort) {
        JedisPoolConfig poolConfig = new JedisPoolConfig();
        poolConfig.setMaxTotal(128);
        Optional<String> sentinelMaster = prop.tryProperty("redis.sentinel-master");
        return sentinelMaster.map(s -> new JedisSentinelPool(s, (Set)ImmutableSet.of((Object)String.format("%s:%s", redisUrl, redisPort)), (GenericObjectPoolConfig)poolConfig)).orElseGet(() -> new JedisPool((GenericObjectPoolConfig)poolConfig, redisUrl, redisPort));
    }

    private RedissonLockProvider instantiateRedissonLockProvider(int redisPort, String redisUrl) {
        LOG.info("Connecting redisCountStorage client to {}:{}", (Object)redisUrl, (Object)redisPort);
        Config redissonConfig = new Config();
        redissonConfig.useSingleServer().setAddress(String.format("%s:%d", redisUrl, redisPort)).setConnectionPoolSize(5);
        return new RedissonLockProvider(Redisson.create((Config)redissonConfig));
    }

    private static void printStartMessage(String host, String port) {
        String address = "http://" + host + ":" + port;
        LOG.info("\n==================================================");
        LOG.info("\n" + String.format("     ___  ___  ___  _  __ _  _     ___  ___     %n    / __|| _ \\/   \\| |/ /| \\| |   /   \\|_ _|    %n   | (_ ||   /| - || ' < | .` | _ | - | | |     %n    \\___||_|_\\|_|_||_|\\_\\|_|\\_|(_)|_|_||___|   %n%n Web Dashboard available at [%s]", address));
        LOG.info("\n==================================================");
    }
}

