/*
 * Decompiled with CFR 0.152.
 */
package ai.grakn.engine.session;

import ai.grakn.GraknSession;
import ai.grakn.GraknTx;
import ai.grakn.GraknTxType;
import ai.grakn.concept.AttributeType;
import ai.grakn.concept.Label;
import ai.grakn.concept.SchemaConcept;
import ai.grakn.exception.GraknException;
import ai.grakn.exception.InvalidKBException;
import ai.grakn.graql.ComputeQuery;
import ai.grakn.graql.Printer;
import ai.grakn.graql.Query;
import ai.grakn.graql.internal.printer.Printers;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import mjson.Json;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.WebSocketException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GraqlSession {
    private final Session session;
    private final boolean infer;
    private GraknTx tx;
    private final GraknSession factory;
    private final String outputFormat;
    private Printer printer;
    private StringBuilder queryStringBuilder = new StringBuilder();
    private final Logger LOG = LoggerFactory.getLogger(GraqlSession.class);
    private static final int QUERY_CHUNK_SIZE = 1000;
    private static final int PING_INTERVAL = 60000;
    private final ExecutorService queryExecutor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("graql-session-%s").build());
    private List<Query<?>> queries = null;

    GraqlSession(Session session, GraknSession factory, String outputFormat, boolean infer) {
        Preconditions.checkNotNull((Object)session);
        this.infer = infer;
        this.session = session;
        this.factory = factory;
        this.outputFormat = outputFormat;
        this.queryExecutor.execute(() -> {
            try {
                this.refreshTx();
                this.printer = this.getPrinter(new AttributeType[0]);
                this.sendTypes();
                this.sendEnd();
            }
            catch (Throwable e) {
                this.LOG.error(ExceptionUtils.getFullStackTrace((Throwable)e));
                this.sendError(e.getMessage());
                this.sendEnd();
                session.close();
                throw e;
            }
        });
        Thread thread = new Thread(this::ping, "graql-session-ping");
        thread.setDaemon(true);
        thread.start();
    }

    private void refreshTx() {
        if (this.tx != null && !this.tx.isClosed()) {
            this.tx.close();
        }
        this.tx = this.factory.open(GraknTxType.WRITE);
    }

    void handleMessage(Json json) {
        switch (json.at("action").asString()) {
            case "query": {
                this.receiveQuery(json);
                break;
            }
            case "end": {
                this.executeQuery();
                break;
            }
            case "commit": {
                this.commit();
                break;
            }
            case "rollback": {
                this.rollback();
                break;
            }
            case "clean": {
                this.clean();
                break;
            }
            case "display": {
                this.setDisplayOptions(json);
                break;
            }
            case "ping": {
                break;
            }
            default: {
                throw new RuntimeException("Unrecognized message: " + json);
            }
        }
    }

    private void ping() {
        while (this.session.isOpen()) {
            try {
                this.sendJson(Json.object((Object[])new Object[]{"action", "ping"}));
            }
            catch (WebSocketException e) {
                if (!this.session.isOpen()) continue;
                this.LOG.error(e.getMessage());
            }
            finally {
                try {
                    Thread.sleep(60000L);
                }
                catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    void close() {
        this.queryExecutor.execute(() -> {
            try {
                this.tx.close();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
        if (this.queries != null) {
            for (Query<?> query : this.queries) {
                if (!(query instanceof ComputeQuery)) continue;
                ((ComputeQuery)query).kill();
            }
        }
    }

    void receiveQuery(Json json) {
        this.queryExecutor.execute(() -> {
            String queryString = json.at("query").asString();
            this.queryStringBuilder.append(queryString);
        });
    }

    Future<?> executeQuery() {
        return this.queryExecutor.submit(() -> {
            String errorMessage = null;
            try {
                String queryString = this.queryStringBuilder.toString();
                this.queryStringBuilder = new StringBuilder();
                this.queries = this.tx.graql().infer(this.infer).parser().parseList(queryString).collect(Collectors.toList());
                this.queries.stream().flatMap(query -> query.resultsString(this.printer)).forEach(this::sendQueryResult);
            }
            catch (GraknException e) {
                errorMessage = e.getMessage();
                this.LOG.error(errorMessage, (Throwable)e);
            }
            catch (Exception e) {
                errorMessage = ExceptionUtils.getFullStackTrace((Throwable)e);
                this.LOG.error(errorMessage, (Throwable)e);
            }
            finally {
                if (errorMessage != null) {
                    if (this.queries != null && !this.queries.stream().allMatch(Query::isReadOnly)) {
                        this.attemptRefresh();
                    }
                    this.sendError(errorMessage);
                }
                this.sendEnd();
            }
        });
    }

    void commit() {
        this.queryExecutor.execute(() -> {
            try {
                this.tx.commit();
            }
            catch (InvalidKBException e) {
                this.sendError(e.getMessage());
            }
            finally {
                this.sendEnd();
                this.attemptRefresh();
            }
        });
    }

    void rollback() {
        this.queryExecutor.execute(() -> {
            this.tx.close();
            this.attemptRefresh();
        });
    }

    void clean() {
        this.queryExecutor.execute(() -> {
            this.tx.admin().delete();
            this.attemptRefresh();
        });
    }

    private void attemptRefresh() {
        try {
            this.refreshTx();
        }
        catch (Throwable e) {
            this.LOG.error("Error during refresh", e);
        }
    }

    void setDisplayOptions(Json json) {
        this.queryExecutor.execute(() -> {
            AttributeType[] displayOptions = (AttributeType[])json.at("display").asJsonList().stream().map(Json::asString).map(arg_0 -> ((GraknTx)this.tx).getAttributeType(arg_0)).filter(Objects::nonNull).toArray(AttributeType[]::new);
            this.printer = this.getPrinter(displayOptions);
        });
    }

    private void sendQueryResult(String result) {
        Iterable splitResult = Splitter.fixedLength((int)1000).split((CharSequence)(result + "\n"));
        for (String resultChunk : splitResult) {
            this.sendJson(Json.object((Object[])new Object[]{"action", "query", "result", resultChunk}));
        }
    }

    private void sendEnd() {
        this.sendJson(Json.object((Object[])new Object[]{"action", "end"}));
    }

    private void sendError(String errorMessage) {
        Iterable splitError = Splitter.fixedLength((int)1000).split((CharSequence)(errorMessage + "\n"));
        for (String errorChunk : splitError) {
            this.sendJson(Json.object((Object[])new Object[]{"action", "error", "error", errorChunk}));
        }
    }

    private void sendTypes() {
        this.sendJson(Json.object((Object[])new Object[]{"action", "types", "types", GraqlSession.getTypes(this.tx).map(Label::getValue).collect(Collectors.toList())}));
    }

    private void sendJson(Json json) {
        this.queryExecutor.execute(() -> {
            this.LOG.debug("Sending message: " + json);
            try {
                this.session.getRemote().sendString(json.toString());
            }
            catch (IOException e) {
                this.LOG.error("Error while sending JSON: " + json, (Throwable)e);
            }
        });
    }

    private static Stream<Label> getTypes(GraknTx graph) {
        return graph.admin().getMetaConcept().subs().map(SchemaConcept::getLabel);
    }

    private Printer getPrinter(AttributeType ... resources) {
        switch (this.outputFormat) {
            case "json": {
                return Printers.json();
            }
        }
        return Printers.graql((boolean)true, (AttributeType[])resources);
    }
}

