/*
 * Grakn - A Distributed Semantic Database
 * Copyright (C) 2016  Grakn Labs Limited
 *
 * Grakn is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Grakn is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Grakn. If not, see <http://www.gnu.org/licenses/gpl.txt>.
 */

package ai.grakn.graql.internal.reasoner.atom;

import ai.grakn.graql.admin.PatternAdmin;
import ai.grakn.graql.VarName;
import ai.grakn.graql.internal.reasoner.query.Query;

import java.util.Map;
import java.util.Set;


/**
 *
 * <p>
 * Interface for atoms.
 * </p>
 *
 * @author Kasper Piskorski
 *
 */
public interface Atomic extends Cloneable{

    Atomic clone();

    default boolean isAtom(){ return false;}
    default boolean isPredicate(){ return false;}

    /**
     * @return true if atom alpha-equivalent
     */
    boolean isEquivalent(Object obj);

    /**
     * @return equivalence hash code
     */
    int equivalenceHashCode();

    /**
     * @return true if the variable name is user defined
     */
    default boolean isUserDefinedName(){ return false;}

    /**
     * @return true if the atom can be resolved by a rule (atom exists in one of the rule's head)
     */
    default boolean isRuleResolvable(){ return false;}
    /**
     * @return true if the atom can form an atomic query
     */
    default boolean isSelectable(){ return false;}

    default boolean isRecursive(){ return false;}

    /**
     * @param name variable name
     * @return true if atom contains an occurrence of the variable name
     */
    default boolean containsVar(VarName name){ return false;}

    /**
     * @return the corresponding base pattern
     * */
    PatternAdmin getPattern();

    /**
     * @return the base pattern combined with possible predicate patterns
     */
    PatternAdmin getCombinedPattern();

    /**
     * @return the query this atom belongs to
     * */
    Query getParentQuery();

    /**
     * @param q query this atom is supposed to belong to
     */
    void setParentQuery(Query q);

    Map<VarName, VarName> getUnifiers(Atomic parentAtom);

    /**
     * change each variable occurrence according to provided mappings (apply unifiers {[from, to]_i})
     * if capture occurs it is marked with a "capture-><name of the captured occurrence>" name
     * @param unifiers contain variable mappings to be applied
     */
    void unify(Map<VarName, VarName> unifiers);

    VarName getVarName();

    /**
     * @return all addressable variable names in the atom
     */
    Set<VarName> getVarNames();
}
